# -*- coding: utf-8 -*-

import logging
import os
from distutils.dir_util import copy_tree

from sandbox import sdk2
from sandbox.projects.market.frontarc.helpers.arc_base import MarketFrontArcBase
from sandbox.projects.market.frontarc.helpers.sandbox_helpers import rich_check_call
from sandbox.projects.market.resources import MARKET_FRONT_CMS_SCREENSHOTS
from sandbox.projects.sandbox_ci.utils import env
from sandbox.projects.common import task_env

ACTUAL_SRC_DIR = "actual_src"
SCREENSHOOTER_LOGFILE = "screenshooter.log"
SCREENSHOT_DIR = "screenshots"
CONTAINER_RESOURCE_ID = 1300990332


class GenericParametersArc(sdk2.Task.Parameters):
    with sdk2.parameters.Group("Приложение") as app:
        arc_path = sdk2.parameters.String(
            "Путь до проекта в arc",
            default_value="market/front/libs/cms-screenshooter",
            required=True
        )

        app_branch = sdk2.parameters.String(
            "Ветка в Аркадии",
            default_value="trunk",
            required=False
        )
        app_commit = sdk2.parameters.String(
            "Коммит",
            default_value="",
            required=False
        )

    with sdk2.parameters.Group("Настройки приложения") as app_settings:
        telegram_secret = sdk2.parameters.YavSecret(
            "Yav secret с токенами телеграма",
            description="Ожидается, что в секрете будут ключи: telegram_bot_token и cms_internal_user_id",
            default="sec-01dyhqjkm6295c2mxk2mbp63vs"  # cms-screenshooter-secrets
        )

        cms_page_types = sdk2.parameters.String(
            "Типы страниц в CMS",
            description="Параметры для ручки `getResourceList` в CMS. Параметры должны быть разделены "
                        "точкой с запятой, без пробелов",
            default_value="franchise;licensor;brand;brand_article;promo;promo_desktop;morda_context;product;"
                          "navnode_touch;catalog",
            required=True
        )

        chat_id = sdk2.parameters.String(
            "ID чата в телеграм для отправки уведомлений",
            required=True
        )

        manual_time = sdk2.parameters.Bool(
            "Задать период для проверки вручную?",
            description="По-умолчанию - предыдущий полный час от момента создания.\nТ.е. если таска была создана "
                        "в 17:53:02, то проверятся будет промежуток от 16:00:00 до 17:00:00",
            default=False,
            required=False
        )

        with manual_time.value[True]:
            check_interval_start = sdk2.parameters.String(
                "Начало периода",
                description="Нижняя граница времени публикации. Формат:\n%b %d, %Y %I:%M:%S %p\n"
                            "Пример: Jan 13, 2020 6:00:00 PM",
                required=True
            )
            check_interval_end = sdk2.parameters.String(
                "Окончание периода",
                description="Верхняя граница времени публикации. Формат:\n%b %d, %Y %I:%M:%S %p\n"
                            "Пример: Jan 13, 2020 7:00:00 PM",
                required=True
            )

    with sdk2.parameters.Group("Environment") as environ_block:
        environ = sdk2.parameters.Dict("Environment variables")


class MarketCmsScreenShooterArc(sdk2.Task, MarketFrontArcBase):
    """
    Task to take screenshots of recently published CMS pages
    """

    class Requirements(task_env.TinyRequirements):
        container_resource = CONTAINER_RESOURCE_ID

    class Parameters(GenericParametersArc):
        pass

    def on_enqueue(self):
        super(MarketCmsScreenShooterArc, self).on_enqueue()

    def on_prepare(self):
        super(MarketCmsScreenShooterArc, self).on_prepare()
        self.arc_mount()

    def on_finish(self, prev_status, status):
        super(MarketCmsScreenShooterArc, self).on_finish(prev_status, status)
        self.arc_unmount()

    def on_wait(self, prev_status, status):
        super(MarketCmsScreenShooterArc, self).on_wait()
        self.arc_unmount()

    def on_execute(self):
        super(MarketCmsScreenShooterArc, self).on_execute()

        self._set_environment()

        app_src_root_dir = os.path.join(self.arcadia, self.Parameters.arc_path)

        self._prepare_repo(app_src_root_dir)

        self._install_req(app_src_root_dir)

        self._call_screenshooter_script(app_src_root_dir)

        self._move_logs_and_pictures(app_src_root_dir)

    def _prepare_repo(self, app_src_path):
        logging.info("Берём сорцы из {}".format(app_src_path))
        if self.Parameters.app_commit:
            self.arc_checkout(self.Parameters.app_commit, track=True)
        else:
            self.arc_checkout(self.Parameters.app_branch, track=True)

    def _install_req(self, app_src_path):
        rich_check_call(
            ["pip3", "install", "--index-url", "https://pypi.yandex-team.ru/simple/", "-r", "requirements.txt"],
            task=self,
            alias="install req",
            cwd=app_src_path
        )

    def _call_screenshooter_script(self, app_src_path):
        rich_check_call(
            ["python3", "__init__.py"],
            task=self,
            alias="ScreenShooter",
            cwd=app_src_path
        )

    def _set_environment(self):
        custom_env = {"LC_TIME": "en_US.UTF-8"}

        if self.Parameters.manual_time:
            custom_env["CHECK_INTERVAL_END"] = self.Parameters.check_interval_end
            custom_env["CHECK_INTERVAL_START"] = self.Parameters.check_interval_start
        else:
            from datetime import timedelta
            import pytz
            local_timezone = pytz.timezone("Europe/Moscow")

            datetime_end = self.created.astimezone(tz=local_timezone)
            datetime_start = datetime_end - timedelta(hours=1)

            custom_env["CHECK_INTERVAL_END"] = datetime_end.strftime("%b %d, %Y %I:00:00 %p")
            custom_env["CHECK_INTERVAL_START"] = datetime_start.strftime("%b %d, %Y %I:00:00 %p")

        logging.info('Date. Checking published pages: from {} to {}'.format(
            custom_env["CHECK_INTERVAL_START"], custom_env["CHECK_INTERVAL_END"]
        ))

        secret = self.Parameters.telegram_secret
        custom_env["TELEGRAM_BOT_TOKEN"] = secret.data()["telegram_bot_token"]
        custom_env["TELEGRAM_CHANNEL_ID"] = self.Parameters.chat_id
        custom_env["CMS_INTERNAL_USER_ID"] = secret.data()["cms_internal_user_id"]

        custom_env["SCREENSHOT_DIR"] = SCREENSHOT_DIR
        custom_env["SC_LOGFILE"] = SCREENSHOOTER_LOGFILE
        custom_env["PAGE_TYPES"] = self.Parameters.cms_page_types

        env.export(custom_env)
        env.export(self.Parameters.environ)

    def _move_logs_and_pictures(self, app_src_path):
        """
        Saves app logs and screenshots in log resource
        """
        old_logfile = os.path.join(app_src_path, os.environ.get("SC_LOGFILE"))
        new_logfile = os.path.join(str(self.log_path()), os.environ.get("SC_LOGFILE"))
        if os.path.exists(old_logfile):
            os.rename(old_logfile, new_logfile)

        screenshot_dir = os.path.join(app_src_path, os.environ.get("SCREENSHOT_DIR"))
        if os.path.exists(screenshot_dir):
            screenshooter_results_path = os.path.join(str(self.path().absolute()), "screenshots")
            if not os.path.exists(screenshooter_results_path):
                os.mkdir(screenshooter_results_path)

            # Если указывать в качестве ресурса папку в аркадии, то ресурс не создаётся и джоба падает с ошибкой.
            # Кажется, это из-за того что аркадия мауниттся. Поэтому копируем на диск для создания ресурса.
            # Can be replaced with shutil.copytree(src, dst, dirs_exist_ok=True) in python 3.8
            copy_tree(screenshot_dir, screenshooter_results_path)

            resource = MARKET_FRONT_CMS_SCREENSHOTS(
                self, "Screenshots captured by CmsScreenShooter", screenshooter_results_path
            )
            resource_data = sdk2.ResourceData(resource)
            resource_data.ready()
