# -*- coding: utf-8 -*-

import os
import re
from distutils.dir_util import copy_tree

from sandbox import sdk2
from sandbox.common.types import misc as ctm
from sandbox.common.types import task as ctt
from sandbox.projects.market.frontarc.helpers.ArcDefaultParameters import ArcDefaultParameters
from sandbox.projects.market.frontarc.helpers.MetatronEnvArc import MetatronEnvArc
from sandbox.projects.market.frontarc.helpers.arc_base import MarketFrontArcBase
from sandbox.projects.market.frontarc.helpers.node import create_node_selector
from sandbox.projects.market.frontarc.helpers.sandbox_helpers \
    import rich_check_call, get_resource_http_proxy_link
from sandbox.projects.market.frontarc.helpers.ubuntu import create_ubuntu_selector, setup_container
from sandbox.projects.market.resources \
    import \
    MARKET_EXPERIMENT_ANALYZER_DATA_GRABBER_INPUT, \
    MARKET_EXPERIMENT_ANALYZER_DATA_GRABBER_OUTPUT
from sandbox.projects.sandbox_ci.utils import env
from sandbox.projects.market.frontarc.constants.grabber import \
    GRABBER_REPO_PATH, INPUT_FILENAME

DISK_SPACE = 3 * 1024  # 3 Gb


class MarketDataGrabberRunnerArc(sdk2.Task, MarketFrontArcBase):
    """
    Сравнение веб-элементов на различных стендах на большом количестве урлов, запуск data-grabber
    """

    root_dir = None
    app_src_dir = None
    grabber_work_dir = None
    full_config_path = None
    input_filepath = None
    output_dir = None
    output_filename = None
    output_filepath = None
    project_name = None

    class Context(sdk2.Context):
        output_resource_id = None

    class Parameters(sdk2.Task.Parameters):
        arc_params = ArcDefaultParameters()
        with sdk2.parameters.Group("Входные данные"):
            input_resource_id = sdk2.parameters.Integer(
                "Input resource id",
                description="id ресурса с input данными",
                required=True,
            )

            input_part_number = sdk2.parameters.Integer(
                "Input part number",
                description="номер части input-данных",
                required=True
            )

        with sdk2.parameters.Group("Параметры data_grabber"):
            test_domain = sdk2.parameters.String(
                "Хост для тестирования",
                required=True,
            )

            control_domain = sdk2.parameters.String(
                "Хост для контроля",
                required=True,
            )

            visitor = sdk2.parameters.String(
                "Какие данные собирать",
                required=True,
            )

            work_dir = sdk2.parameters.String(
                "Из какой директории проекта запускать тесты",
                required=True,
                default_value=None
            )

            config_path = sdk2.parameters.String(
                "Путь к конфигам из рабочей директории",
                required=True,
                default_value=None
            )

            skip = sdk2.parameters.Integer(
                "Сколько уров пропустить в инпуте",
                required=True,
                default_value=0,
            )

            limit = sdk2.parameters.Integer(
                "Сколько строчек инпута обработать",
                required=True,
                default_value=1000,
            )

        with sdk2.parameters.Group('Environment') as environ_block:
            ubuntu_version = create_ubuntu_selector()
            node_version = create_node_selector()
            environ = sdk2.parameters.Dict('Environment variables')

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64
        disk_space = DISK_SPACE

    def on_enqueue(self):
        super(MarketDataGrabberRunnerArc, self).on_enqueue()
        setup_container(self)

    def on_prepare(self):
        env.export(self.Parameters.environ)
        self.arc_mount()

        self.arc_checkout_smart(
            branch=self.Parameters.head_branch,
            pr=self.Parameters.pr_number,
            commit=self.Parameters.commit_hash
        )

        self.project_name = str(self.Parameters.root_path).split("/")[-1]

    def on_execute(self):
        with MetatronEnvArc(self, nodejs_version=self.Parameters.node_version):
            # task/
            self.root_dir = str(self.path())
            # <arcadia>/.../marketfront
            self.app_src_dir = os.path.join(self.arcadia, self.Parameters.root_path)
            # <arcadia>/.../marketfront/market/platform.desktop
            self.grabber_work_dir = os.path.join(self.app_src_dir, self.Parameters.work_dir)
            # <arcadia>/.../marketfront/market/platform.desktop/configs/data-grabber.js
            self.full_config_path = os.path.join(self.grabber_work_dir, self.Parameters.config_path)

            # task/output
            self.output_dir = os.path.join(self.root_dir, "output")
            self.output_filename = "output_{}.json".format(self.Parameters.input_part_number)
            # task/output/output_{}.json
            self.output_filepath = os.path.join(self.output_dir, self.output_filename)

            self._prepare()
            self._clone_repo()
            self._install_node_deps()
            self._get_input_filepath()
            self._run_data_grabber()

    def on_wait(self, prev_status, status):
        super(MarketDataGrabberRunnerArc, self).on_wait(prev_status, status)
        self.arc_unmount()

    def on_finish(self, prev_status, status):
        super(MarketDataGrabberRunnerArc, self).on_finish(prev_status, status)
        self._create_output_resource()
        self.arc_unmount()

    def _prepare(self):
        os.mkdir(self.output_dir)

        if 'NODE_PATH' not in os.environ:
            os.environ['NODE_PATH'] = '.'

        self.test_domain = str(self.Parameters.test_domain).strip()
        self.control_domain = str(self.Parameters.control_domain).strip()

        domain_regexp = r'(?:https?://)?([^/]+)/?'

        test_domain_match = re.search(domain_regexp, self.test_domain)
        if test_domain_match:
            self.test_domain = test_domain_match.group(1)

        control_domain_match = re.search(domain_regexp, self.control_domain)
        if control_domain_match:
            self.control_domain = control_domain_match.group(1)

    def _clone_repo(self):
        with self.memoize_stage.clone_repo(max_runs=1):
            self.arc_checkout(
                branch=self.Parameters.head_branch,
                track=True,
            )

    def _install_node_deps(self):
        with MetatronEnvArc(self, nodejs_version=self.Parameters.node_version):
            rich_check_call(
                ["npm", "run", "bootstrap"],
                task=self, alias="bootstrap", cwd=self.grabber_work_dir
            )
            # need to install peer dependencies for data-grabber
            data_grabber_dir = os.path.join(self.arcadia, GRABBER_REPO_PATH)
            rich_check_call(
                ["npm", "install"],
                task=self, alias="npm i grabber", cwd=data_grabber_dir
            )

    def _get_input_filepath(self):
        input_resource = sdk2.Resource.find(
            type=MARKET_EXPERIMENT_ANALYZER_DATA_GRABBER_INPUT,
            id=self.Parameters.input_resource_id
        ).first()

        resource_data = sdk2.ResourceData(input_resource)

        self.input_filepath = resource_data.path.joinpath(INPUT_FILENAME)

    def _run_data_grabber(self):
        # <arcadia>/market/front/libs/data-grabber/bin/dvg.js
        grabber_bin_path = os.path.join(self.arcadia, GRABBER_REPO_PATH, 'bin', 'dvg.js')
        rich_check_call(
            [
                "node",
                grabber_bin_path,
                "--config={}".format(self.full_config_path),
                "--inputFile={}".format(self.input_filepath),
                "--outputFile={}".format(self.output_filepath),
                "--testDomain={}".format(self.test_domain),
                "--controlDomain={}".format(self.control_domain),
                "--visitor={}".format(self.Parameters.visitor),
                "--skip={}".format(self.Parameters.skip),
                "--limit={}".format(self.Parameters.limit),
                "--screenshot",
                "--verbose"
            ],
            task=self, alias="data_grabber", cwd=self.grabber_work_dir
        )

    def _create_output_resource(self):
        # Don't do anything if output dir is empty
        if not os.listdir(self.output_dir):
            return

        output_resource = MARKET_EXPERIMENT_ANALYZER_DATA_GRABBER_OUTPUT(
            self,
            "data-grabber output resource",
            self.output_dir,
            type='market-analyzer-data-grabber-output',
            project=self.project_name,
            report_description=self.project_name,
            input_part_number=self.Parameters.input_part_number,
            status=ctt.Status.SUCCESS,
            root_path=self.output_filename
        )

        # Copy viewer module to output
        # Can be replaced with shutil.copytree(src, dst, dirs_exist_ok=True) in python 3.8
        copy_tree(
            os.path.join(self.arcadia, GRABBER_REPO_PATH, 'viewer'),
            self.output_dir
        )

        output_resource_data = sdk2.ResourceData(output_resource)
        output_resource_data.ready()

        if os.path.exists('output/index.html'):
            http_report_url = '{}/{}?part={}'.format(get_resource_http_proxy_link(output_resource), 'index.html', self.Parameters.input_part_number)
            self.set_info(
                "Отчёт: <a href=\"{url}\">{url}</a>".format(url=http_report_url),
                do_escape=False
            )

        self.Context.output_resource_id = output_resource.id
