# coding=utf-8

from sandbox.projects.market.frontarc.helpers.ArcDefaultParameters import ArcDefaultParameters
from sandbox.sdk2.helpers import subprocess as sp
import logging
import os
import os.path

import sandbox.common.types.misc as ctm
from sandbox.common.types import task as ctt

from sandbox import sdk2
from sandbox.common.urls import get_task_link
from sandbox.projects.market.frontarc.helpers.MetatronEnvArc import MetatronEnvArc
from sandbox.projects.market.frontarc.helpers.arc_base import MarketFrontArcBase
from sandbox.projects.market.frontarc.helpers.sandbox_helpers import rich_check_call, get_resource_http_proxy_link
from sandbox.projects.market.front.resources import MarketCIReport
from sandbox.projects.market.frontarc.helpers.node import create_node_selector
from sandbox.projects.market.frontarc.helpers.ubuntu import create_ubuntu_selector, setup_container
from sandbox.projects.market.frontarc.helpers.startrack import ReleaseTicket, cut_text, link_text, colored_text, hidden_id
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.common import task_env


class MarketFapiIntegrationArc(sdk2.Task, MarketFrontArcBase, ):
    """
    Таска для запуска интеграционных тестов для FAPI
    """

    HTML_REPORT_DIR = "html_reports"
    HTML_REPORT_FILE = "index.html"

    class Parameters(ArcDefaultParameters):

        app_src_dir = sdk2.parameters.String(
            "Кастомный путь корня приложения внутри репозитория",
            description='например: market'
        )

        base_url = sdk2.parameters.String(
            "Url FAPI инстанса",
            required=True
        )

        auth_token_vault_key = sdk2.parameters.String(
            "Название vault-key, в котором лежит oauth токен",
            required=True
        )

        testpalm_run_issue = ReleaseTicket(
            'Релизный тикет',
            description='Пример: "MARKETVERSTKA-12345"'
        )

        check_env = sdk2.parameters.Dict("Переменные окружения для запуска проверки")

        ubuntu_version = create_ubuntu_selector()
        node_version = create_node_selector()

    class Requirements(task_env.TinyRequirements):
        dns = ctm.DnsType.DNS64
        environments = [
            PipEnvironment('yandex_tracker_client', version="1.3", custom_parameters=["--upgrade-strategy only-if-needed"]),
            PipEnvironment('startrek_client', version="2.3.0", custom_parameters=["--upgrade-strategy only-if-needed"])
        ]

    def on_enqueue(self):
        super(MarketFapiIntegrationArc, self).on_enqueue()
        setup_container(self)

    def on_prepare(self):
        super(MarketFapiIntegrationArc, self).on_prepare()
        self.arc_mount()

    @property
    def check_env(self):
        env = os.environ.copy()

        env.update({
            "APP_PLATFORM": "api",
            "BLUE_PLATFORM": "api",
            "FAPI_URL": self.Parameters.base_url,
            "FAPI_AUTH_TOKEN": sdk2.Vault.data(self.Parameters.auth_token_vault_key),
        })

        if self.Parameters.check_env:
            env.update(self.Parameters.check_env)

        return env

    @property
    def html_report_path(self):
        return os.path.join(self.app_src_dir, self.HTML_REPORT_DIR, self.HTML_REPORT_FILE)

    @property
    def app_src_dir(self):
        if self.Parameters.app_src_dir:
            return os.path.join(self.arcadia, self.Parameters.root_path, self.Parameters.app_src_dir)

        return  os.path.join(self.arcadia, self.Parameters.root_path)

    def on_execute(self):
        self.report_url = ''

        with MetatronEnvArc(self, nodejs_version=self.Parameters.node_version):
            self._clone_repo()
            self._build()

            try:
                rich_check_call(
                    ["npm", "run", "integration"],
                    task=self, alias="npm_integration", cwd=self.app_src_dir, env=self.check_env
                )

            except Exception as e:
                self._exception = e
                raise
            finally:
                res = self._create_html_report_resource()
                self.report_url = get_resource_http_proxy_link(res)
                self.set_info(
                    "Отчёт: <a href=\"{url}\">{url}</a>".format(url=self.report_url),
                    do_escape=False
                )

    def on_success(self, prev_status):
        self._send_report(ctt.Status.SUCCESS)

    def on_failure(self, prev_status):
        self._send_report(ctt.Status.FAILURE)

    def on_break(self, prev_status, status):
        self._send_report(ctt.Status.FAILURE)

    def _build(self):
        rich_check_call(
            ["make"],
            task=self, alias="make", cwd=self.app_src_dir, env=self.check_env
        )

    def _clone_repo(self):
        self.arc_checkout(
            branch=self.Parameters.head_branch,
            track=True,
        )

    # Итоговый путь будет root/html_reports/index.html
    def _export_generated_report(self):
        target_report_dir = os.path.join(str(self.path()), self.HTML_REPORT_DIR)
        target_report_file = os.path.join(target_report_dir, self.HTML_REPORT_FILE)
        os.mkdir(target_report_dir)
        sp.Popen(['mv', self.html_report_path, target_report_dir])

        return target_report_file

    def _create_html_report_resource(self):
        # Выносим сгенерированный ресурс за пределы арка, иначе sdk2.Resource его не видит.
        exported_report_file = self._export_generated_report()

        report_res = MarketCIReport(self, "CI Report", exported_report_file)


        report_res_data = sdk2.ResourceData(report_res)
        report_res_data.ready()

        return report_res

    def _send_report(self, status):
        actual_issue = self.Parameters.testpalm_run_issue

        if not actual_issue:
            logging.debug("Release ticket not specified")
            return

        if status == ctt.Status.SUCCESS:
            reported_status = colored_text('green', 'PASSED')
        else:
            reported_status = colored_text('red', 'FAILED')

        task_link = link_text(get_task_link(self.id), 'Таска')

        if self.report_url:
            additional_info = link_text(self.report_url, 'Отчет здесь')
        else:
            additional_info = cut_text('Инфо из таски', self.info)

        platform_prefix = ''
        if self.Parameters.app_src_dir and str(self.Parameters.app_src_dir).strip() == 'market':
            platform_prefix = 'white'
        if self.Parameters.app_src_dir and str(self.Parameters.app_src_dir).strip() == 'api':
            platform_prefix = 'blue'

        report = '{platform}FAPI интеграционные тесты: {status}\n{task_link}\n{info}\n{hidden_id}'.format(
            status=reported_status,
            task_link=task_link,
            info=additional_info,
            hidden_id=hidden_id('{platform}fapi-integration-report'.format(platform=platform_prefix)),
            platform=platform_prefix,
        )

        oauth_token = sdk2.Vault.data("robot-metatron-st-token")

        from startrek_client import Startrek

        st = Startrek(useragent="robot-metatron", token=oauth_token)

        issue = st.issues[actual_issue]
        issue.comments.create(text=report)

        logging.debug("Report was sent to {}".format(actual_issue))
