# coding=utf-8

import os
from pydoc import describe
import subprocess
import tempfile
import pipes
import json
import logging

from sandbox import sdk2
import sandbox.common.types.misc as ctm
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.common.utils import get_task_link
from sandbox.common.errors import TaskError, TaskFailure

from sandbox.projects.market.frontarc.helpers.sandbox_helpers import rich_check_call, \
    get_resource_http_proxy_link
from sandbox.projects.market.frontarc.helpers.node import create_node_selector
from sandbox.projects.market.frontarc.helpers.ubuntu import create_ubuntu_selector, setup_container
from sandbox.projects.market.frontarc.helpers.MetatronEnvArc import MetatronEnvArc
from sandbox.projects.market.frontarc.helpers.startrack import hidden_id
from sandbox.projects.market.frontarc.helpers.ci import extract_ticket_id
from sandbox.projects.common import task_env

from sandbox.projects.market.frontarc.helpers.ArcDefaultParameters import ArcDefaultParameters
from sandbox.projects.market.frontarc.helpers.arc_base import MarketFrontArcBase

APP_REPO_OWNER = "market"
ACTUAL_SRC_DIR = "actual_src"
CHECK_SRC_DIR = "check_src"

COMPATIBLE_TYPE_CHECKER_CONFIG_PATH = "configs/compatible-type-checker.json"
REPORT_FILE_NAME = "compatible-type-checker.json"

FILE_STATUS = {
    "NOT_CHANGED": "Не изменился",
    "CHANGED": "Изменился",
    "ADDED": "Добавился",
    "REMOVED": "Удалился",
    "INVALID": "Некорректный",
}

class GenericParametersArc(ArcDefaultParameters):
    with sdk2.parameters.Group('Arcanum') as arcanum:
        actual_branch = sdk2.parameters.String(
            "Актуальная ветка", 
            description='Ветка с которой будем сравнивать типы, например trunk',
            default="trunk", 
            required=False
        )

    with sdk2.parameters.Group("Приложение") as app:

        app_src_dir = sdk2.parameters.String(
            "Кастомный путь корня приложения внутри репозитория",
            description='Например market или api',
        )

        pr_title = sdk2.parameters.String(
            "Заголовок PR"
        )

        ubuntu_version = create_ubuntu_selector()
        node_version = create_node_selector()

    with sdk2.parameters.Group("Отладка") as Dev:
        debug = sdk2.parameters.Bool(
            "Я программист",
            default=False,
            required=False
        )

        with debug.value[True]:
            checker_version = sdk2.parameters.String(
                "Compatible Type Checker version",
                default_value="latest",
                required=False,
            )

            project_conf = sdk2.parameters.String(
                "Compatible Type Checker config path",
                default_value=COMPATIBLE_TYPE_CHECKER_CONFIG_PATH,
                required=False
            )

            strict = sdk2.parameters.Bool(
                "Strict mode",
                default_value=True,
            )

            check_version_config = sdk2.parameters.Bool(
                "Use check version config",
                default_value=False,
            )

class MarketFrontApiCheckArc(sdk2.Task, MarketFrontArcBase):
    """
    Task to check files for API compatible
    """

    class Parameters(GenericParametersArc):
        pass

    class Requirements(task_env.TinyRequirements):
        dns = ctm.DnsType.DNS64
        environments = [
            PipEnvironment('yandex_tracker_client', version="1.3", custom_parameters=["--upgrade-strategy only-if-needed"]),
            PipEnvironment('startrek_client', version="2.3.0", custom_parameters=["--upgrade-strategy only-if-needed"])
        ]

    @property
    def _task_url(self):
        return get_task_link(self.id)

    @property
    def _report_url(self):
        return "{}/{}".format(get_resource_http_proxy_link(self.log_resource), REPORT_FILE_NAME)

    @property
    def report_path(self):
        return os.path.join(str(self.log_path()), REPORT_FILE_NAME)

    checker_dir = tempfile.mkdtemp()
    actual_version_root = tempfile.mkdtemp()

    @staticmethod
    def _get_report(report_path):
        with open(report_path, "r") as file:
            return json.load(file)

    def on_enqueue(self):
        super(MarketFrontApiCheckArc, self).on_enqueue()
        setup_container(self)

    def on_prepare(self):
        super(MarketFrontApiCheckArc, self).on_prepare()
        self.arc_mount()

    def on_execute(self):
        super(MarketFrontApiCheckArc, self).on_execute()

        with MetatronEnvArc(self, nodejs_version=self.Parameters.node_version):


            self._prepare_actual_version()
            self._prepare_check_version()
            self._prepare_checker()

            self._call_check()

            self._report = MarketFrontApiCheckArc._get_report(self.report_path)

            self._send_report()

    def on_success(self, prev_status):
        super(MarketFrontApiCheckArc, self).on_success(prev_status)

        if self._report["compatible"] is False and self.Parameters.strict:
            raise TaskFailure("The version being tested is not compatible")

    # Подготавливает утилиту для сравнения в папку self.checker_dir
    def _prepare_checker(self):
        rich_check_call(
            ["npm", "i",
             "--registry=http://npm.yandex-team.ru",
             "@yandex-market/compatible-type-checker@{}".format(self.Parameters.checker_version)],
            task=self,
            alias="install-checker",
            cwd=self.checker_dir
        )

    # Если задан путь к приложению внутри репы, например market, вернет market/front/apps/marketfront/market 
    # иначе market/front/apps/marketfront
    @property
    def src_dir(self):
        if self.Parameters.app_src_dir:
            return os.path.join(self.Parameters.root_path, self.Parameters.app_src_dir)

        return  self.Parameters.root_path

    # Путь к папке с тестируемой (текущей) версией
    @property
    def check_src_dir(self): 
        return  os.path.join(self.arcadia, self.src_dir)


    # Путь к папке с актуальной (образцовой) версией
    @property
    def actual_src_dir(self): 
        return  os.path.join(self.actual_version_root, self.src_dir)

    # Подготавливает текущую (проверяемую) версию в папке self.check_src_dir
    def _prepare_check_version(self):
        self.arc_checkout(self.Parameters.head_branch)

        rich_check_call(
            ["npm", "run", "bootstrap"],
            task=self,
            alias="bootstrap_check_version",
            cwd=self.check_src_dir
        )

    # Подготавливает актуальную (образцовую) версию в папке self.actual_version_root
    def _prepare_actual_version(self):

        # Экспортируем актуальную ветку во временную папку
        self.arc_export(self.Parameters.root_path, self.actual_version_root, self.Parameters.actual_branch)

        rich_check_call(
            ["npm", "run", "bootstrap"],
            task=self,
            alias="bootstrap_actual_version",
            cwd=self.actual_src_dir
        )

    # Запуск утилиты сравнения
    def _call_check(self):
        config_path = self.check_src_dir if self.Parameters.check_version_config else self.actual_src_dir

        rich_check_call(
            [
                "./node_modules/.bin/compatible-type-checker",
                "-c", os.path.join(config_path, self.Parameters.project_conf),
                "-o", self.report_path,
                "--av", self.actual_src_dir,
                "--cv", self.check_src_dir,
            ],
            task=self,
            alias="compatible-type-checker",
            cwd=self.checker_dir
        )

        if not os.path.isfile(self.report_path):
            raise TaskError("Report not found from path {}".format(self.report_path))

    def _send_report(self):
        ticket_id = extract_ticket_id(self.Parameters.pr_title)

        if ticket_id is None:
            logging.debug("The ticket number is not found")
            return

        oauth_token = sdk2.Vault.data("robot-metatron-st-token")

        from startrek_client import Startrek

        st = Startrek(useragent="robot-metatron", token=oauth_token)

        comment = self._build_ticket_comment()

        issue = st.issues[ticket_id]
        issue.comments.create(text=comment)

        logging.debug("Report was sent to {}".format(ticket_id))

    def _build_ticket_comment(self):
        is_success = self._report["compatible"] is True


        caption = "===== Результат проверки совместимости API{}: {}"
        caption_platform_suffix = self.platform_suffix()
        caption_text = "!!(green)СОВМЕСТИМО!!!" if is_success else "!!(red)НЕ СОВМЕСТИМО!!!"

        comment = caption.format(caption_platform_suffix, caption_text)
        comment += "\n\n- <{Отчет\n#|"
        comment += "**||Статус|Совместимый|Актуальный файл|Проверяемый файл|Изменения||**\n"

        for file in self._report["files"]:
            if file['status'] == 'NOT_CHANGED':
                continue

            validate_result = file["validateResult"]
            actual_file_info = file["actualFile"] or "Отсутствует"

            if validate_result and validate_result["actualFileError"]:
                actual_file_info += "\n<{{Ошибка парсинга\n%%{}%%}}>".format(validate_result["actualFileError"])

            check_file_info = file["checkFile"] or "Отсутствует"

            if validate_result and validate_result["checkFileError"]:
                check_file_info += "\n<{{Ошибка парсинга\n%%{}%%}}>".format(validate_result["checkFileError"])

            if validate_result and validate_result["diff"]:
                diff_info = "<{{JSON\n%%(json){}%%}}>".format(json.dumps(validate_result["diff"], indent=4))
            elif validate_result and validate_result["error"]:
                diff_info = "<{{Ошибка валидации\n%%{}%%}}>".format(validate_result["error"])
            else:
                diff_info = "Отсутствуют"

            comment += "||{}|{}|{}|{}|{}||".format(
                FILE_STATUS[file["status"]],
                "!!(green)Да!!" if file["compatible"] else "!!(red)Нет!!",
                actual_file_info,
                check_file_info,
                diff_info
            )

        comment += '|#}>'
        comment += "\n- (({} Таск.))".format(self._task_url)
        comment += '\n{}'.format(hidden_id('fapi-check-report'))

        return comment

    def platform_suffix(self):
        if self.Parameters.project_name == "marketfront":
            if self.Parameters.app_src_dir == "market":
                return " (белый)"
            if self.Parameters.app_src_dir == "api":
                return " (синий)"

        return ""
