# -*- coding: utf-8 -*-
import logging
import os
import pipes

import sandbox.sdk2.path as spath
from sandbox import sdk2
from sandbox.projects.market.frontarc.helpers.MetatronEnvArc import MetatronEnvArc
from sandbox.projects.market.frontarc.helpers.github import clone_repo_for_diff
from sandbox.projects.market.frontarc.helpers.sandbox_helpers import rich_check_call
from sandbox.projects.market.frontarc.helpers.ubuntu import create_ubuntu_selector, setup_container
from sandbox.projects.market.frontarc.helpers.node import NODE_DEFAULT
from sandbox.projects.market.frontarc.helpers.node import create_node_selector
from sandbox.projects.market.resources import MARKET_TESTPALM_IDS
from sandbox.projects.sandbox_ci.utils import env

GH_OWNER = 'MARKET'


class MarketFrontAutotestFilterArc(sdk2.Task):
    """
    Таска для генерации списка TestPalm id, отфильтрованных по указанному имени фильтра
    """

    ROOT_DIR = ''
    APP_SRC_PATH = ''

    class Parameters(sdk2.Task.Parameters):
        ubuntu_version = create_ubuntu_selector()
        node_version = create_node_selector()

        with sdk2.parameters.Group('GitHub репозиторий проекта') as github_repo_block:
            app_owner = sdk2.parameters.String(
                'GitHub owner',
                description='Логин владельца репозитория или название организации',
                default=GH_OWNER,
                required=True
            )
            app_repo = sdk2.parameters.String(
                "Репозиторий",
                default='partnernode',
                required=True
            )
            app_base_branch = sdk2.parameters.String(
                "Базовая ветка",
                default='master',
                required=True
            )
            app_compare_branch = sdk2.parameters.String(
                "Рабочая ветка",
                default='master',
                required=True
            )

        with sdk2.parameters.Group("Настройки приложения") as app_settings:
            filter_type = sdk2.parameters.String(
                "Имя фильтра (critical, sox, etc...)",
                default='',
                required=True
            )

        with sdk2.parameters.Group('Environment') as environ_block:
            environ = sdk2.parameters.Dict('Environment variables')

    def on_enqueue(self):
        super(MarketFrontAutotestFilterArc, self).on_enqueue()
        setup_container(self)

    @property
    def node_version(self):
        return self.Parameters.node_version or NODE_DEFAULT

    @property
    def project_name(self):
        return self.Parameters.app_repo

    @property
    def report_description(self):
        return self.project_name

    @property
    def TESTPALM_API_TOKEN(self):
        return os.environ["TESTPALM_API_TOKEN"]

    def _clone_repo(self):
        clone_repo_for_diff(
            pipes.quote(self.Parameters.app_owner),
            pipes.quote(self.Parameters.app_repo),
            pipes.quote(self.Parameters.app_base_branch),
            pipes.quote(self.Parameters.app_compare_branch),
            self.APP_SRC_PATH
        )

    def _prepare_dirs(self):
        self.ROOT_DIR = str(self.path())
        self.APP_SRC_PATH = os.path.join(
            self.ROOT_DIR, self.Parameters.app_repo)

    def _prepare_env(self):
        # export order for correct priority
        os.environ['TESTPALM_ACCESS_TOKEN'] = self.TESTPALM_API_TOKEN
        env.export(self.Parameters.environ)

    def _configure(self):
        rich_check_call(
            ["make", "configure"],
            task=self, alias="configure", cwd=self.APP_SRC_PATH
        )

    def _build(self):
        logging.info(
            'Running project build with environment: {}'.format(os.environ))

        rich_check_call(
            ["make", "bootstrap"],
            task=self, alias="bootstrap", cwd=self.APP_SRC_PATH
        )

    def _run_testpalm_scope(self):
        logging.info(
            'Running testpalm scripts'
        )

        rich_check_call(
            ["npm", "run", "testcases-filter", self.Parameters.filter_type],
            task=self, alias="testcases-filter", cwd=self.APP_SRC_PATH
        )

        src_testpalm_ids_report_path = os.path.join(
            self.APP_SRC_PATH, 'txt_reports/testpalm_ids_report')
        testpalm_ids_report = spath.Path(
            src_testpalm_ids_report_path).read_bytes()

        resource = sdk2.ResourceData(MARKET_TESTPALM_IDS(
            self, "TestPalm IDS", "testpalm_ids"
        ))
        resource.path.write_bytes(testpalm_ids_report)
        resource.ready()

    def on_prepare(self):
        with MetatronEnvArc(self, nodejs_version=self.Parameters.node_version), self.memoize_stage.preparation(max_runs=1):
            self._prepare_dirs()
            self._prepare_env()
            self._clone_repo()
            self._configure()
            self._build()

    def on_execute(self):
        with MetatronEnvArc(self, nodejs_version=self.Parameters.node_version), self.memoize_stage.execute(max_runs=1):
            self._prepare_dirs()
            self._prepare_env()
            self._run_testpalm_scope()
