# -*- coding: utf-8 -*-

import json
import logging
import os
import tempfile
from typing import List, Set

from sandbox import sdk2
from sandbox.common.errors import InvalidResource, TaskFailure
from sandbox.common.types import misc as ctm
from sandbox.projects.common import task_env

from sandbox.projects.market.frontarc.MarketAutotestsHermioneArc import MarketAutotestsHermioneArc
from sandbox.projects.market.frontarc.helpers.MetatronEnvArc import MetatronEnvArc
from sandbox.projects.market.frontarc.helpers.sandbox_helpers import get_resource_http_proxy_link
from sandbox.projects.market.frontarc.helpers.ubuntu import create_ubuntu_selector, setup_container
from sandbox.projects.market.resources import MARKET_AUTOTEST_REPORT, MARKET_FRONT_AUTOTEST_REPORT_COMPARATOR_RESULT
from sandbox.sandboxsdk.environments import PipEnvironment

DISK_SPACE = 3 * 1024  # 3 Gb
SUITES_FILE_NAME = "suites.json"
SUITES_PATH = os.path.join("data", SUITES_FILE_NAME)
COMPARISON_RESULT_FILENAME = "result.json"
OK_STATUS = "OK"
WARN_STATUS = "WARN"
CRIT_STATUS = "CRIT"
NEW_FAILED = "new_failed"
OLD_FAILED = "old_failed"
BROKEN_TEST_STATUS_LIST = ["failed", "broken"]
TICKET_TEXT = "====Сравниваем прогоны автотестов {}. Сравнение результатов"

class MarketFrontAutotestReportComparatorArc(sdk2.Task):
    """
    Сравнивает между собой прогоны из двух тасок
    Результат складывает в файлик отчёта
    """
    root_dir = None
    first_report_data = None
    second_report_data = None
    first_resource_branch = None
    second_resource_branch = None
    first_resource = None
    second_resource = None
    test_type = None
    first_failed_tests = None
    second_failed_tests = None
    compare_result = None
    compare_status = None

    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.Group("Таски") as release_versions:
            first_resource_id = sdk2.parameters.String(
                "Первый ресурс",
                description="Первый ресурс с отчетом для сравнения",
            )

            second_resource_id = sdk2.parameters.String(
                "Второй ресурс",
                description="Второй ресурс с отчетом для сравнения",
            )

        with sdk2.parameters.Group("Конфиги") as configs:
            crit_diff = sdk2.parameters.Integer(
                "Разница для сигнала",
                default=10,
                description="Какой дифф в количестве упавших считать критичным"
            )

            ticket = sdk2.parameters.String(
                "Тикет для отчета",
                description="В тикете будет ссылка на отчёт и форматированная версия отчёта. Если не нужно - оставить пустым ",
            )

            crit_tag = sdk2.parameters.String(
                "Тег в тикете при крите",
                description="В случае критической разницы будет проставлен тег (если поле заполнено)",
                required=False
            )

            fail_task_after_crit_diff = sdk2.parameters.Bool(
                "Ронять ли таску в случае крит разницы",
                description="В случае критической разницы sandbox таска будет зафейлена",
                default=False
            )

            ubuntu_version = create_ubuntu_selector()

    class Requirements(task_env.TinyRequirements):
        dns = ctm.DnsType.DNS64
        disk_space = DISK_SPACE
        environments = [
            PipEnvironment("yandex_tracker_client", version="1.3", custom_parameters=["--upgrade-strategy only-if-needed"]),
            PipEnvironment("startrek_client", version="2.3.0", custom_parameters=["--upgrade-strategy only-if-needed"])
        ]

    def on_enqueue(self):
        super(MarketFrontAutotestReportComparatorArc, self).on_enqueue()
        setup_container(self)

    def on_prepare(self):
        self.root_dir = tempfile.mkdtemp()

    def on_execute(self):
        self._get_results()
        self._compare_results()
        self._create_a_resource_with_comparison_result()
        self._send_startrek_report_to_release_ticket()
        self._end_task()

    def _get_results(self):
        first_report = self._get_report(self.Parameters.first_resource_id)
        second_report = self._get_report(self.Parameters.second_resource_id)

        self.first_report_data = first_report[0]
        self.first_resource_branch = first_report[1].Parameters.head_branch or "trunk"
        first_test_type = first_report[1].Parameters.project_build_context
        self.first_resource = first_report[2]

        self.second_report_data = second_report[0]
        self.second_resource_branch = second_report[1].Parameters.head_branch or "trunk"
        second_test_type = second_report[1].Parameters.project_build_context
        self.second_resource = second_report[2]

        if first_test_type != second_test_type:
            logging.error("Типы тестов не совпадают %s %s" % (first_test_type, second_test_type))
            raise TaskFailure("Test types are different %s %s" % (first_test_type, second_test_type))

        self.test_type = first_test_type

        self.first_failed_tests = set(find_all_test_names_recursive(self.first_report_data, BROKEN_TEST_STATUS_LIST))  # type: Set[str]
        self.second_failed_tests = set(find_all_test_names_recursive(self.second_report_data, BROKEN_TEST_STATUS_LIST))  # type: Set[str]

    def _compare_results(self):
        self.compare_result = self._compare_test_sets(self.first_failed_tests, self.second_failed_tests)
        self.compare_status = self.compare_result["status"].strip()

    def _get_report(self, report_id):
        report = sdk2.Resource.find(
            resource_type=MARKET_AUTOTEST_REPORT,
            id=report_id
        ).first()

        if not report:
            logging.error("Не найден ресурс MARKET_AUTOTEST_REPORT автотестов с идентификатором %s" % (self.Parameters.first_resource_id))
            raise TaskFailure("Resource MARKET_AUTOTEST_REPORT not found  with id %s" % (self.Parameters.first_resource_id))

        parent_task = sdk2.Task.find(
            task_type=MarketAutotestsHermioneArc,
            id=report.task_id,
            children=True
        ).first()

        if not parent_task:
            logging.error("Не найдена родительская таска MARKET_AUTOTESTS_HERMIONE_ARC автотестов с идентификатором %s" % (str(report.task_id)))
            raise TaskFailure("Parent task MARKET_AUTOTESTS_HERMIONE_ARC not found with id %s" % (str(report.task_id)))

        resource_data = sdk2.ResourceData(report)

        result_json_path = os.path.join(str(resource_data.path), SUITES_PATH)

        with open(result_json_path) as result_json:
            # return [json.load(result_json), parent_task.Parameters.head_branch or "trunk", parent_task.Parameters.project_build_context]
            return [json.load(result_json), parent_task, report]

    def _compare_test_sets(self, first_set, second_set):
        # type: (Set[str], Set[str]) -> dict
        # Тут приведение обратно к list, потому что set - не сериализуется json
        first_failed = list(second_set.intersection(first_set))
        second_failed = list(second_set.difference(first_set))

        diff_size = len(second_failed)

        if diff_size == 0:
            return {"status": OK_STATUS, NEW_FAILED: second_failed, OLD_FAILED: first_failed}

        if diff_size < int(self.Parameters.crit_diff):
            return {"status": WARN_STATUS + " упало %d новых тестов" % diff_size, NEW_FAILED: second_failed, OLD_FAILED: first_failed}

        return {"status": CRIT_STATUS + " В ветке %s упало %d новых тестов по сравнению с %s" % (self.second_resource_branch, diff_size, self.first_resource_branch), NEW_FAILED: second_failed, OLD_FAILED: first_failed}

    def _create_a_resource_with_comparison_result(self):
        logging.info("Записываем результат в json-файл")
        with open(COMPARISON_RESULT_FILENAME, "w") as f:
            json.dump(self.compare_result, f, indent=4, ensure_ascii=False)

        # 3 раза пытаемся создать ресурс с отчётом
        max_retry_number = 3
        for i in range(0, max_retry_number):
            try:
                logging.info("Пытаемся создать ресурс с результатом таски")
                self.result_resource = MARKET_FRONT_AUTOTEST_REPORT_COMPARATOR_RESULT(
                    task=self,
                    description="Результат сравнения автотестов бранча %s с %s" % (self.second_resource_branch, self.first_resource_branch),
                    path=COMPARISON_RESULT_FILENAME,
                )
                sdk2.ResourceData(self.result_resource).ready()
                logging.info("Ресурс создан")
                # Выходим из цикла ретраев
                break
            except InvalidResource as ex:
                if i + 1 < max_retry_number:
                    logging.info(str(ex))
                    continue
                else:
                    raise ex

    def _send_startrek_report_to_release_ticket(self):
        if self.Parameters.ticket == "":
            return

        logging.info("Пишем результат прогона в тикет")
        with MetatronEnvArc(self):

            from startrek_client import Startrek
            oauth_token = sdk2.Vault.data("robot-metatron-st-token")
            st = Startrek(useragent="robot-metatron", token=oauth_token)

            current_issue = None
            try:
                current_issue = st.issues[self.Parameters.ticket]
            except NotFound:
                logging.info("Тикет не найден. Отчёт не пишем.")

            if not current_issue:
                logging.info("Релизный тикет не найден. Отчёт не пишем.")
                return

            comment_text = self._comment_text_from_result()
            logging.info("Записываем комментарий %s в тикет" % comment_text)
            comments = list(current_issue.comments.get_all())

            for comment in comments:
                if (TICKET_TEXT.format(self.test_type)) in comment.text:
                    comment.delete()

            current_issue.comments.create(text=comment_text)

            if self.Parameters.crit_tag:
                crit_tag_full = self.Parameters.crit_tag + '_' + self.test_type
                if self.compare_status.find(CRIT_STATUS) >= 0:
                    logging.info("Добавляем тег %s" % crit_tag_full)
                    st.issues[current_issue.key].update(
                        tags={'add': [crit_tag_full]}
                    )
                else:
                    logging.info("Убираем тег %s" % crit_tag_full)
                    st.issues[current_issue.key].update(
                        tags={'remove': [crit_tag_full]}
                    )

            logging.info("Записали отчёт в тикет")

    def _comment_text_from_result(self):
        comment_strings = [
            TICKET_TEXT.format(self.test_type),
            " прогонов: %s (((%s %s))) с %s (((%s %s)))\n" % (
                self.second_resource_branch,
                self.second_resource.url,
                self.second_resource.id,

                self.first_resource_branch,
                self.first_resource.url,
                self.first_resource.id,
            ),
            "((%s Полный отчёт))\n" % (get_resource_http_proxy_link(self.result_resource))
        ]

        res = self.compare_result
        status = self.compare_status

        try:
            new_failed = res[NEW_FAILED]
            if len(new_failed) > 0:
                new_failed = ["%%" + name + "%%" for name in new_failed]
                comment_strings.append("<{diff:\n%s\n}>\n" % "\n\n".join(new_failed))
        except KeyError:
            # Если нет new_failed - ничего не делаем
            pass

        if status.find(OK_STATUS) >= 0:
            comment_strings.append("====!!(green)%s!!\n" % status)
            pass
        elif status.find(WARN_STATUS) >= 0:
            comment_strings.append("====##%s##\n" % self.test_type)
            comment_strings.append("====!!(yellow)%s!!\n" % status)
        elif status.find(CRIT_STATUS) >= 0:
            comment_strings.append("===##%s##\n" % self.test_type)
            comment_strings.append("====!!(red)%s!!\n" % status)
        else:
            comment_strings.append("##%s##\n" % self.test_type)
            comment_strings.append("====%s\n" % status)

        return "".join(comment_strings)

    def _end_task(self):
        if self.Parameters.fail_task_after_crit_diff and self.compare_status.find(CRIT_STATUS) >= 0:
            raise TaskFailure("Crit diff in reports")

def find_all_test_names_recursive(data, status_list):
    # type: (dict, List[str]) -> List[str]
    test_names = []
    if "children" in data:
        for child in data["children"]:
            test_names += find_all_test_names_recursive(child, status_list)
        return test_names

    # Мы дошли до листа дерева, и надо его записать в список скипов
    try:
        status = data["status"]  # type: str
        name = data["name"]  # type: str
    except KeyError as e:
        logging.warning("Не удалось получить атрибуты `name` и `status` у объекта. Объект:\n%s" % json.dumps(data), exc_info=e)
        return test_names

    if status in status_list:
        test_names.append(name)

    return test_names
