# coding=utf-8

import logging
import os

from sandbox import sdk2
import sandbox.common.types.misc as ctm

from sandbox.projects.market.resources import MARKET_FRONT_BUNDLE_STATS
from sandbox.projects.market.frontarc.helpers.path import raise_if_path_not_exists
from sandbox.projects.market.frontarc.helpers.MetatronEnvArc import MetatronEnvArc
from sandbox.projects.market.frontarc.helpers.node import create_node_selector
from sandbox.projects.market.frontarc.helpers.ubuntu import create_ubuntu_selector, setup_container
from sandbox.projects.market.frontarc.helpers.github import clone_repo
from sandbox.projects.market.frontarc.helpers.sandbox_helpers import rich_check_call

DISK_SPACE = 6 * 1024


class MarketFrontBuildBundleStatsArc(sdk2.Task):
    """
    Таска для создания артефакта сборки Statoscope
    """

    APP_REPO_DIR = "app_src"

    class Parameters(sdk2.Task.Parameters):
        ubuntu_version = create_ubuntu_selector()
        node_version = create_node_selector()

        github_owner = sdk2.parameters.String(
            "Github owner",
            required=True
        )

        github_repo = sdk2.parameters.String(
            "Github repo",
            required=True
        )

        github_branch = sdk2.parameters.String(
            "Github branch",
            required=True
        )

        app_ref = sdk2.parameters.String(
            "Application reference",
            description="Identifies an app of repository. For example, 'market_touch'",
            required=True
        )

        bundle_ref = sdk2.parameters.String(
            "Bundle reference",
            description="Identifies an bundle of repository. For example, 'client'",
            required=True
        )

        app_dir = sdk2.parameters.String(
            "Application root dir",
            description="For example, 'market/platform.touch'. In that folder 'build_stats_command' will be executed.",
            required=True
        )

        build_stats_artefacts_dir = sdk2.parameters.String(
            "Dir to the stats build artefacts (from root)",
            description="For example, 'market/platform.desktop/statoscope_reports'",
            required=True
        )

        bundle_stats_filename = sdk2.parameters.String(
            "Bundle stats filename",
            required=True
        )

        build_stats_command = sdk2.parameters.String(
            "Command to run script to build bundle stats",
            description="For example, 'npm run statoscope:stats'."
                        "Keep in mind that that is important to build only one selected bundle.",
            required=True
        )

    @property
    def _repo_path(self):
        """
        Absolute path to the repository
        """
        root_path = str(self.path())

        return os.path.join(root_path, self.APP_REPO_DIR)

    @property
    def _app_path(self):
        """
        Absolute path to the app
        """
        return os.path.join(self._repo_path, self.Parameters.app_dir)

    @property
    def _artefact_path(self):
        """
        Absolute path to the build artefacts
        """
        return os.path.join(
            self._repo_path,
            self.Parameters.build_stats_artefacts_dir,
            self.Parameters.bundle_stats_filename
        )

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64
        disk_space = DISK_SPACE

    def on_enqueue(self):
        super(MarketFrontBuildBundleStatsArc, self).on_enqueue()
        setup_container(self)

    def on_execute(self):
        super(MarketFrontBuildBundleStatsArc, self).on_execute()

        with MetatronEnvArc(self, nodejs_version=self.Parameters.node_version):
            self._clone_repo()
            self._build_stats_artefacts()
            self._check_artefacts_exist()
            self._create_stats_resource()

    def _clone_repo(self):
        logging.info("Cloning repo")

        clone_repo(
            owner=self.Parameters.github_owner,
            name=self.Parameters.github_repo,
            ref=self.Parameters.github_branch,
            target_dir=self.APP_REPO_DIR
        )

    def _build_stats_artefacts(self):
        logging.info("Building stats artefacts")

        rich_check_call(
            ["bash", "-c", self.Parameters.build_stats_command],
            task=self, alias="Build bundle stats artefacts", cwd=self._app_path
        )

    def _check_artefacts_exist(self):
        logging.info("Checking bundle stats artefacts exist")

        raise_if_path_not_exists(
            self._artefact_path,
            "Can't create stats resource, {} not exists".format(self._artefact_path)
        )

    def _create_stats_resource(self):
        logging.info("Creating bundle stats resources")

        resource = MARKET_FRONT_BUNDLE_STATS(
            task=self,
            description="Bundle stats",
            path=self._artefact_path,
            app_ref=self.Parameters.app_ref,
            bundle_ref=self.Parameters.bundle_ref
        )
        resource_data = sdk2.ResourceData(resource)
        resource_data.ready()
