# coding=utf-8

import logging
import os
import tempfile

import sandbox.common.types.misc as ctm
import sandbox.sdk2.path as spath
from sandbox import sdk2
from sandbox.projects.market.frontarc.helpers.ArcDefaultParameters import ArcDefaultParameters
from sandbox.projects.market.frontarc.helpers.MetatronEnvArc import MetatronEnvArc
from sandbox.projects.market.frontarc.helpers.arc_base import MarketFrontArcBase
from sandbox.projects.market.frontarc.helpers.node import NODE_DEFAULT
from sandbox.projects.market.frontarc.helpers.node import create_node_selector
from sandbox.projects.market.frontarc.helpers.sandbox_helpers import rich_check_call
from sandbox.projects.market.frontarc.helpers.ubuntu import create_ubuntu_selector, setup_container
from sandbox.projects.market.resources import MARKET_APP
from sandbox.projects.sandbox_ci.utils import env
from sandbox.projects.common import task_env

KILL_TIMEOUT = 30 * 60  # 30 min
DISK_SPACE = 3 * 1024  # 3 Gb
GH_OWNER = 'MARKET'


# todo: выпилить сборку из таски с автотестами (MarketAutotests, в сути -- копипаста оттуда)
class MarketFrontBuildDevArc(sdk2.Task, MarketFrontArcBase):
    """
    Сборка приложения в Dev окружении. Для автотестов и CI
    """

    APP_SRC_PATH = ''
    root_dir = ''
    app_branch_final = ''

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = KILL_TIMEOUT
        ubuntu_version = create_ubuntu_selector()
        node_version = create_node_selector()

        arc_params = ArcDefaultParameters()

        with sdk2.parameters.Group('Настройки таски') as task_param_block:
            app_src_dir = sdk2.parameters.String("Кастомный путь корня приложения внутри репозитория")

            use_release_branch = sdk2.parameters.Bool(
                'Использовать текущую релизную ветку',
                description='Отметьте, если нужно поискать ветку с текущим релизом, и прогнать АТ из неё '
                            '(если не будет найдена, прогонится из ветки, указанной в app_branch)',
                default=False
            )
            with use_release_branch.value[True]:
                app_release_branch_starts_with = sdk2.parameters.String(
                    'Начало названия релизной ветки для поиска',
                    default='release/',
                    required=True
                )

        with sdk2.parameters.Group('Environment') as environ_block:
            environ = sdk2.parameters.Dict('Environment variables')

    class Requirements(task_env.BuildLinuxRequirements):
        dns = ctm.DnsType.DNS64
        disk_space = DISK_SPACE
        cores = 4
        ram = 1024 * 10

    def on_enqueue(self):
        super(MarketFrontBuildDevArc, self).on_enqueue()
        setup_container(self)

    def on_prepare(self):
        super(MarketFrontBuildDevArc, self).on_prepare()
        self.arc_mount()

        self._prepare_env()
        self._prepare_app_branch()

        self.arc_checkout_smart(
            branch=self.app_branch_final,
            commit=self.Parameters.commit_hash
        )

    def on_finish(self, prev_status, status):
        super(MarketFrontBuildDevArc, self).on_finish(prev_status, status)
        self.arc_unmount()

    def on_wait(self, prev_status, status):
        super(MarketFrontBuildDevArc, self).on_wait(prev_status, status)
        self.arc_unmount()

    def on_execute(self):
        super(MarketFrontBuildDevArc, self).on_execute()

        # NB: MetatronEnvArc clears env on exit
        with MetatronEnvArc(self, nodejs_version=self.Parameters.node_version), self.memoize_stage.preparation(max_runs=1):
            self.APP_SRC_PATH = os.path.join(self.arcadia, self.Parameters.root_path)

            if self.Parameters.app_src_dir:
                self.APP_SRC_PATH = os.path.join(self.APP_SRC_PATH, self.Parameters.app_src_dir)

            self._configure()
            self._build()
            self._create_app_pack()

    @property
    def project_name(self):
        parts = self.Parameters.root_path.split('/')
        return parts[-1]

    @property
    def node_version(self):
        return self.Parameters.node_version or NODE_DEFAULT

    @property
    def app_resource_id(self):
        return self._app_resource_id

    @app_resource_id.setter
    def app_resource_id(self, value):
        self._app_resource_id = value

    def _configure(self):
        rich_check_call(
            ["make", "configure"],
            task=self, alias="configure", cwd=self.APP_SRC_PATH
        )

    def _build(self):
        logging.info('Running project build with environment: {}'.format(os.environ))

        rich_check_call(
            ["make", "bootstrap"],
            task=self, alias="bootstrap", cwd=self.APP_SRC_PATH
        )

    def _create_app_pack(self):
        pack_app_archive_path = tempfile.mktemp(suffix=".tar.gz", prefix=self.project_name)
        dir_path = os.path.join(self.arcadia, self.Parameters.root_path, '..')

        rich_check_call(
            ["tar", "-C", dir_path, "-czf", pack_app_archive_path, self.project_name],
            task=self, alias="create_app_pack"
        )

        self._app_create_resource(pack_app_archive_path)

    def _app_create_resource(self, pack_app_archive_path):
        resource = MARKET_APP(
            self, "App tarball", "{}.tar.gz".format(self.project_name),
            app_repo=self.project_name,
            app_branch=self.app_branch_final,
        )
        app_res = sdk2.ResourceData(resource)

        app_res.path.write_bytes(spath.Path(pack_app_archive_path).read_bytes())
        app_res.ready()

        self.app_resource_id = resource.id

    def _prepare_env(self):
        # NB: last export overrides previous
        env.export(self.Parameters.environ)
        return

    def _prepare_app_branch(self):
        self.app_branch_final = self.Parameters.head_branch

        if self.Parameters.use_release_branch:
            self.arc_fetch(branch=self.Parameters.app_release_branch_starts_with)
            arc_branch_output = self.arc_branch(all=True, as_dict=True)

            for branch_entry in arc_branch_output:
                if branch_entry['name'].startswith(self.Parameters.app_release_branch_starts_with):
                    self.app_branch_final = branch_entry['name']
                    break
