# coding=utf-8

from sandbox import sdk2
from sandbox.projects.market.frontarc.helpers.MetatronEnvArc import MetatronEnvArc
from sandbox.projects.market.frontarc.helpers.node import create_node_selector
from sandbox.projects.market.frontarc.helpers.ubuntu import create_ubuntu_selector, setup_container
from sandbox.projects.market.frontarc.helpers.arc_base import MarketFrontArcBase
from sandbox.projects.market.frontarc.helpers.compression import compress_brotli, compress_zopfli
import sandbox.projects.market.frontarc.helpers.s3 as s3

import os
import logging


class MarketFrontBuildExportStaticArc(sdk2.Task, MarketFrontArcBase):
    """
    Task for uploading export-static files to s3
    market/front/libs/export-static

    Filters some files internally
    - Readme.md
    - _scripts/
    - `^\..*` files

    If `trunk` is chosen as branch will upload all files
    If `some-branch` is chosen as branch will upload its diff to trunk
    """

    __RESOURCES_DIR = os.path.join(os.path.dirname(__file__), "..", "helpers")
    __EXPORT_STATIC_PATH = "market/front/libs/export-static"
    __APP_SRC_PATH = ""
    __S3_PREFIX = s3.s3_path_by_repo(__EXPORT_STATIC_PATH)

    class Parameters(sdk2.Task.Parameters):
        logging.error('HelloMoto')
        ubuntu_version = create_ubuntu_selector()
        node_version = create_node_selector()
        with sdk2.parameters.Group('Arcanum') as arcanum:
            # Сделано с оглядкой на sandbox.projects.market.frontarc.helpers.ArcDefaultParameters
            # Но путь, например, у нас хардкоженный, поэтому обязательный параметр оттуда нам не нужен
            head_branch = sdk2.parameters.String("Ветка, с которой будет запущена загрузка в s3", required=True)

            with head_branch.value["trunk"]:
                # Если выбрать trunk, то закачается весь репозиторий в s3.
                # Это дорого, и нужно только, если нам зачем-то понадобилось сделать именно это.
                # Для обычного сценария хватит загрузки диффа с ветки
                i_understand_repercussions_checkbox = sdk2.parameters.Bool(
                    "Я осознаю все последствия выбора trunk в качестве ветки, и я проконсультировался с инфраструктурой"
                    " перед запуском таски",
                    default=False,
                    required=True
                )

    def on_enqueue(self):
        super(MarketFrontBuildExportStaticArc, self).on_enqueue()
        setup_container(self)

    def on_prepare(self):
        super(MarketFrontBuildExportStaticArc, self).on_prepare()
        self.arc_mount()

    def on_finish(self, prev_status, status):
        super(MarketFrontBuildExportStaticArc, self).on_finish(prev_status, status)
        self.arc_unmount()

    def on_execute(self):
        super(MarketFrontBuildExportStaticArc, self).on_execute()

        self.__APP_SRC_PATH = os.path.join(
            self.arcadia, self.__EXPORT_STATIC_PATH
        )

        os.environ.update({
            "PATH": ":".join([
                os.environ['PATH'],
                os.path.dirname(self.arc.binary_path)
            ])
        })

        logging.info({
            "export_static_path": self.__EXPORT_STATIC_PATH,
            "APP_SRC_PATH": self.__APP_SRC_PATH,
            "root": self.arcadia
        })

        with MetatronEnvArc(self, nodejs_version=self.Parameters.node_version):
            conn = s3.get_connection()

            self.__prepare_repos()
            self.__upload(conn)

    def __prepare_repos(self):
        self.arc_checkout(branch=self.Parameters.head_branch)

    def __get_branch_files_to_upload(self):
        show_output = self.arc_show(as_dict=True, name_status=True)
        file_arc_paths = [
            os.path.join(self.arcadia, file_desc["path"])
            for file_desc in show_output[0]["names"]
            if (
                file_desc["path"].startswith(self.__EXPORT_STATIC_PATH)
                and file_desc["status"] in ["modified", "new file"]
            )
        ]

        logging.info("DIFF FILES")
        logging.info(file_arc_paths)

        return file_arc_paths

    def __get_trunk_files_to_upload(self):
        file_arc_paths = []

        for path, subdirs, files in os.walk(self.__APP_SRC_PATH):
            for file_name in files:
                file_arc_paths.append(os.path.join(path, file_name))

        return file_arc_paths

    def __make_rel_paths(self, file_arc_paths):
        result = []

        for path in file_arc_paths:
            # >>> os.path.relpath('market/front/libs/export-static/_/i/foo/bar.png', 'market/front/libs/export-static')
            # '_/i/foo/bar.png'
            rel_path = os.path.relpath(path, self.__APP_SRC_PATH)
            name = os.path.basename(rel_path)

            # Фильтруем то, что не стоит выкладывать
            if (
                rel_path.startswith('_scripts') or
                name.lower() == 'readme.md' or
                name.startswith('.')
            ):
                continue

            result.append((path, rel_path))

        return result

    def __upload(self, s3_connection):
        files = self.__make_rel_paths(
            self.__get_trunk_files_to_upload()
            if self.Parameters.head_branch == "trunk"
            else self.__get_branch_files_to_upload()
        )

        logging.info("Will compress and upload {} files".format(len(files)))
        logging.info("Will compress and upload next files \n{}".format("\n".join([f[0] for f in files])))
        compression_callbacks = []

        # Создаем таски на пожатие файлов
        for (abs_path, rel_path) in files:
            filename = os.path.basename(rel_path)

            _, __, check_zopfli_report_cb = compress_zopfli(self, abs_path, filename)
            _, __, check_brotli_report_cb = compress_brotli(self, abs_path, filename, check_size=False)

            compression_callbacks.append((
                check_zopfli_report_cb,
                check_brotli_report_cb
            ))

        # Загружаем исходник и пожатые варианты. Предварительно проверяем, что всё файл успешно пожался
        for (
            (abs_path, rel_path),
            (check_zopfli_report_cb, check_brotli_report_cb)
        ) in zip(files, compression_callbacks):
            check_zopfli_report_cb()
            check_brotli_report_cb()
            uploading_path = os.path.join(self.__S3_PREFIX, rel_path)

            logging.info('Uploading `{}` to `{}`'.format(abs_path, uploading_path))
            s3.upload_file(
                s3_connection,
                s3.S3_BUCKET_NAME,
                abs_path,
                uploading_path
            )

            logging.info('Uploading gz/br to: {}'.format(uploading_path))

            s3.upload_file(s3_connection, s3.S3_BUCKET_NAME, abs_path + ".gz", "{}.gz".format(uploading_path))
            s3.upload_file(s3_connection, s3.S3_BUCKET_NAME, abs_path + ".br", "{}.br".format(uploading_path))
