# coding=utf-8

from sandbox import sdk2
import sandbox.sdk2.path as spath

import os.path
import tempfile
import sandbox.common.types.misc as ctm
from sandbox.projects.common.nanny import nanny

from sandbox.projects.market.frontarc.helpers.ArcDefaultParameters import ArcDefaultParameters
from sandbox.projects.market.frontarc.helpers.arc_base import MarketFrontArcBase
from sandbox.projects.market.frontarc.helpers.sandbox_helpers import rich_check_call
from sandbox.projects.market.frontarc.helpers.node import create_node_selector
from sandbox.projects.market.frontarc.helpers.ubuntu import create_ubuntu_selector, setup_container
from sandbox.projects.market.frontarc.helpers.MetatronEnvArc import MetatronEnvArc
from sandbox.projects.sandbox_ci.utils import env

from sandbox.projects.market.resources import MARKET_KADAVR_APP

class MarketFrontBuildKadavrArc(nanny.ReleaseToNannyTask2, sdk2.Task, MarketFrontArcBase):
    """
    Сборка Кадавра в ресурс
    """

    APP_SRC_PATH = ''

    class Parameters(sdk2.Task.Parameters):
        arc_params = ArcDefaultParameters()

        with sdk2.parameters.Group("Сборка") as app:
            ubuntu_version = create_ubuntu_selector()
            node_version = create_node_selector()
            prune = sdk2.parameters.Bool("Делать prune зависимостей", default=True, required=True)

        with sdk2.parameters.Group('Environment') as environ_block:
            environ = sdk2.parameters.Dict('Environment variables')

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64

    def on_enqueue(self):
        super(MarketFrontBuildKadavrArc, self).on_enqueue()
        setup_container(self)

    def on_prepare(self):
        super(MarketFrontBuildKadavrArc, self).on_prepare()
        self.arc_mount()

        self._prepare_env()
        self.arc_checkout_default()

    def on_execute(self):
        super(MarketFrontBuildKadavrArc, self).on_execute()

        with MetatronEnvArc(self, nodejs_version=self.Parameters.node_version):
            self.APP_SRC_PATH = os.path.join(self.arcadia, self.Parameters.root_path)

            self._build_app()
            self._create_app_pack()

    @property
    def project_name(self):
        parts = self.Parameters.root_path.split('/')
        return parts[-1]

    def on_finish(self, prev_status, status):
        super(MarketFrontBuildKadavrArc, self).on_finish(prev_status, status)
        self.arc_unmount()

    def on_wait(self, prev_status, status):
        super(MarketFrontBuildKadavrArc, self).on_wait(prev_status, status)
        self.arc_unmount()

    def _build_app(self):
        rich_check_call(
            ["npm", "run", "ci:build"],
            task=self, alias="npm_ci:build", cwd=self.APP_SRC_PATH
        )

        self._prune_deps()

    def _prune_deps(self):
        if self.Parameters.prune:
            rich_check_call(
                ["npm", "prune", "--production"],
                task=self, alias="npm_prune", cwd=self.APP_SRC_PATH
            )

    def _create_app_pack(self):
        pack_app_archive_path = tempfile.mktemp(suffix=".tar.gz", prefix=self.project_name)
        dir_path = os.path.join(self.arcadia, self.Parameters.root_path, '..')

        rich_check_call(
            ["tar", "-C", dir_path, "-czf", pack_app_archive_path, self.project_name],
            task=self, alias="create_app_pack"
        )

        self._app_create_resource(pack_app_archive_path)

    def _app_create_resource(self, pack_app_archive_path):
        app_res = sdk2.ResourceData(MARKET_KADAVR_APP(
            self, "Kadavr tarball", "{}.tar.gz".format(self.project_name)
        ))

        app_res.path.write_bytes(spath.Path(pack_app_archive_path).read_bytes())
        app_res.ready()

    def _prepare_env(self):
        # NB: last export overrides previous
        env.export(self.Parameters.environ)
        return

