# coding=utf-8

import logging
from subprocess32 import CalledProcessError

from sandbox import common
import sandbox.common.types.misc as ctm
from sandbox import sdk2
import time

from sandbox.projects.market.frontarc.helpers.MetatronEnvArc import MetatronEnvArc
from sandbox.sdk2.vcs.git import Git
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.projects.market.frontarc.helpers.sandbox_helpers import rich_check_call, get_resource_http_proxy_link
from sandbox.projects.market.frontarc.helpers.ubuntu import create_ubuntu_selector, setup_container
from sandbox.projects.market.frontarc.helpers.node import create_node_selector
from sandbox.projects.market.frontarc.helpers.ci import propagate_github_event_data
from sandbox.projects.sandbox.sandbox_lxc_image import RichTextTaskFailure
from sandbox.projects.sandbox_ci.utils.github import GitHubApi
from sandbox.projects.market.frontarc.MarketFrontCIArc import GitHubStatusArc


class MarketFrontCiAntidropArc(sdk2.Task):
    """
    Унифицированная таска для проверки, что в github-пуллреквесте не упали метрики

    Для добавления проверки в репозиторий, нужно добавить в конфиг sandbox-ci
    в секцию github_event_handlers/pull_request/sandbox_tasks подобный хэндлер:

    {
        "type": "MARKET_FRONT_CI_ANTIDROP",
        "custom_fields": [
          { "name": "github_context", "value": "flow coverage" },
          { "name": "check_command", "value": "bash scripts/flow_coverage.bash" },
          { "name": "higher_is_better", "value": true }
        ]
    }
    """
    TARGET_BRANCH_SRC_DIR = "target_branch_src"
    MERGING_BRANCH_SRC_DIR = "merging_branch_src"

    GITHUB_TOKEN_VAULT_KEY = "robot-metatron-github-token"

    github_status = None
    github_description = "Проверка свалилась, смотри логи ->"

    _github_api_token = None

    # Статус мержа из хука. Если None — значит, на момент отправки хука ещё неизвестен.
    pr_is_mergeable = None

    # Используется, чтобы взорвать таску, если упали без отчёта
    _exception = None

    class Parameters(sdk2.Task.Parameters):
        # значения проставляются в on_save
        github_owner = sdk2.parameters.String(
            "Github owner",
            default_value="market"
        )

        github_repo = sdk2.parameters.String(
            "Github repo"
        )

        commit_hash = sdk2.parameters.String(
            "Хэш коммита, которому проставляется статус"
        )

        pr_number = sdk2.parameters.Integer(
            "Номер пулл-реквеста"
        )

        github_context = sdk2.parameters.String(
            "Контекст в github (flow coverage, eslint warnings, etc.)"
        )

        head_branch = sdk2.parameters.String(
            "Тестируемая ветка"
        )

        check_command = sdk2.parameters.String(
            "Команда для запуска"
        )

        higher_is_better = sdk2.parameters.Bool(
            "Больше — лучше",
            default=False,
            description="Если True, то значения из PR не должны падать, если False — расти"
        )

        ubuntu_version = create_ubuntu_selector()
        node_version = create_node_selector()

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64

    def on_save(self):
        super(MarketFrontCiAntidropArc, self).on_save()
        propagate_github_event_data(self)

    def on_enqueue(self):
        super(MarketFrontCiAntidropArc, self).on_enqueue()
        setup_container(self)

        # В on_enqueue нет доступка к секретам, поэтому будем делать запросы через специальный сервер, который их
        # добавит.
        # @see: SANDBOX-3882, FEI-10158
        return GitHubApi(host='github-token-proxy.si.yandex-team.ru').create_status(
            owner=self.Parameters.github_owner,
            repo=self.Parameters.github_repo,
            context=self._github_context,
            sha=self.Parameters.commit_hash,
            state=GitHubStatusArc.PENDING,
            url=common.utils.get_task_link(self.id),
            description="Проверка в очереди"
        )

    def on_execute(self):
        super(MarketFrontCiAntidropArc, self).on_execute()

        with MetatronEnvArc(self, nodejs_version=self.Parameters.node_version):
            try:
                if self.pr_is_mergeable is None:
                    for i in range(5, 120, 5):
                        logging.debug(
                            "Github doesn't know, if PR is mergeable yet. Sleeping {} seconds and retrying".format(i)
                        )
                        time.sleep(i)
                        mergeable = self._is_pr_mergeable()
                        if mergeable is not None:
                            self.pr_is_mergeable = mergeable
                            break

                if not self.pr_is_mergeable:
                    self.github_description = "PR с конфликтом. Проверка не выполнялась"
                    self.github_status = GitHubStatusArc.FAILURE
                    self.set_info(self.github_description)
                    raise ValueError("Can't run check on a confilcting branch")

                self._checkout_target_branch()
                self._checkout_merging_branch()
                target_branch_results, merging_branch_results = self._run_checks()

                self._compare_results(target_branch_results, merging_branch_results)
            except (CalledProcessError, RichTextTaskFailure) as e:
                self.github_status = GitHubStatusArc.FAILURE
                self._exception = e
            except Exception as e:
                self.github_status = GitHubStatusArc.ERROR
                self._exception = e

            finally:
                github_api_token = sdk2.Vault.data(self.GITHUB_TOKEN_VAULT_KEY)
                GitHubApi(token=github_api_token).create_status(
                    owner=self.Parameters.github_owner,
                    repo=self.Parameters.github_repo,
                    context=self._github_context,
                    sha=self.Parameters.commit_hash,
                    state=self.github_status,
                    url=common.utils.get_task_link(self.id),
                    description=self.github_description
                )

                if self._exception:
                    raise self._exception

    def on_break(self, prev_status, status):
        super(MarketFrontCiAntidropArc, self).on_break(prev_status, status)

        github_api_token = sdk2.Vault.data(self.GITHUB_TOKEN_VAULT_KEY)
        GitHubApi(token=github_api_token).create_status(
            owner=self.Parameters.github_owner,
            repo=self.Parameters.github_repo,
            context=self._github_context,
            sha=self.Parameters.commit_hash,
            state=GitHubStatusArc.ERROR,
            url=common.utils.get_task_link(self.id),
            description=self.github_description
        )

    def _checkout_target_branch(self):
        target_branch = GitHubApi(token=self._github_api_token).get_pr_info(
            str(self.Parameters.github_owner),
            str(self.Parameters.github_repo),
            int(self.Parameters.pr_number)
        )["base"]["ref"]

        git = Git(self._git_url)
        git.clone(self.TARGET_BRANCH_SRC_DIR, target_branch)

    def _checkout_merging_branch(self):
        git = Git(self._git_url)
        git.clone(self.MERGING_BRANCH_SRC_DIR, self.Parameters.head_branch)

        rich_check_call(
            ["git", "fetch", "origin", "pull/{}/merge:MERGING_BRANCH".format(self.Parameters.pr_number)],
            self, "git_fetch_merging_branch", cwd=self.MERGING_BRANCH_SRC_DIR
        )

        rich_check_call(
            ["git", "checkout", "MERGING_BRANCH"],
            self, "git_checkout_merging_branch", cwd=self.MERGING_BRANCH_SRC_DIR
        )

    def _run_checks(self):
        with sdk2.helpers.ProcessLog(self, "target_branch_check") as process_log_target:
            with sdk2.helpers.ProcessLog(self, "merging_branch_check") as process_log_merging:
                target_branch_check = sp.Popen(
                    ["bash", "-c", self.Parameters.check_command],
                    stdout=sp.PIPE, stderr=process_log_target.stderr, cwd=self.TARGET_BRANCH_SRC_DIR
                )

                merging_branch_check = sp.Popen(
                    ["bash", "-c", self.Parameters.check_command],
                    stdout=sp.PIPE, stderr=process_log_merging.stderr, cwd=self.MERGING_BRANCH_SRC_DIR
                )

                while target_branch_check.poll() is None or merging_branch_check.poll() is None:
                    time.sleep(0.5)

                logging.debug("checks finished. return codes are: {}, {}".format(
                    target_branch_check.returncode,
                    merging_branch_check.returncode,
                ))

                target_branch_results = self._get_branch_results("target_branch", target_branch_check)
                merging_branch_results = self._get_branch_results("merging_branch", merging_branch_check)

                return target_branch_results, merging_branch_results

    def _get_branch_results(self, alias, popen_object):
        logging.debug("_get_branch_results: {}".format(alias))
        log_path = "{}.err.log".format(alias)

        if popen_object.returncode != 0:
            raise RichTextTaskFailure(
                self,
                "{} check failed, see the log below. <b><a href='{}'>{}</a></b>".format(
                    alias,
                    "/".join((get_resource_http_proxy_link(self.log_resource), log_path)),
                    log_path),
            )

        results = popen_object.stdout.read()
        logging.info("{} results: {}".format(alias, results))

        return results

    def _compare_results(self, target_branch_results, merging_branch_results):
        self.github_status = GitHubStatusArc.SUCCESS
        self.github_description = "Проверка прошла успешно!"

        target_branch_results = map(str.strip, target_branch_results.split("\n"))
        merging_branch_results = map(str.strip, merging_branch_results.split("\n"))

        for i in range(len(target_branch_results)):
            target_branch_result = float(target_branch_results[i])
            merging_branch_result = float(merging_branch_results[i])

            if merging_branch_result == target_branch_result:
                continue
            if merging_branch_result > target_branch_result and self.Parameters.higher_is_better:
                continue
            else:
                if self.Parameters.higher_is_better:
                    self.github_description = "Падение с '{}' до '{}'!"\
                        .format(target_branch_result, merging_branch_result)
                else:
                    self.github_description = "Рост с '{}' до '{}'!"\
                        .format(target_branch_result, merging_branch_result)

                self.github_status = GitHubStatusArc.FAILURE
                break

    @property
    def _github_context(self):
        return "[Sandbox CI] {}".format(self.Parameters.github_context)

    @property
    def _git_url(self):
        return "https://github.yandex-team.ru/{}/{}.git".format(
            self.Parameters.github_owner,
            self.Parameters.github_repo
        )

    @staticmethod
    def _git_clone(url, ref, target_dir='.'):
        git = Git(url)
        git.clone(str(target_dir), ref)

    def _is_pr_mergeable(self):
        pr_info = GitHubApi(token=self._github_api_token).get_pr_info(
            str(self.Parameters.github_owner),
            str(self.Parameters.github_repo),
            int(self.Parameters.pr_number)
        )

        return pr_info["mergeable"]
