# coding=utf-8

import logging
import os.path
import os
import lxml.etree as etree
from subprocess32 import CalledProcessError

from sandbox import common
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
from sandbox.common.types import client as ctc
from sandbox import sdk2
import time
from sandbox.projects.market.frontarc.helpers.MetatronEnvArc import MetatronEnvArc
from sandbox.projects.market.frontarc.helpers.sandbox_helpers import rich_check_call, get_resource_http_proxy_link
from sandbox.projects.market.front.resources import MarketCIReport
from sandbox.projects.market.frontarc.helpers.ubuntu import create_ubuntu_selector, setup_container
from sandbox.projects.market.frontarc.helpers.node import create_node_selector
from sandbox.projects.market.frontarc.helpers.ci import propagate_github_event_data, extract_ticket_id
from sandbox.projects.sandbox.sandbox_lxc_image import RichTextTaskFailure
from sandbox.projects.market.frontarc.helpers.github import change_status, get_pr_info, clone_repo, GitHubStatusArc
from sandbox.sandboxsdk.environments import PipEnvironment


class MarketFrontCIArc(sdk2.Task):
    """
    Унифицированная таска для маркетных github-проверок

    Поддерживает html- и junit-репорты (junit конвертит в html).
    На данный момент, ради унификации, пути до репортов захардкожены:
        html_reports/index.js и junit_reports/junitresults.xml


    Для добавления проверки в репозиторий, нужно добавить в конфиг sandbox-ci
    в секцию github_event_handlers/pull_request/sandbox_tasks подобный хэндлер:
    Для добавления проверки в репозиторий, нужно добавить JSON-файл с подобным содержимым:
    {
        "type": "MARKET_FRONT_CI",
        "custom_fields": [
          { "name": "github_context", "value": "unit tests" },
          { "name": "check_command", "value": "make configure && make bootstrap && bash scripts/test.bash -r" }
        ]
    }

    Помните, что нужен вебхук https://sandbox-ci.si.yandex-team.ru/v1/hooks/github на событие Pull request.
    И Content type у него должен быть application/json
    """

    APP_SRC_DIR = "app_src"
    GITHUB_TOKEN_VAULT_KEY = "robot-metatron-github-token"

    HTML_REPORT_DIR = "html_reports"
    HTML_REPORT_FILE = "index.html"

    JUNIT_REPORT_DIR = "junit_reports"
    JUNIT_REPORT_FILE = "junitresults.xml"

    TXT_REPORT_DIR = "txt_reports"
    TXT_REPORT_FILE = "report.txt"

    github_status = None
    github_description = "Свалились без отчёта, смотри логи →"

    # Статус мержа из хука. Если None — значит, на момент отправки хука ещё неизвестен.
    pr_is_mergeable = None

    # Используется, чтобы взорвать таску, если упали без отчёта
    _exception = None

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 1200

        # значения проставляются в on_save
        github_owner = sdk2.parameters.String(
            "Github owner",
            default_value="market"
        )

        github_repo = sdk2.parameters.String(
            "Github repo"
        )

        commit_hash = sdk2.parameters.String(
            "Хэш коммита, которому проставляется статус"
        )

        pr_title = sdk2.parameters.String(
            "Заголовок пулл-реквеста"
        )

        pr_number = sdk2.parameters.Integer(
            "Номер пулл-реквеста"
        )

        github_context = sdk2.parameters.String(
            "Контекст в github (codestyle, unit tests, etc.)"
        )

        head_branch = sdk2.parameters.String(
            "Тестируемая ветка"
        )

        check_command = sdk2.parameters.String(
            "Команда для запуска"
        )

        app_root_path = sdk2.parameters.String(
            "Кастомный путь корня приложения внутри репозитория"
        )

        send_report_to_st = sdk2.parameters.Bool(
            "Отправить отчёт в трекер"
        )

        check_env = sdk2.parameters.Dict("Переменные окружения для запуска проверки")

        ubuntu_version = create_ubuntu_selector()
        node_version = create_node_selector()

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64
        ram = 32 * 1024
        cores = 16
        disk_space = 16 * 1024
        client_tags = (ctc.Tag.MULTISLOT | ctc.Tag.GENERIC)
        environments = [
            PipEnvironment('startrek_client', version="2.3.0", custom_parameters=["--upgrade-strategy only-if-needed"])
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    def _prepare_env(self):
        os.environ['CURRENT_GIT_BRANCH'] = self.Parameters.head_branch

    def on_save(self):
        super(MarketFrontCIArc, self).on_save()
        propagate_github_event_data(self)

    def on_enqueue(self):
        super(MarketFrontCIArc, self).on_enqueue()
        setup_container(self)

        self.Requirements.semaphores = ctt.Semaphores(
            acquires=[
                ctt.Semaphores.Acquire(
                    name="market_front_ci_semaphore",
                    capacity=1000,
                    weight=1
                )
            ],
            release=(
                ctt.Status.Group.BREAK, ctt.Status.Group.FINISH
            )
        )

        # В on_enqueue нет доступка к секретам, поэтому будем делать запросы через специальный сервер, который их
        # добавит.
        # @see: SANDBOX-3882, FEI-10158
        return change_status(
            owner=self.Parameters.github_owner,
            repo=self.Parameters.github_repo,
            context=self._github_context,
            sha=self.Parameters.commit_hash,
            state=GitHubStatusArc.PENDING,
            url=common.utils.get_task_link(self.id),
            description="Проверка в очереди"
        )

    @property
    def html_report_path(self):
        return os.path.join(self.APP_SRC_DIR, self.HTML_REPORT_DIR, self.HTML_REPORT_FILE)

    @property
    def junit_report_path(self):
        return os.path.join(self.APP_SRC_DIR, self.JUNIT_REPORT_DIR, self.JUNIT_REPORT_FILE)

    @property
    def txt_report_path(self):
        return os.path.join(self.APP_SRC_DIR, self.TXT_REPORT_DIR, self.TXT_REPORT_FILE)

    @property
    def check_env(self):
        env = os.environ.copy()
        if self.Parameters.check_env:
            env.update(self.Parameters.check_env)
        return env

    def on_execute(self):
        super(MarketFrontCIArc, self).on_execute()

        with MetatronEnvArc(self, nodejs_version=self.Parameters.node_version):
            try:
                if self.pr_is_mergeable is None:
                    for i in range(5, 120, 5):
                        logging.debug(
                            "Github doesn't know, if PR is mergeable yet. Sleeping {} seconds and retrying".format(i)
                        )
                        time.sleep(i)
                        mergeable = self._is_pr_mergeable()
                        if mergeable is not None:
                            self.pr_is_mergeable = mergeable
                            break

                if not self.pr_is_mergeable:
                    self.github_description = "PR с конфликтом. Проверка не выполнялась"
                    self.github_status = GitHubStatusArc.FAILURE
                    self.set_info(self.github_description)
                    raise ValueError("Can't run check on a confilcting branch")

                self._prepare_env()
                self._run_check()
                self.github_status = GitHubStatusArc.SUCCESS
                self.github_description = "Проверка прошла успешно!"
            except (CalledProcessError, RichTextTaskFailure) as e:
                self.github_status = GitHubStatusArc.FAILURE
                self._exception = e
            except Exception as e:
                self.github_status = GitHubStatusArc.ERROR
                self._exception = e

            finally:
                if not os.path.exists(self.html_report_path) and os.path.exists(self.junit_report_path):
                    logging.debug("{} not found, but {} is. Converting..."
                                  .format(self.html_report_path, self.junit_report_path))
                    self._convert_junit_to_html(self.junit_report_path, self.html_report_path)

                mean_report_path = None
                if os.path.exists(self.html_report_path):
                    logging.debug("Found HTML report: {}".format(self.html_report_path))
                    mean_report_path = self.html_report_path
                elif os.path.exists(self.txt_report_path):
                    logging.debug("Found TXT report: {}".format(self.txt_report_path))
                    mean_report_path = self.txt_report_path

                if mean_report_path is not None:
                    if self.github_status == GitHubStatusArc.FAILURE:
                        logging.debug(type(self._exception))
                        self.github_description = "Неудача! Смотри отчёт →"
                        logging.debug("Произошла ошибка при запуске процесса: ")
                        if hasattr(self._exception, 'message'):
                            logging.debug(self._exception.message)
                        if hasattr(self._exception, 'get_task_info'):
                            logging.debug(self._exception.get_task_info())

                    res = self._create_html_report_resource(os.path.dirname(mean_report_path))
                    http_report_url = '{}/{}'.format(get_resource_http_proxy_link(res),
                                                     os.path.basename(mean_report_path))
                    self.set_info(
                        "Отчёт: <a href=\"{url}\">{url}</a>".format(url=http_report_url),
                        do_escape=False
                    )
                    github_check_link = http_report_url
                else:
                    logging.debug("CI check report not found")
                    github_check_link = common.utils.get_task_link(self.id)
                    # https://st.yandex-team.ru/MARKETFRONTECH-288
                    self.github_status = GitHubStatusArc.FAILURE
                    self.github_description = "Неудача! Отчёт не найден"

                self._change_status(
                    owner=self.Parameters.github_owner,
                    repo=self.Parameters.github_repo,
                    context=self._github_context,
                    sha=self.Parameters.commit_hash,
                    state=self.github_status,
                    url=github_check_link,
                    description=self.github_description
                )

                if mean_report_path is None:
                    if self._exception:
                        raise self._exception
                    else:
                        raise ValueError("CI check report not found")

    def on_break(self, prev_status, status):
        super(MarketFrontCIArc, self).on_break(prev_status, status)

        self._change_status(
            owner=self.Parameters.github_owner,
            repo=self.Parameters.github_repo,
            context=self._github_context,
            sha=self.Parameters.commit_hash,
            state=GitHubStatusArc.ERROR,
            url=common.utils.get_task_link(self.id),
            description=self.github_description
        )

    @property
    def _github_context(self):
        return "[Sandbox CI] {}".format(self.Parameters.github_context)

    @staticmethod
    def _convert_junit_to_html(junit_path, html_path):
        with open(os.path.join(os.path.dirname(__file__), "junit-to-html.xsl")) as xslt_file:
            junit_to_html_xslt = etree.parse(xslt_file)

            with open(junit_path) as junit_file:
                junit_xml = etree.parse(junit_file)
                transform = etree.XSLT(junit_to_html_xslt)
                html = etree.tostring(transform(junit_xml), encoding="utf-8")

                html_dir = os.path.dirname(html_path)
                if not os.path.exists(html_dir):
                    os.makedirs(html_dir)

                with open(html_path, "w") as html_file:
                    logging.debug("Writing HTML report to {}...".format(html_path))
                    html_file.write(html)

    def _run_check(self):
        clone_repo(
            self.Parameters.github_owner,
            self.Parameters.github_repo,
            self.Parameters.head_branch,
            self.APP_SRC_DIR
        )

        rich_check_call(
            ["git", "fetch", "origin", "pull/{}/merge:MERGING_BRANCH".format(self.Parameters.pr_number)],
            self, "git_fetch_merge", cwd=self.APP_SRC_DIR
        )

        rich_check_call(
            ["git", "checkout", "MERGING_BRANCH"],
            self, "git_checkout", cwd=self.APP_SRC_DIR
        )

        # клонируем мы в обычный APP_SRC
        # но перед запуском проверки подменяем директорию, если того требует структура проекта
        if self.Parameters.app_root_path is not None:
            self.APP_SRC_DIR = os.path.join(self.APP_SRC_DIR, self.Parameters.app_root_path)

        rich_check_call(
            ["bash", "-c", self.Parameters.check_command],
            self, "ci_check_command", cwd=self.APP_SRC_DIR, env=self.check_env
        )

    def _create_html_report_resource(self, report_path):
        report_res = MarketCIReport(self, "CI Report", report_path)

        report_res_data = sdk2.ResourceData(report_res)
        report_res_data.ready()

        return report_res

    def _is_pr_mergeable(self):
        pr_info = get_pr_info(
            str(self.Parameters.github_owner),
            str(self.Parameters.github_repo),
            int(self.Parameters.pr_number)
        )

        return pr_info["mergeable"]


    COMMENT_FORMAT = """
Статистика прогона CI ({context}):
#|
||Статус|Отчёт|Таск||
||{status}|{report_url}|{task_url}||
|#
<!-- <# consolidated-report-manager-short-report-regex #>||{report_url}|{task_url}|| -->
<# <!-- CI_REPORTS_ENTRY({context}) --> #>
<!--<# <div id="consolidated-reports-comment" /> #>-->
"""

    def _send_startrek_report(self, state, report_url = ''):
        if not self.Parameters.send_report_to_st:
            logging.info('Will not send report to Startrek')
            return

        logging.info("Sending report to Startrek")

        oauth_token = sdk2.Vault.data('robot-metatron-st-token')

        if not oauth_token:
            logging.error("Cannot get robot-metatron oauth_token")
            return

        issue_number = self._get_startrek_issue()
        if not issue_number:
            logging.error("Startrek issue number not found in PR title, will not report")
            return

        from startrek_client import Startrek
        st = Startrek(useragent='robot-metatron', token=oauth_token)
        issue = st.issues[issue_number]

        if not issue:
            logging.error("Cannot find issue {}".format(issue_number))
            return


        fmt_status = ''
        if state == GitHubStatusArc.SUCCESS:
            fmt_status = '!!(green)PASSED!!'
        elif state == GitHubStatusArc.PENDING:
            fmt_status = '!!(orange)PENDING!!'
        else: fmt_status = '!!(red)FAILED!!'

        fmt_report_url = "(({} Отчёт))".format(report_url) if report_url else ''
        fmt_task_url = "((https://sandbox.yandex-team.ru/task/{}/view Таск))".format(self.id)

        comment = self.COMMENT_FORMAT.format(context=self.Parameters.github_context, status=fmt_status, report_url=fmt_report_url, task_url=fmt_task_url)
        issue.comments.create(text=comment, params={'notify': False})

    def _get_startrek_issue(self):
        return extract_ticket_id(self.Parameters.pr_title)

    def _get_reports_entry_marker(self, ci_check_name):
        return self.CI_REPORTS_ENTRY_MARKER.format(ci_check_name)

    def _get_new_text(self, comment_text, comment_marker, old_text):
            old_text = old_text or ""
            parts = old_text.strip().split(comment_marker)
            if len(parts) > 1:
                parts[1] = comment_text
            else:
                parts.extend([comment_text, "\n"])
            return comment_marker.join(parts)

    def _change_status(self, owner, repo, context, sha, state, url, description):
        result = change_status(owner=owner, repo=repo, context=context, sha=sha, state=state, url=url, description=description)
        self._send_startrek_report(state, url)
        return result
