# coding=utf-8

from os.path import join
import json
import logging

from sandbox import common
from sandbox import sdk2

from sandbox.projects.sandbox_ci.utils.github import GitHubApi
from sandbox.projects.sandbox.sandbox_lxc_image import RichTextTaskFailure
from sandbox.projects.market.frontarc.helpers.ubuntu import create_ubuntu_selector, setup_container
from sandbox.projects.market.frontarc.helpers.github import \
    GitHubStatusArc, GitHubStatusDescriptionArc, clone_repo_for_merge_commit

from sandbox.projects.market.frontarc.MarketFrontCIPackageJsonChecksArc.checks import \
    CHECKS, ValidationError, CheckExecutionError


class MarketFrontCiPackageJsonChecksArc(sdk2.Task):
    """
        Таска для проверки содержимого в package.json

        Для добавления проверки в репозиторий, нужно добавить в конфиг sandbox-ci
        в секцию github_event_handlers/pull_request/sandbox_tasks подобный хэндлер:

        {
            "type": "MARKET_FRONT_CI_PACKAGE_JSON_CHECKS",
            "custom_fields": [
                {
                    "name": "Check list",
                    "value": {
                        "check_duplicates_in_dependencies: {},
                        "restrict_yandex_team_github": {},
                        "exact_versions_are_specified": {},
                    }
                }
            ]
        }
    """

    CLONED_REPO_SRC_DIR = "cloned_src"

    GITHUB_TOKEN_VAULT_KEY = "robot-metatron-github-token"
    GITHUB_HOST = "github-token-proxy.si.yandex-team.ru"
    GITHUB_CONTEXT = "[Sandbox CI] package.json checks"

    class Requirements(sdk2.Requirements):
        cores = 1
        disk_space = 1000
        ram = 1000

    class Parameters(sdk2.Parameters):
        default_checks = """{
            "check_yandex_team_github_ref": {},
            "check_duplicates_in_dependencies": {},
            "check_exact_versions": {}
        }"""
        default_owner = "market"
        default_repo = "marketfront"
        default_path_to_package_json = "package.json"

        ubuntu_version = create_ubuntu_selector()

        github_owner = sdk2.parameters.String("Github owner", default=default_owner)
        github_repo = sdk2.parameters.String("Github repo", default=default_repo)
        commit_hash = sdk2.parameters.String("Commit hash")
        path_to_package_json = sdk2.parameters.String("Path to package.json", default=default_path_to_package_json)
        checks_list = sdk2.parameters.JSON("Checks list", default=default_checks)

    def on_enqueue(self):
        super(MarketFrontCiPackageJsonChecksArc, self).on_enqueue()
        setup_container(self)

        GitHubApi(host=self.GITHUB_HOST).create_status(
            owner=self.Parameters.github_owner,
            repo=self.Parameters.github_repo,
            context=self.GITHUB_CONTEXT,
            sha=self.Parameters.commit_hash,
            state=GitHubStatusArc.PENDING,
            url=common.utils.get_task_link(self.id),
            description=GitHubStatusDescriptionArc.pending
        )

    def on_execute(self):
        super(MarketFrontCiPackageJsonChecksArc, self).on_execute()

        exception = None

        try:
            self._clone_repo()

            package_json = self._read_package_json()

            self._run_checks(package_json)

            github_status = GitHubStatusArc.SUCCESS
            github_description = GitHubStatusDescriptionArc.success

        except RichTextTaskFailure as e:
            exception = e

            github_status = GitHubStatusArc.ERROR
            github_description = GitHubStatusDescriptionArc.error

        except Exception as e:
            exception = e

            github_status = GitHubStatusArc.FAILURE
            github_description = GitHubStatusDescriptionArc.failure

        finally:
            github_api_token = sdk2.Vault.data(self.GITHUB_TOKEN_VAULT_KEY)
            GitHubApi(token=github_api_token).create_status(
                owner=self.Parameters.github_owner,
                repo=self.Parameters.github_repo,
                context=self.GITHUB_CONTEXT,
                sha=self.Parameters.commit_hash,
                state=github_status,
                url=common.utils.get_task_link(self.id),
                description=github_description
            )

            if exception is not None:
                raise exception

    def _clone_repo(self):
        clone_repo_for_merge_commit(
            owner=self.Parameters.github_owner,
            name=self.Parameters.github_repo,
            commit_sha=self.Parameters.commit_hash,
            target_dir=self.CLONED_REPO_SRC_DIR
        )

    def _read_package_json(self):
        path_to_package_json = join(self.CLONED_REPO_SRC_DIR, self.Parameters.path_to_package_json)
        with open(path_to_package_json, 'r') as fin:
            return json.loads(fin.read())

    def _run_checks(self, package_json):
        error_messages = []

        for (name, args) in self.Parameters.checks_list.items():
            try:
                logging.debug([name, args])
                CHECKS[name](package_json, **args)

            except CheckExecutionError as e:
                error_messages.append(e.message)

            except ValidationError as e:
                raise Exception(e.message)

            except Exception as e:
                raise e

        if error_messages:
            raise RichTextTaskFailure(self, '\n'.join(error_messages))

    @property
    def _git_url(self):
        # Формируем ссылку на репозиторий
        return "https://github.yandex-team.ru/{}/{}.git".format(
            self.Parameters.github_owner,
            self.Parameters.github_repo
        )
