# coding=utf-8
import logging
import re
import time

import sandbox.common.types.misc as ctm
from sandbox import sdk2
from sandbox.projects.common import task_env
from sandbox.projects.market.frontarc.helpers.MetatronEnvArc import ST_OAUTH_TOKEN_VAULT_KEY
from sandbox.sandboxsdk.environments import PipEnvironment


class MarketFrontCIPollStartrekFields(sdk2.Task):
    """
    Таска для ожидания значения в поле ST тикета.
    Раз в минуту поллит значение поля в тикете
    Завершается по таймауту с ошибкой или успешно когда все значения соответствуют ожиданиям
    """

    class Requirements(task_env.TinyRequirements):
        dns = ctm.DnsType.DNS64
        disk_space = 700  # 700 Mb, потому что пакеты и всякая ерунда
        ram = 64  # 64 Mb
        environments = [
            PipEnvironment("yandex_tracker_client", version="1.3", custom_parameters=["--upgrade-strategy only-if-needed"]),
            PipEnvironment("startrek_client", version="2.3.0", custom_parameters=["--upgrade-strategy only-if-needed"])
        ]

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 4 * 24 * 60 * 60  # 4 days

        ticket_id = sdk2.parameters.String(
            "Номер тикета",
            description="в формате QUEUE-12345"
        )

        expected_values = sdk2.parameters.Dict(
            "Список полей ST и ожидаемых значений для них",
            value_type=sdk2.parameters.String
        )

        wiki_link = sdk2.parameters.String(
            "Ссылка на объяснение чего и зачем ждёт джоба",
            default="https://wiki.yandex-team.ru/users/lengl/zapusk-pokommitnyx-pajjplajjnov-vo-fronte-marketa"
        )

    def on_execute(self):
        super(MarketFrontCIPollStartrekFields, self).on_execute()

        assert re.match("^[A-Z]+-\\d+$", str(self.Parameters.ticket_id)) is not None, "Тикет нужен в формате `QUEUE-ID`"

        from startrek_client import Startrek
        oauth_token = sdk2.Vault.data(ST_OAUTH_TOKEN_VAULT_KEY)
        st = Startrek(useragent="robot-metatron", token=oauth_token)
        issue = st.issues[str(self.Parameters.ticket_id)]

        self.set_info("Ждём пока в полях тикета будут ожидаемые значения. <a href=\"{}\">Что это значит?</a>"
                      .format(self.Parameters.wiki_link), do_escape=False)

        while not self.has_expected_field_values(issue):
            time.sleep(60)
            issue = st.issues[str(self.Parameters.ticket_id)]

    def has_expected_field_values(self, issue):
        logging.info("Провееряем значения в тикете %s", str(issue.key))
        result = True
        for field_name, expected_value in self.Parameters.expected_values.items():
            try:
                logging.debug("Поле %s, expected: %s, actual: %s", field_name, str(expected_value), str(issue[field_name]))
                result = result and str(issue[field_name]) == expected_value
            except KeyError as e:
                msg = "В тикете {} отсутствует поле {}".format(str(issue.key), field_name)
                self.set_info(msg)
                raise e
        return result
