# coding=utf-8

import json
import logging

import requests

import sandbox.common.types.misc as ctm
from sandbox import common
from sandbox import sdk2
from sandbox.projects.common import task_env
from sandbox.projects.market.frontarc.helpers.ArcDefaultParameters import ArcDefaultParameters
from sandbox.projects.market.frontarc.helpers.ArcanumApi import ArcanumApi, ReviewRequestFields
from sandbox.projects.market.frontarc.helpers.MetatronEnvArc import TSUM_TOKEN_VAULT_KEY, ARC_TOKEN_VAULT_KEY
from sandbox.projects.market.frontarc.helpers.node import create_node_selector
from sandbox.projects.market.frontarc.helpers.ubuntu import create_ubuntu_selector, setup_container
from sandbox.projects.sandbox_ci.utils.request import send_request
from sandbox.sdk2 import Task, Vault


class MarketFrontCiStartMtArc(Task):
    """
    Таска для запуска и перезапуска мультитестинга.
    """

    TSUM_MT_PAGE_URL = "https://tsum.yandex-team.ru/pipe/projects/{project}/multitestings/environments/{name}"
    TSUM_MT_API_URL = "https://tsum.yandex-team.ru/api/multitestings/project/{project}/environments/{name}/"
    RESTART_METHOD = 'createOrRestart'
    KILL_METHOD = 'cleanupAndArchiveForce'

    class Parameters(Task.Parameters):
        should_destroy = sdk2.parameters.Bool(
            "Вместо запуска джоба убьет пайп",
            default=False
        )

        test_environment = sdk2.parameters.Bool(
            "Тестовый паспорт",
            default=False
        )

        arc_params = ArcDefaultParameters()

        ticket_id = sdk2.parameters.String(
            "Номер тикета"
        )

        tsum_project = sdk2.parameters.String(
            "Название проекта"
        )

        tsum_pipeline_type = sdk2.parameters.String(
            "Тип пайплайна"
        )

        postfix = sdk2.parameters.String(
            "Постфикс в названии пайплайна"
        )

        ubuntu_version = create_ubuntu_selector()
        node_version = create_node_selector()

    class Requirements(task_env.TinyRequirements):
        dns = ctm.DnsType.DNS64

    class Context(Task.Context):
        ticket_id = None
        is_release = None

    def on_save(self):
        super(MarketFrontCiStartMtArc, self).on_save()

        self.Context.ticket_id = self.Parameters.ticket_id

    def on_enqueue(self):
        super(MarketFrontCiStartMtArc, self).on_enqueue()
        setup_container(self)

        if self.Context.ticket_id is None:
            self.set_info("Номер тикета не найден, игнорируем")
            return

        self.set_info("Пайплайн: <a href=\"{}\">{}</a>".format(self._tsum_page_url, self._tsum_page_url), do_escape=False)

    def on_execute(self):
        super(MarketFrontCiStartMtArc, self).on_execute()

        if self.Context.is_release:
            logging.debug("Found a release ticket")
            return

        if self.Context.ticket_id is None:
            logging.debug("The ticket number is not found")
            return

        if self.Parameters.should_destroy:
            self._kill_mt()
        else:
            self._create_or_restart_mt()

    def on_success(self, prev_status):
        super(MarketFrontCiStartMtArc, self).on_success(prev_status)

    def on_failure(self, prev_status):
        super(MarketFrontCiStartMtArc, self).on_failure(prev_status)

    def on_break(self, prev_status, status):
        super(MarketFrontCiStartMtArc, self).on_break(prev_status, status)

    @property
    def _task_url(self):
        return common.utils.get_task_link(self.id)

    @property
    def _tsum_mt_name(self):
        assert self.Context.ticket_id

        name = self.Context.ticket_id.lower()

        if self.Parameters.postfix:
            name += "-" + self.Parameters.postfix

        return name

    @property
    def _tsum_page_url(self):
        assert not self.Context.is_release
        assert self.Context.ticket_id

        return self.TSUM_MT_PAGE_URL.format(
            project=self.Parameters.tsum_project,
            name=self._tsum_mt_name
        )

    @property
    def _tsum_api_url(self):
        return self.TSUM_MT_API_URL.format(
            project=self.Parameters.tsum_project,
            name=self._tsum_mt_name
        )

    def _kill_mt(self):
        token = Vault.data(TSUM_TOKEN_VAULT_KEY)

        headers = {
            "Authorization": token,
            "Content-Type": "application/json"
        }

        res = send_request("post", self._tsum_api_url + MarketFrontCiStartMtArc.KILL_METHOD, headers=headers, data=json.dumps({}))

        res.raise_for_status()

        try:
            assert res.json()['name'] == self._tsum_mt_name
        except Exception:
            logging.error(u"Bad TSUM response: {}".format(res.text))

            raise Exception("Something is wrong, see logs")

    def _create_or_restart_mt(self):
        token = Vault.data(TSUM_TOKEN_VAULT_KEY)
        arc_api = ArcanumApi(Vault.data(ARC_TOKEN_VAULT_KEY))

        headers = {
            "Authorization": token,
            "Content-Type": "application/json"
        }

        data = {
            "name": self._tsum_mt_name,
            "type": "USE_EXISTING_PIPELINE",
            "pipelineId": self.Parameters.tsum_pipeline_type,
            "resources": {
                "fd4f152f-9f11-4b91-8a35-e58503b6bdf6": {
                    "tickets": [self.Context.ticket_id]
                }
            }
        }

        data["resources"]["486e1cc9-c7cf-43aa-b8c1-bf19eb8d861d"] = {
            "useMarketTest": str(self.Parameters.test_environment).lower(),
        }

        # Ресурс: ru.yandex.market.tsum.pipelines.common.resources.ReleaseBranch
        data["resources"]["3c19ca69-7e7e-489e-9f6b-faf14639f58c"] = {
            "branchName": self.Parameters.head_branch,
            "repoFullName": self.Parameters.root_path,
            "nodeVersion": str(self.Parameters.node_version)
        }

        diff_set_id = None
        task_description = self.Parameters.task_description

        if self.Parameters.pr_number:
            diff_set_id = self.Parameters.diff_set_id

            if not diff_set_id:
                pr_info = arc_api.get_review_request_info(self.Parameters.pr_number, [
                    ReviewRequestFields.active_diff_set,
                    ReviewRequestFields.summary,
                ])

                diff_set_id = pr_info[ReviewRequestFields.active_diff_set]['id']


        # Ресурс ru.yandex.market.tsum.pipelines.common.resources.ArcDefaultParams
        data["resources"]["34c16cdd-4260-44ae-8c89-31f01209c498"] = {
            "rootPath": self.Parameters.root_path,
            "diffSetId": diff_set_id,
            "commitHash": self.Parameters.commit_hash,
            "prNumber": self.Parameters.pr_number,
            "headBranch": self.Parameters.head_branch,
            "taskDescription": task_description,
        }

        # Ресурс ru.yandex.market.tsum.pipelines.common.jobs.datasource.DataSourceProperty
        if self.Parameters.tsum_pipeline_type in ["mt-partner-front-arc", "mt-partner-front-prestable-arc"]:
            data["resources"]["64b0e129-af5a-46b1-bd90-61b362b6de89"] = {
                "type": "JAVA",
                "key": "cocon_commit_id",
                "value": None
            }

        logging.debug(data)

        res = send_request("post", self._tsum_api_url + MarketFrontCiStartMtArc.RESTART_METHOD, headers=headers, data=json.dumps(data))

        try:
            res.raise_for_status()
        except requests.exceptions.HTTPError as e:
            e.args += ("Tsum response text: {}".format(res.text),)
            raise e

        try:
            assert res.json()['name'] == self._tsum_mt_name
        except Exception:
            logging.error(u"Bad TSUM response: {}".format(res.text))

            raise Exception("Something is wrong, see logs")

    @sdk2.report(title="TSUM Link")
    def build_tsum_link(self):
        return '<a href="{tsum_url}">{tsum_url}</a>'.format(tsum_url=self._tsum_page_url)
