# coding=utf-8

import json
import logging

from sandbox import sdk2
from sandbox.sdk2 import Task, Vault
import sandbox.common.types.misc as ctm
from sandbox.projects.sandbox_ci.utils.request import send_request


SLACK_WORKFLOW_ALERTS_TOKEN_VAULT_KEY = "marketfrontech-content-alerts-workflow"
SLACK_WORKFLOW_LOG_TOKEN_VAULT_KEY = "marketfrontech-content-log-workflow"
SLACK_DUTY_USERNAME = "@frontech_speed_duty"


def path_or(default_value, path, d):
    pointer = d
    try:
        for prop in path:
            pointer = pointer[prop]

        return pointer
    except (KeyError, TypeError), e:
        return default_value


def link(href, text):
    return u"<{href}|{text}>".format(href=href, text=text)


def send_slack_message(token, message):
    res = send_request(
        "post",
        "https://hooks.slack.com/workflows/{token}".format(token=token),
        headers={
            "Content-Type": "application/json"
        },
        data=json.dumps({
            "message": message,
        })
    )

    try:
        res.raise_for_status()
    except Exception:
        logging.error(u"Bad response: {}".format(res.text))

        raise Exception("Something is wrong, see logs")

    return res


def prettify_message(message):
    return "\n".join([line.strip() for line in message.split("\n")])


class MarketFrontCmsPublicationMonitoringArc(Task):
    """
    Таска для мониторинга публикации СMS

    Публикует в слак-канал #marketfrontech-content-alerts призывы дежурного скорости для "важных" публикаций
    Публикует все без призывов в слак-канал #marketfrontech-content-log
    """

    class Parameters(Task.Parameters):
        cms_publication_context_resource_id = sdk2.parameters.Integer(
            'Id ресурса с контекстом CMS публикации'
        )

    class Requirements(Task.Requirements):
        dns = ctm.DnsType.DNS64

    def _load_publication_context(self):
        id = self.Parameters.cms_publication_context_resource_id

        cms_publication_context_resource = sdk2.Resource["CMS_PUBLICATION_CONTEXT"].find(id=id).first()
        resource_data = sdk2.ResourceData(cms_publication_context_resource)
        resource_data_txt = resource_data.path.read_bytes()

        return json.loads(resource_data_txt)

    def on_execute(self):
        super(MarketFrontCmsPublicationMonitoringArc, self).on_execute()

        ctx = self._load_publication_context()

        token_alerts = Vault.data(SLACK_WORKFLOW_ALERTS_TOKEN_VAULT_KEY)
        token_log = Vault.data(SLACK_WORKFLOW_LOG_TOKEN_VAULT_KEY)

        desktop_is_monitoring = path_or("false", ['meta_params', 'desktop', 'isMonitoring'], ctx) == "true"
        touch_is_monitoring = path_or("false", ['meta_params', 'touch', 'isMonitoring'], ctx) == "true"
        should_call_on_duty = desktop_is_monitoring or touch_is_monitoring

        raw_message = u"""
        Опубликована новая версия страницы «{page_title}».
        Результаты проверок {pipeline_link}, {edit_page_link}, ссылка на опубликованную страницу: {preview_link}.
        """.format(
            page_title=ctx["page_title"],
            pipeline_link=link(ctx["pipeline_link"], "в пайплайне"),
            edit_page_link=link(ctx["cms_editor_link"], "страница в редакторе CMS"),
            preview_link=" и ".join([
                link(ctx["page_preview_link"][key], name) for name, key in [
                    ("десктоп", "desktop"), ("тач", "touch")
                ] if ctx["page_preview_link"][key] is not None
            ]) or "не удалось отобразить ссылки"
        )
        if should_call_on_duty:
            raw_message += "\n" + "Эта страница находится под SLA."

            alert_message = SLACK_DUTY_USERNAME + "\n" + raw_message
            send_slack_message(token_alerts, prettify_message(alert_message))

        send_slack_message(token_log, prettify_message(raw_message))
