# coding=utf-8

import logging
import json
import urllib
import re

from sandbox import sdk2
from sandbox.sdk2 import Task
import sandbox.common.types.misc as ctm
from sandbox.projects.sandbox_ci.utils.request import send_request
from sandbox.projects.market.front.resources import CmsPublicationContext

CMS_API_URL = "http://mbo-cms-api.vs.market.yandex.net/export/widgets-relations?draft=false&page_id={page_id}"
TEMPLATOR_URL = "http://templator.vs.market.yandex.net:29338/tarantino/getcontextpage?{params}"
CMS_EDITOR_URL = "https://cms.market.yandex.ru/editor/documents/{page_id}/edit"
PLATFORM_TO_CMS_DEVICE = {
    "desktop": "desktop",
    "touch": "phone",
}


def fetch(url):
    res = send_request("get", url)

    try:
        res.raise_for_status()
    except Exception:
        logging.error(u"Bad fetch response: {}".format(res.text))
        logging.error(u"Request url: {}".format(url))

        raise Exception("Something is wrong, see logs")

    return res


def path_or(default_value, path, d):
    pointer = d
    try:
        for prop in path:
            pointer = pointer[prop]

        return pointer
    except (KeyError, TypeError, IndexError), e:
        return default_value


def find_or(default_value, generator):
    return next(generator, default_value)


def fetch_cms_exports_params(page_id):
    """Получает и форматирует параметры из api CMS для похода в темплатор за разметкой"""
    # "a=b#c=d:12345\ne=f#g=h:12345"
    export_data = fetch(CMS_API_URL.format(page_id=page_id)).text

    if len(export_data.strip()) == 0:
        return []

    # ["a=b#c=d", "e=f#g=h"]
    step_1 = [line.rstrip().split(":")[0] for line in export_data.split("\n")]

    # [["a=b", "c=d"], ["e=f", "g=h"]]
    step_2 = [export.split("#") for export in step_1]

    # [[("a", "b"), ("c", "d")], [("e", "f"), ("g", "h")]]]
    step_3 = [tuple(group.split("=") for group in export) for export in step_2]

    # [{"a": "b", "c": "d"}, {"e": "f", "g": "h"}]
    step_4 = [dict(export) for export in step_3]

    step_5 = filter(lambda params: "domain" in params and params["domain"] == "ru" and params["zoom"] == "full", step_4)

    return step_5


def fetch_json_schema(qp):
    return fetch(TEMPLATOR_URL.format(params=urllib.urlencode(qp))).json()


def get_preview_link(params):
    is_desktop = params["device"] == PLATFORM_TO_CMS_DEVICE["desktop"]
    is_touch = params["device"] == PLATFORM_TO_CMS_DEVICE["touch"]

    if params["type"] == "mp_promo" and is_desktop:
        return "https://pokupki.market.yandex.ru/special/{}".format(params["semantic_id"])
    if params["type"] == "mp_promo" and is_touch:
        return "https://m.pokupki.market.yandex.ru/special/{}".format(params["semantic_id"])
    if params["type"] == "mp_morda" and is_desktop:
        return "https://pokupki.market.yandex.ru"
    if params["type"] == "mp_morda" and is_touch:
        return "https://m.pokupki.market.yandex.ru/?pda-redir=1"
    if params["type"] == "morda_context" and is_desktop:
        return "https://market.yandex.ru"
    if params["type"] == "navnode_touch" and params["nid"] == "54415" and is_touch:
        return "https://m.market.yandex.ru/?pda-redir=1"
    else:
        return None


def split_by_platform(cb):
    return {
        "desktop": cb("desktop"),
        "touch": cb("touch"),
    }


def extract_pipeline_link_from_desc(description):
    try:
        return re.search('<a href="(.*)">', description).group(1)
    except IndexError:
        return description

class MarketFrontCmsPublicationPrepareContextArc(Task):
    """
    Таска для подготовки контекста для пайплайна публикации CMS страницы.
    """

    class Parameters(Task.Parameters):
        page_id = sdk2.parameters.String("Идентификатор страницы в CMS")

    class Requirements(Task.Requirements):
        dns = ctm.DnsType.DNS64

    def on_execute(self):
        super(MarketFrontCmsPublicationPrepareContextArc, self).on_execute()

        page_id = self.Parameters.page_id

        # Отправляемся в бэкэнд cms, чтобы узнать о доступных экспортах страницы
        exports_params = fetch_cms_exports_params(page_id)
        params = split_by_platform(lambda platform:
            find_or(None, (p for p in exports_params if p["device"] == PLATFORM_TO_CMS_DEVICE[platform]))
        )

        # Отправляемся в темплатор за json`ом (версткой)
        json_schema = split_by_platform(lambda platform:
            fetch_json_schema(params[platform]) if params[platform] is not None else None
        )

        page_title = path_or(
            None, ["desktop", "result", 0, "name"], json_schema
        ) or path_or(
            None, ["touch", "result", 0, "name"], json_schema
        )

        context = {
            "page_id": page_id,
            "page_title": page_title,

            "cms_editor_link": CMS_EDITOR_URL.format(page_id=page_id),
            "pipeline_link": extract_pipeline_link_from_desc(self.Parameters.description),

            "params": params,
            "json_schema": json_schema,

            "meta_params": split_by_platform(lambda platform:
                 path_or(None, [platform, "result", 0, "performanceControl"], json_schema)
            ),

            "page_preview_link": split_by_platform(lambda platform:
                get_preview_link(params[platform]) if params[platform] is not None else None
            ),
        }

        stored_context = sdk2.ResourceData(CmsPublicationContext(
            self, "CMS Publication context", "context.json"
        ))
        stored_context.path.write_bytes(json.dumps(context))
        stored_context.ready()


