# coding=utf-8

from sandbox.projects.market.frontarc.helpers.MetatronEnvArc import MetatronEnvArc
from sandbox.projects.market.frontarc.helpers.sandbox_helpers import rich_check_call
from sandbox.projects.market.frontarc.helpers.ubuntu import create_ubuntu_selector, setup_container
from sandbox.projects.market.frontarc.helpers.node import create_node_selector
from sandbox.sandboxsdk.environments import PipEnvironment
import sandbox.common.types.misc as ctm
from sandbox.projects.market.frontarc.helpers.arc_base import MarketFrontArcBase
from sandbox.projects.common import task_env

import datetime
import json
import logging
import os.path
from sandbox import sdk2


class MarketFrontCollectCodeStatsArc(sdk2.Task, MarketFrontArcBase):
    """
    Собирает метрики по коду проекта и отправляет их в Statface.
    В качестве параметров надо указать репо и ветку проекта из github.yandex-team.ru.
    Так же надо указать путь внутри исходников проекта до манифеста, описывающего
    собираемые метрики. Манифест должен быть json-файлом со следующей стрктурой:

    [
        {
            "statfaceReport": "Market_Partner/CodeHealth/Coverage",
            "collectScript": "make configure && make bootstrap && npm run report:coverage",
            "resultFile": "reports/coverage.json"
        },
    ]

    , где statfaceReport - путь до отчета в statface, collectScript - скрипт из репо проекта,
    собирающей метрики, resultFile - файл с результатами работы collectScript-а. resultFile должен
    быть json-файлом с мапой вида "поле statface-очета" -> "значение поля", или массивом таких мап.
    Так для отчета Market_Partner/CodeHealth/Coverage, resultFile выглядит селующим образом:

    {"lines":0.77,"functions":1.6,"branches":0.6,"statements":0.71}

    а для отчета Market_Partner/CodeHealth/CLOC, resultFile выглядит селующим образом:
    [
    {"repository":"URL","lang":"JavaScript","files":1000,"blank":2,"comment":3,"code":100500},
    {"repository":"URL","lang":"JSON","files":20,"blank":0,"comment":0,"code":100},
    ]

    """

    STATFACE_TOKEN_VAULT_KEY = "robot-metatron-statbox-token"

    class Parameters(sdk2.Task.Parameters):
        arc_repo_path = sdk2.parameters.String(
            "Путь к приложению в аркадии",
            description='Пример: market/front/apps/marketfront',
            required=True
        )

        arc_branch = sdk2.parameters.String(
            "Ветка в arc",
            default_value="trunk"
        )

        app_root_path = sdk2.parameters.String(
            "Кастомный путь корня приложения внутри репозитория",
            default_value="",
            required=False
        )

        stats_manifest = sdk2.parameters.String(
            "Путь до манифеста с описанием собираемых метрик"
        )

        ubuntu_version = create_ubuntu_selector()
        node_version = create_node_selector()

    class Requirements(task_env.TinyRequirements):
        dns = ctm.DnsType.DNS64
        environments = [
            PipEnvironment("python-statface-client", "0.78.0", use_wheel=True),
        ]

    def on_enqueue(self):
        super(MarketFrontCollectCodeStatsArc, self).on_enqueue()
        setup_container(self)

    def on_prepare(self):
        super(MarketFrontCollectCodeStatsArc, self).on_prepare()
        self.arc_mount()

    def on_finish(self, prev_status, status):
        super(MarketFrontCollectCodeStatsArc, self).on_finish(prev_status, status)
        self.arc_unmount()

    def on_wait(self):
        super(MarketFrontCollectCodeStatsArc, self).on_wait()
        self.arc_unmount()

    def on_execute(self):
        super(MarketFrontCollectCodeStatsArc, self).on_execute()

        with MetatronEnvArc(self, nodejs_version=self.Parameters.node_version):
            self.arc_checkout(branch=self.Parameters.arc_branch, track=True)
            self._collect_metrics()

    def _collect_metrics(self):
        if self.Parameters.app_root_path:
            config_path = os.path.join(self.arcadia, self.Parameters.arc_repo_path, self.Parameters.app_root_path, self.Parameters.stats_manifest)
        else:
            config_path = os.path.join(self.arcadia, self.Parameters.arc_repo_path, self.Parameters.stats_manifest)

        with open(config_path) as file:
            config = json.load(file)
            for metric_config in config:
                self._collect_code_metric(
                    metric_config["statfaceReport"],
                    metric_config["collectScript"],
                    metric_config["resultFile"],
                )

    def _collect_code_metric(self, statface_report, collect_script, result_file):
        logging.info("Collecting data for report {}".format(statface_report))
        alias = "collect_{}".format(statface_report.lower().replace("/", "_"))

        if self.Parameters.app_root_path:
            app_root_dir = os.path.join(self.arcadia, self.Parameters.arc_repo_path, self.Parameters.app_root_path)
        else:
            app_root_dir = os.path.join(self.arcadia, self.Parameters.arc_repo_path)

        rich_check_call(["bash", "-c", collect_script], self, alias, cwd=app_root_dir)

        result_file_path = os.path.join(app_root_dir, result_file)

        with open(result_file_path) as result_file:
            result = json.load(result_file)
            if not isinstance(result, list):
                result = [result]

            today = datetime.datetime.now().strftime('%Y-%m-%d')
            for row in result:
                if 'fielddate' not in row:
                    row['fielddate'] = today
            self._send_to_statface(statface_report, result)

    def _send_to_statface(self, report_id, data):
        from statface_client import StatfaceClient, STATFACE_PRODUCTION
        logging.info("Uploading data to {}".format(report_id))
        oauth_token = sdk2.Vault.data(self.STATFACE_TOKEN_VAULT_KEY)
        client = StatfaceClient(oauth_token=oauth_token, host=STATFACE_PRODUCTION)
        report = client.get_report(report_id)
        report.upload_data("daily", data)
