# -*- coding: utf-8 -*-
import logging
import os
import pipes

import sandbox.sdk2.path as spath
from sandbox import sdk2
from sandbox.projects.market.frontarc.helpers.MetatronEnvArc import MetatronEnvArc
from sandbox.projects.market.frontarc.helpers.sandbox_helpers import rich_check_call
from sandbox.projects.market.frontarc.helpers.ubuntu import create_ubuntu_selector, setup_container
from sandbox.projects.market.frontarc.helpers.node import NODE_DEFAULT
from sandbox.projects.market.frontarc.helpers.node import create_node_selector
from sandbox.projects.market.frontarc.helpers.ArcDefaultParameters import ArcDefaultParameters
from sandbox.projects.market.frontarc.helpers.DeliveryPipelineParams import DeliveryPipelineParams
from sandbox.projects.market.frontarc.helpers.arc_base import MarketFrontArcBase
from sandbox.projects.market.resources import MARKET_TESTPALM_IDS
from sandbox.projects.sandbox_ci.utils import env
from sandbox.projects.common import task_env


class MarketFrontDiffectorArc(sdk2.Task, MarketFrontArcBase):
    """
    Таска для отслеживания измененных страниц и получения TestPalm id данных страниц
    """

    ROOT_DIR = ''
    APP_SRC_PATH = ''

    class Requirements(task_env.TinyRequirements):
        ram = 1024 * 12
        disk_space = 10 * 1000

    class Parameters(sdk2.Task.Parameters):
        ubuntu_version = create_ubuntu_selector()
        node_version = create_node_selector()
        arc_parameters = ArcDefaultParameters()
        dp_params = DeliveryPipelineParams()

        with sdk2.parameters.Group('Environment') as environ_block:
            environ = sdk2.parameters.Dict('Environment variables')

    def on_enqueue(self):
        super(MarketFrontDiffectorArc, self).on_enqueue()
        setup_container(self)

    @property
    def node_version(self):
        return self.Parameters.node_version or NODE_DEFAULT

    @property
    def project_name(self):
        return self.Parameters.app_repo

    @property
    def report_description(self):
        return self.project_name

    @property
    def TESTPALM_API_TOKEN(self):
        return os.environ["TESTPALM_API_TOKEN"]

    @property
    def ARCANUM_OAUTH_TOKEN(self):
        # thanks to Metatron
        return os.environ["ARC_TOKEN"]

    def _clone_repo(self):
        self.arc_checkout(
            branch=self.Parameters.head_branch,
        )

    def _prepare_dirs(self):
        self.ROOT_DIR = str(self.path())
        self.APP_SRC_PATH = os.path.join(
            self.arcadia, self.Parameters.root_path
        )

    def _prepare_env(self):
        # export order for correct priority
        os.environ['TESTPALM_ACCESS_TOKEN'] = self.TESTPALM_API_TOKEN
        os.environ['PATH'] = ":".join([
            os.environ['PATH'],
            os.path.dirname(self.arc.binary_path)
        ])

        # Для TBD пайплайнов diffector должен сравнивать ревизии. Исключение - сборка хотфикса из ветки
        if (self.Parameters.dpp_revision and self.Parameters.dpp_stableRevision) and not self.Parameters.dpp_branch:
            base = self.Parameters.dpp_lastRunningRevision if self.Parameters.dpp_lastRunningRevision else self.Parameters.dpp_stableRevision
            os.environ["DIFFECTOR_BASE_BRANCH"] = '--baseBranch={}'.format(base)
            os.environ["DIFFECTOR_COMPARE_BRANCH"] = '--compareBranch={}'.format(self.Parameters.dpp_revision)

        env.export(self.Parameters.environ)

    def _configure(self):
        rich_check_call(
            ["make", "configure"],
            task=self, alias="configure", cwd=self.APP_SRC_PATH
        )

    def _bootstrap(self):
        logging.info(
            'Running project bootstrap with environment: {}'.format(os.environ))

        rich_check_call(
            ["make", "bootstrap"],
            task=self, alias="bootstrap", cwd=self.APP_SRC_PATH
        )

    def _run_testpalm_scope(self):
        logging.info(
            'Running diffector and testpalm scripts'
        )

        rich_check_call(
            ["npm", "run", "testcases"],
            task=self, alias="testcases", cwd=self.APP_SRC_PATH
        )

        src_testpalm_ids_report_path = os.path.join(
            self.APP_SRC_PATH, 'txt_reports/testpalm_ids_report')
        testpalm_ids_report = spath.Path(
            src_testpalm_ids_report_path).read_bytes()

        resource = sdk2.ResourceData(MARKET_TESTPALM_IDS(
            self, "TestPalm IDS", "testpalm_ids"
        ))
        resource.path.write_bytes(testpalm_ids_report)
        resource.ready()

    def on_prepare(self):
        with MetatronEnvArc(self, nodejs_version=self.Parameters.node_version), self.memoize_stage.preparation(max_runs=1):
            self.arc_mount()
            self._prepare_dirs()
            self._prepare_env()
            self._clone_repo()
            self._configure()
            self._bootstrap()

    def on_execute(self):
        with MetatronEnvArc(self, nodejs_version=self.Parameters.node_version), self.memoize_stage.execute(max_runs=1):
            self._prepare_dirs()
            self._prepare_env()
            self._run_testpalm_scope()

    def on_finish(self, prev_status, status):
        self.arc_unmount()

    def on_wait(self, prev_status, status):
        self.arc_unmount()
