# coding=utf-8

import logging
import os
import os.path
import pipes
import re
import shutil
import tempfile

from sandbox.sandboxsdk import environments
import sandbox.common.types.misc as ctm
from sandbox import sdk2
import sandbox.sdk2.path as spath
from sandbox.projects.market.frontarc.helpers.arc_base import MarketFrontArcBase
from sandbox.projects.common.nanny import nanny

import sandbox.projects.market.frontarc.helpers.s3 as s3
from sandbox.projects.market.frontarc.helpers.MetatronEnvArc import MetatronEnvArc
from sandbox.projects.market.frontarc.helpers.startrack import FixVersion
from sandbox.projects.market.frontarc.helpers.sandbox_helpers import rich_check_call, \
    rich_check_start, rich_check_report
from sandbox.projects.market.frontarc.helpers.node import create_node_selector
from sandbox.projects.market.frontarc.helpers.ubuntu import create_ubuntu_selector, setup_container, UBUNTU_TRUSTY
from sandbox.projects.market.frontarc.helpers.tsum import send_report_to_tsum
from sandbox.projects.common import task_env
from sandbox.projects.market.frontarc.helpers.compression import compress_brotli, compress_zopfli


class GenericParametersArc(sdk2.Task.Parameters):
    copy_static = sdk2.parameters.Bool("Копировать фриз статики в приложение", required=True, default=True)
    build_static = sdk2.parameters.Bool("Проводить сборку статического пакета", required=True, default=True)
    with build_static.value[True]:
        fix_version = FixVersion("Фикс-версия релиза в формате YYYY.R.PS", required=True)

    with sdk2.parameters.Group("Приложение") as app:
        need_merge_commit = sdk2.parameters.Bool("Нужен merge-коммит", required=False, default=True)
        app_src_dir = sdk2.parameters.String("Кастомный путь корня приложения внутри репозитория")
        app_branch = sdk2.parameters.String(
            "Ветка репозитория приложения",
            default_value="",
            required=False
        )
        app_commit = sdk2.parameters.String(
            "Коммит",
            default_value="",
            required=False
        )
        dist_dir = sdk2.parameters.String("Папка дистрибутива сборки")

    """
    Протокол для работы с репозиторием common
    """
    common_scheme = ""

    """
    URI репозитория common (без протокола)
    """
    common_repo = ""

    with sdk2.parameters.Group("common") as common:
        common_branch = sdk2.parameters.String(
            "Ветка репозитория common",
            default_value="",
            required=False
        )

    with sdk2.parameters.Group("ЦУМ") as tsum:
        tsum_send_report = sdk2.parameters.Bool(
            "Отсылать отчет о приложении",
            required=True,
            default=False
        )

        with tsum_send_report.value[True]:
            tsum_api_hosts = sdk2.parameters.List(
                'API hosts',
                sdk2.parameters.String,
                default=[
                    'https://tsum-agent-api.market.yandex.net:4209',
                    'https://tsum-agent-api-testing.market.yandex.net:4209'
                ],
                required=True
            )

    with sdk2.parameters.Group("Штуки") as misc:
        ubuntu_version = create_ubuntu_selector()
        node_version = create_node_selector()
        prune = sdk2.parameters.Bool("Делать prune проекта", default=True, required=True)
        with prune.value[True]:
            prune_old_style = sdk2.parameters.Bool("Old style npm prune", default=True, required=True)

    environ = sdk2.parameters.Dict('Environment variables')


class MarketFrontNodejsBuildAbstractArc(nanny.ReleaseToNannyTask2, sdk2.Task, MarketFrontArcBase):
    """
    Унифицированная сборка маркетфронтовых приложений на nodejs
    """

    RESOURCES_DIR = os.path.join(os.path.dirname(__file__), "..", "helpers")
    YENV = "production"
    NPM_REGISTRY = "http://npm.yandex-team.ru"
    APP_SRC_DIR = "app_src"
    STATIC_CARCH = "all"

    @property
    def resource_type(self):
        """
        Тип ресурса для архива с приложением
        """
        raise NotImplementedError

    @property
    def app_repo(self):
        """
        Путь к приложению приложения в аркадии
        """
        raise NotImplementedError

    @property
    def static_root(self):
        """
        Путь статики внутри пакета (н.п market-skubi/_)
        """
        return None

    @property
    def dist_dir(self):
        """
        Папка дистрибутива приложения, относительно корня репозитория
        """
        return self.Parameters.dist_dir if self.Parameters.dist_dir else None

    @property
    def s3_prefix(self):
        return s3.s3_path_by_repo(self.app_repo)

    def make_dist_path(self, app_src_path):
        return app_src_path if not self.dist_dir else os.path.join(app_src_path, self.dist_dir)

    @property
    def make_env(self):
        make_env = os.environ.copy()
        make_env["YENV"] = self.YENV

        if self.Parameters.common_branch:
            make_env.update({
                "COMMON_SCHEME": self.Parameters.common_scheme,
                "COMMON_REPO": self.Parameters.common_repo,
                "COMMON_REV": self.Parameters.common_branch,
            })

        if self.dist_dir:
            make_env["DIST_DIR"] = self.dist_dir

        if self.Parameters.prune and not self.Parameters.prune_old_style:
            make_env["PRUNE"] = "1"

        if self.Parameters.environ:
            make_env.update(self.Parameters.environ)

        return make_env



    class Parameters(GenericParametersArc):
        pass

    class Requirements(task_env.BuildLinuxRequirements):
        dns = ctm.DnsType.DNS64
        environments = (
            environments.PipEnvironment('boto', use_wheel=True),
        )
        cores = 6
        ram = 1024 * 12
        disk_space = 32 * 1000

    def on_enqueue(self):
        super(MarketFrontNodejsBuildAbstractArc, self).on_enqueue()
        setup_container(self)


    def on_prepare(self):
        super(MarketFrontNodejsBuildAbstractArc, self).on_prepare()
        self.arc_mount()

    def on_finish(self, prev_status, status):
        super(MarketFrontNodejsBuildAbstractArc, self).on_finish(prev_status, status)
        self.arc_unmount()

    def on_wait(self):
        super(MarketFrontNodejsBuildAbstractArc, self).on_wait()
        self.arc_unmount()

    def on_execute(self):
        super(MarketFrontNodejsBuildAbstractArc, self).on_execute()

        with MetatronEnvArc(self, nodejs_version=self.Parameters.node_version):
            app_src_path = os.path.join(self.arcadia, self.app_repo)
            logging.info('app_src_path: ${}'.format(app_src_path))
            dist_path = self.make_dist_path(app_src_path)
            logging.info('dist_path: ${}'.format(dist_path))

            self._prepare_repos()

            static_version = None
            if self.Parameters.build_static:
                static_version = "{}.t{}".format(self.Parameters.fix_version, self.id)

            self._app_build(app_src_path, dist_path, static_version)

            # todo: refactor this
            #  @see: https://st.yandex-team.ru/MARKETFRONTECH-1045
            #  @see: https://st.yandex-team.ru/MARKETFRONTECH-1042
            if self.Parameters.build_static:
                self._upload_static_to_s3(app_src_path, dist_path)

            if self.Parameters.copy_static:
                self._static_copy(app_src_path, dist_path)

            pack_app_archive_path = self._create_app_pack(app_src_path, dist_path)

            if self.Parameters.tsum_send_report:
                resource = sdk2.Resource.find(
                    resource_type=self.resource_type.__name__,
                    task_id=self.id
                ).first()
                with sdk2.helpers.ProcessLog(self, "tsum-report"):
                    send_report_to_tsum(
                        archive=pack_app_archive_path,
                        sandbox_resource_type=self.resource_type.__name__,
                        sandbox_ticket_num=self.id,
                        sandbox_resource_id=getattr(resource, "id", None),
                        tsum_api_hosts=self.Parameters.tsum_api_hosts
                    )

    def prebuild(self, app_src_path):
        pass

    def _app_build(self, app_src_path, dist_path, static_version):
        self.prebuild(app_src_path)

        if self.Parameters.app_src_dir:
            working_dir = os.path.join(app_src_path, self.Parameters.app_src_dir)
        else:
            working_dir = app_src_path

        logging.info('working_dir for make: {}'.format(working_dir))
        rich_check_call(
            ["make"],
            task=self, alias="make", cwd=working_dir, env=self.make_env
        )

        self._prune_deps(dist_path)

        if self.Parameters.build_static:
            if self.Parameters.app_src_dir and self.platform:
                version_dir = os.path.join(app_src_path, self.Parameters.app_src_dir, self.platform)
            else:
                version_dir = dist_path
            with open(os.path.join(version_dir, "version.txt"), "a") as f:
                f.write("package_version={}".format(static_version))

    def _create_app_pack(self, app_src_path, dist_path):
        app_repo_match = re.search(".+/([^/.]+?)(:?.git)?$", self.app_repo)
        assert app_repo_match is not None

        # todo: remove after white and blue are unified
        #  @see: https://st.yandex-team.ru/MARKETFRONTECH-506
        #  @see: https://st.yandex-team.ru/MARKETFRONT-15211
        if self.Parameters.app_src_dir and self.platform:
            working_dir = os.path.join(app_src_path, self.Parameters.app_src_dir, self.platform)
        else:
            working_dir = dist_path

        pack_app_path_o = spath.Path(tempfile.gettempdir(), "application")

        self._prepare_pack_app(pack_app_path_o)

        rsync_filter_path = os.path.join(working_dir, ".rsync-filter")

        rich_check_call(
            [
                "rsync", "-aHz", "--delete", "--force", "--exclude-from=" + rsync_filter_path,
                                                       working_dir + "/", str(pack_app_path_o / "app")
            ],
            task=self, alias="rsync"
        )

        pack_app_archive_path = tempfile.mktemp(suffix=".tar.gz", prefix="app")

        rich_check_call(
            ["tar", "-C", str(pack_app_path_o / "../"), "-czf", pack_app_archive_path, "application"],
            task=self, alias="create_app_pack"
        )

        self._app_create_resource(pack_app_archive_path)

        return pack_app_archive_path

    def _app_create_resource(self, pack_app_archive_path):
        app_res = sdk2.ResourceData(self.resource_type(
            self, "App tarball", "app.tar.gz"
        ))
        if app_res.path.exists():
            app_res.path.unlink()
        app_res.path.write_bytes(spath.Path(pack_app_archive_path).read_bytes())
        app_res.ready()

    @staticmethod
    def _prepare_pack_app(pack_app_path_o):
        if pack_app_path_o.exists():
            shutil.rmtree(str(pack_app_path_o))

        pack_app_path_o.mkdir()

        for appdir in ["app", "run"]:
            (pack_app_path_o / appdir).mkdir()

    def _prune_deps(self, app_src_path):
        if self.Parameters.prune and self.Parameters.prune_old_style:
            rich_check_call(
                ["npm", "prune", "--production"],
                task=self, alias="npm_prune", cwd=app_src_path
            )


    def _prepare_repos(self):
        if self.Parameters.app_commit and self.Parameters.need_merge_commit:
            self.arc_checkout(self.Parameters.app_commit)
        else:
            self.arc_checkout(self.Parameters.app_branch)

    @staticmethod
    def _package_name_version(version):
        return re.sub(r"[.]", "-", version)

    def _upload_static_to_s3(self, app_src_path, dist_path):
        # todo: remove after white and blue are unified
        #  @see: https://st.yandex-team.ru/MARKETFRONTECH-506
        #  @see: https://st.yandex-team.ru/MARKETFRONT-15211
        if self.Parameters.app_src_dir and self.platform:
            working_dir = os.path.join(app_src_path, self.Parameters.app_src_dir, self.platform)
        else:
            working_dir = dist_path

        with MetatronEnvArc(self):
            conn = s3.get_connection()

        static_path = os.path.join(working_dir, "s3", "market-static")

        if not os.path.exists(static_path):
            static_path = os.path.join(working_dir, "freeze", "static")

        filename_blacklist = ["manifest.json"]

        processes = []

        for root, dirs, files in os.walk(static_path):
            for filename in files:
                if filename in filename_blacklist:
                    continue
                file_path = os.path.join(root, filename)

                _, __, check_zopfli_report_cb = compress_zopfli(self, file_path, filename)
                _, __, check_brotli_report_cb = compress_brotli(self, file_path, filename, check_size=False)

                processes.append((filename, file_path, check_zopfli_report_cb, check_brotli_report_cb))

        for (filename, file_path, check_zopfli_report_cb, check_brotli_report_cb) in processes:
            check_zopfli_report_cb()
            check_brotli_report_cb()

            uploading_path = os.path.join(self.s3_prefix, os.path.relpath(file_path, static_path))
            logging.info('Uploading to: {}'.format(uploading_path))
            s3.upload_file(conn, s3.S3_BUCKET_NAME, file_path, uploading_path)

            logging.info('Uploading gz/br to: {}'.format(uploading_path))
            s3.upload_file(conn, s3.S3_BUCKET_NAME, file_path + ".gz", "{}.gz".format(uploading_path))
            s3.upload_file(conn, s3.S3_BUCKET_NAME, file_path + ".br", "{}.br".format(uploading_path))

    def _static_copy(self, app_src_path, dist_path):
        # todo: remove after white and blue are unified
        #  @see: https://st.yandex-team.ru/MARKETFRONTECH-506
        #  @see: https://st.yandex-team.ru/MARKETFRONT-15211
        if self.Parameters.app_src_dir and self.platform:
            working_dir = os.path.join(app_src_path, self.Parameters.app_src_dir, self.platform)
        else:
            working_dir = dist_path

        logging.info('working dir path for static copy: {}'.format(working_dir))

        freeze_static_path = os.path.join(working_dir, "freeze/static")
        s3_static_path = os.path.join(working_dir, "s3", pipes.quote(s3.S3_BUCKET_NAME), pipes.quote(self.s3_prefix))

        rich_check_call(["mkdir", "-vp", s3_static_path], self, 'static-s3-mkdir')
        rich_check_call(["rm", "-vrf", s3_static_path], self, 'static-s3-remove')
        rich_check_call(["mv", "-vfT", freeze_static_path, s3_static_path], self, 'static-s3-move')

        if self.static_root is not None:
            nginx_static_path = os.path.join(working_dir, pipes.quote(self.static_root))

            # https://a.yandex-team.ru/arc_vcs/market/sre/nanny/front/touch/tmpl/conf/nginx/sites-enabled/63-touch.conf?rev=r9341073#L216
            rich_check_call(["mkdir", "-vp", nginx_static_path], self, 'static-nginx-mkdir')
            rich_check_call(["rm", "-vrf", nginx_static_path], self, 'static-nginx-remove')
            rich_check_call(["cp", "-vrfT", s3_static_path, nginx_static_path], self, 'static-nginx-copy')


