# coding=utf-8

import json
import logging
import os
import os.path
import pipes
import re
import shutil
import tempfile
import sys
import inspect

import sandbox.common.types.misc as ctm
from sandbox import sdk2
import sandbox.sdk2.path as spath
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sdk2.vcs.git import Git
from sandbox.projects.common.debpkg import DebRelease
from sandbox.projects.common.nanny import nanny

from sandbox.projects.market.frontarc.helpers.MetatronEnvArc import MetatronEnvArc
from sandbox.projects.market.frontarc.helpers.startrack import FixVersion
from sandbox.projects.market.frontarc.helpers.sandbox_helpers import rich_check_call
from sandbox.projects.market.frontarc.helpers.node import create_node_selector
from sandbox.projects.market.frontarc.helpers.ubuntu import create_ubuntu_selector, setup_container
from sandbox.projects.market.frontarc.helpers.tsum import send_report_to_tsum
import sandbox.projects.market.resources as market_resources
from sandbox.projects.market.front.resources import MarketFrontBuildUnifiedStaticResource

# проверяем объект пакета на то что он класс и подкласс MarketFrontCommonNodeAppResource но не сам класс
# (скажи это быстро, 3 раза)
def check_marker(obj):
    return inspect.isclass(obj) and issubclass(obj, market_resources.MarketFrontCommonNodeAppResource) \
           and obj != market_resources.MarketFrontCommonNodeAppResource


def get_node_app_resources():
    return inspect.getmembers(sys.modules[market_resources.__name__], check_marker)


def get_resource_types():
    return dict(map(lambda (name, resource): (resource.__name__, resource), get_node_app_resources()))


RESOURCE_TYPES = get_resource_types()

class MarketFrontNodejsBuildBaseArc(nanny.ReleaseToNannyTask2, sdk2.Task):
    """
    Сборка маркетфронтовых приложений на nodejs
    """

    PACKAGES_REPO = "git://github.yandex-team.ru/market/packages.git"
    RESOURCES_DIR = os.path.join(os.path.dirname(__file__), "..", "helpers")
    YENV = "production"
    NPM_REGISTRY = "http://npm.yandex-team.ru"
    APP_SRC_DIR = "app_src"
    STATIC_CARCH = "all"
    MAKEDEBPKG_PATH = "/opt/makedebpkg"

    @property
    def static_package_name(self):
        """
        Имя статического пакета
        """
        return self.Parameters.static_package_name

    @property
    def static_root(self):
        """
        Путь статики внутри пакета (н.п market-skubi/_)
        """
        return self.Parameters.static_root

    @property
    def resource_type(self):
        """
        Тип ресурса для архива с приложением
        """
        return self.Parameters.resource_type

    @property
    def app_repo(self):
        """
        Адрес репозитория приложения
        """
        return self.Parameters.app_repo

    @property
    def dist_dir(self):
        """
        Папка дистрибутива приложения, относительно корня репозитория
        """
        return self.Parameters.dist_dir if self.Parameters.dist_dir else None

    def make_dist_path(self, app_src_path):
        return app_src_path if not self.dist_dir else os.path.join(app_src_path, self.dist_dir)

    @property
    def make_env(self):
        make_env = os.environ.copy()

        # build_env имеет приоритет ниже чем остальные параметры влияющие на окружение
        if self.Parameters.build_env:
            make_env.update(self.Parameters.build_env)

        make_env["YENV"] = self.YENV

        if self.Parameters.common_branch:
            make_env.update({
                "COMMON_SCHEME": self.Parameters.common_scheme,
                "COMMON_REPO": self.Parameters.common_repo,
                "COMMON_REV": self.Parameters.common_branch,
            })

        if self.dist_dir:
            make_env["DIST_DIR"] = self.dist_dir

        if self.Parameters.prune and not self.Parameters.prune_old_style:
            make_env["PRUNE"] = "1"

        return make_env

    """
    Ветка репозитория https://github.yandex-team.ru/market/packages
    """
    packages_branch = "master"

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64


    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.String("Тип ресурса приложения", required=True, multiline=True) as resource_type:
            for type_name in RESOURCE_TYPES:
                resource_type.values[type_name] = resource_type.Value(type_name)

        with sdk2.parameters.Group("статика") as static:
            static_root = sdk2.parameters.String("Путь статики внутри пакета (н.п market-skubi/_)", required=True)
            copy_static = sdk2.parameters.Bool("Копировать фриз статики в приложение", required=True, default=True)
            build_static = sdk2.parameters.Bool("Проводить сборку статического пакета", required=True, default=True)
            with build_static.value[True]:
                fix_version = FixVersion("Фикс-версия релиза в формате YYYY.R.PS", required=True)
                static_package_name = sdk2.parameters.String("Имя статического пакета", required=True)

        with sdk2.parameters.Group("Приложение") as app:
            app_repo = sdk2.parameters.String("Адрес репозитория приложения", required=True)
            app_branch = sdk2.parameters.String(
                "Ветка репозитория приложения",
                default_value="master",
                required=True
            )
            dist_dir = sdk2.parameters.String("Папка дистрибутива сборки")

        """
        Протокол для работы с репозиторием common
        """
        common_scheme = ""

        """
        URI репозитория common (без протокола)
        """
        common_repo = ""

        with sdk2.parameters.Group("common") as common:
            common_branch = sdk2.parameters.String(
                "Ветка репозитория common",
                default_value="",
                required=False
            )

        with sdk2.parameters.Group("ЦУМ") as tsum:
            tsum_send_report = sdk2.parameters.Bool(
                "Отсылать отчет о приложении",
                required=True,
                default=False
            )

            with tsum_send_report.value[True]:
                tsum_api_hosts = sdk2.parameters.List(
                    'API hosts',
                    sdk2.parameters.String,
                    default=[
                        'https://tsum-agent-api.market.yandex.net:4209',
                        'https://tsum-agent-api-testing.market.yandex.net:4209'
                    ],
                    required=True
                )

        with sdk2.parameters.Group("Сборка") as misc:
            ubuntu_version = create_ubuntu_selector()
            node_version = create_node_selector()
            prune = sdk2.parameters.Bool("Делать prune проекта", default=True, required=True)
            with prune.value[True]:
                prune_old_style = sdk2.parameters.Bool("Old style npm prune", default=True, required=True)
            build_env = sdk2.parameters.Dict("Переменные окружения сборки")

    def on_enqueue(self):
        super(MarketFrontNodejsBuildBaseArc, self).on_enqueue()
        setup_container(self)

    def on_execute(self):
        with MetatronEnvArc(self, nodejs_version=self.Parameters.node_version):
            root_dir = tempfile.mkdtemp()
            packages_path = os.path.join(root_dir, "packages")
            unified_static_path = os.path.join(packages_path, "yandex-market-unified-static")
            app_src_path = os.path.join(packages_path, self.APP_SRC_DIR)
            dist_path = self.make_dist_path(app_src_path)

            self._prepare_repos(app_src_path, packages_path)

            static_version = None
            if self.Parameters.build_static:
                static_version = "{}.t{}".format(self.Parameters.fix_version, self.id)

            self._app_build(app_src_path, dist_path, static_version)

            if self.Parameters.copy_static:
                self._static_copy(dist_path)

            if self.Parameters.build_static:

                changes_path_o = self._static_build(packages_path, dist_path, unified_static_path, static_version)
                self._upload_static(packages_path, changes_path_o)

            pack_app_archive_path = self._create_app_pack(dist_path)

            if self.Parameters.tsum_send_report:
                resource = sdk2.Resource.find(
                    resource_type=self.resource_type,
                    task_id=self.id
                ).first()
                with sdk2.helpers.ProcessLog(self, "tsum-report"):
                    send_report_to_tsum(
                        archive=pack_app_archive_path,
                        sandbox_resource_type=self.resource_type,
                        sandbox_ticket_num=self.id,
                        sandbox_resource_id=getattr(resource, "id", None),
                        tsum_api_hosts=self.Parameters.tsum_api_hosts
                    )

    @staticmethod
    def _git_clone(url, ref, target_dir='.'):
        git = Git(url)
        git.clone(str(target_dir), ref)

    def _upload_static(self, packages_path, changes_path_o):
        with sdk2.helpers.ProcessLog(self, "perl") as process_log:
            perl = sp.Popen(
                ["perl", os.path.join(self.RESOURCES_DIR, "dupload_config_parser.pl"),
                    os.path.join(packages_path, "dupload.conf")],
                stdout=sp.PIPE, stderr=process_log.stderr
            )

            (stdout, stderr) = perl.communicate()
            perl.wait()
            logging.info("perl stdout: %s", stdout)

            dupload_conf = json.loads(stdout)

        with DebRelease(dupload_conf, login="robot-metatron"):
            rich_check_call(
                ["dupload", "--to", "verstka", str(changes_path_o)],
                task=self, alias="dupload"
            )

    def _app_build(self, app_src_path, dist_path, static_version):
        rich_check_call(
            ["make"],
            task=self, alias="make", cwd=app_src_path, env=self.make_env
        )

        self._prune_deps(dist_path)

        if self.Parameters.build_static:
            with open(os.path.join(dist_path, "version.txt"), "a") as f:
                f.write("package_version={}".format(static_version))

    def _create_app_pack(self, app_src_path):
        app_repo_match = re.search(".+/([^/.]+?)(:?.git)?$", self.app_repo)
        assert app_repo_match is not None

        pack_app_dir = app_repo_match.group(1)
        pack_app_path_o = spath.Path(tempfile.gettempdir(), pack_app_dir)

        self._prepare_pack_app(pack_app_path_o)

        rsync_filter_path = os.path.join(app_src_path, ".rsync-filter")

        rich_check_call(
            [
                "rsync", "-aHz", "--delete", "--force", "--exclude-from=" + rsync_filter_path,
                                                       app_src_path + "/", str(pack_app_path_o / "app")
            ],
            task=self, alias="rsync"
        )

        pack_app_archive_path = tempfile.mktemp(suffix=".tar.gz", prefix="app")

        rich_check_call(
            ["tar", "-C", str(pack_app_path_o / "../"), "-czf", pack_app_archive_path, pack_app_dir],
            task=self, alias="create_app_pack"
        )

        self._app_create_resource(pack_app_archive_path)

        return pack_app_archive_path

    def _app_create_resource(self, pack_app_archive_path):
        resource_type = RESOURCE_TYPES[self.resource_type]
        app_res = sdk2.ResourceData(resource_type(
            self, "App tarball", "app.tar.gz"
        ))
        if app_res.path.exists():
            app_res.path.unlink()
        app_res.path.write_bytes(spath.Path(pack_app_archive_path).read_bytes())
        app_res.ready()

    @staticmethod
    def _prepare_pack_app(pack_app_path_o):
        if pack_app_path_o.exists():
            shutil.rmtree(str(pack_app_path_o))

        pack_app_path_o.mkdir()

        for appdir in ["app", "run"]:
            (pack_app_path_o / appdir).mkdir()

    def _prune_deps(self, app_src_path):
        if self.Parameters.prune and self.Parameters.prune_old_style:
            rich_check_call(
                ["npm", "prune", "--production"],
                task=self, alias="npm_prune", cwd=app_src_path
            )

    def _prepare_repos(self, app_src_path, packages_path):
        self._git_clone(self.PACKAGES_REPO, pipes.quote(self.packages_branch), packages_path)
        self._git_clone(
            pipes.quote(self.app_repo),
            pipes.quote(self.Parameters.app_branch),
            app_src_path
        )

    @staticmethod
    def _package_name_version(version):
        return re.sub(r"[.]", "-", version)

    def _static_copy(self, app_src_path):
        freeze_static_path = os.path.join(app_src_path, "freeze/static")
        static_path = os.path.join(app_src_path, pipes.quote(self.static_root))

        static_path_o = spath.Path(static_path)
        if not static_path_o.exists():
            static_path_o.mkdir(parents=True)

        freeze_static_path_o = spath.Path(freeze_static_path)
        assert freeze_static_path_o.exists()

        for path in freeze_static_path_o.iterdir():
            dest = os.path.join(static_path, path.name)
            logging.info("moving %s to %s", str(path), dest)
            path.rename(dest)

        freeze_static_path_o.rmdir()

    def _static_build(self, packages_path, app_src_path, unified_static_path, static_version):
        makedebpkg_config_path = os.path.join(packages_path, "makedebpkg.conf")

        static_path = os.path.join(app_src_path, pipes.quote(self.static_root))
        os.symlink(self.MAKEDEBPKG_PATH, os.path.join(packages_path, ".makedebpkg"))

        static_fullname = "{}-{}".format(
            self.static_package_name,
            self._package_name_version(static_version)
        )

        os.environ.update({
            "PATH": self.MAKEDEBPKG_PATH + ":" + os.environ["PATH"],
            "PACKAGE": self.static_package_name,
            "VERSION": static_version,
            "PACKAGE_FULLNAME": static_fullname,
            "SOURCES": static_path,
            "CARCH": self.STATIC_CARCH
        })

        rich_check_call(
            [
                self.MAKEDEBPKG_PATH + "/makedebpkg", "--config", makedebpkg_config_path, "--log", "--sign",
                "--nocheck", "--key", pipes.quote(os.environ["GPGKEY"])
            ],
            task=self, alias="build_static", cwd=unified_static_path
        )

        return self._static_create_resource(static_fullname, static_version, unified_static_path)

    def _static_create_resource(self, static_fullname, static_version, unified_static_path):
        with open(os.path.join(unified_static_path, "PKGBUILD")) as f:
            match = re.search("pkgrel=(\d+)", f.read())
            assert match is not None
            pkgrel = match.group(1)

        static_version_with_pkgrel = static_version + "-" + pkgrel

        static_res = sdk2.ResourceData(MarketFrontBuildUnifiedStaticResource(
            self, "Static package", "static",
            fix_version=self.Parameters.fix_version,
            version=static_version_with_pkgrel
        ))

        if not static_res.path.exists():
            static_res.path.mkdir()

        def make_pkg_path(pkg_path_suffix):
            path_o = spath.Path(
                unified_static_path,
                static_fullname + "_" + static_version_with_pkgrel + pkg_path_suffix
            )
            return path_o

        changes_suffix = "_" + self.STATIC_CARCH + ".changes"
        for suffix in ["_" + self.STATIC_CARCH + ".deb", changes_suffix, ".dsc"]:
            file_path_o = make_pkg_path(suffix)

            logging.info("static file path %s", str(file_path_o))
            assert file_path_o.exists()
            dest = (static_res.path / file_path_o.name)
            if dest.exists():
                dest.unlink()

            dest.write_bytes(file_path_o.read_bytes())

        static_res.ready()

        return make_pkg_path(changes_suffix)
