# -*- coding: utf-8 -*-

import logging
import os
from pprint import pformat

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types import misc as ctm
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.market.frontarc.helpers.sandbox_helpers import rich_check_call
from sandbox.projects.market.frontarc.helpers.ubuntu import create_ubuntu_selector, setup_container
from sandbox.projects.market.frontarc.helpers.node import create_node_selector
from sandbox.projects.market.frontarc.helpers.MetatronEnvArc import MetatronEnvArc

from sandbox.projects.market.frontarc.MarketFrontPorchmarkSaveReportArc.parse import parse

DISK_SPACE = 100  # 100Mb

# see https://st.yandex-team.ru/GOALZ-87197#5faee653f8f9e4508c8cc7e2
DEFAULT_DB_HOST = "man-o0g0hvwacmisk691.db.yandex.net," \
                  "vla-vr7489tltey95p3e.db.yandex.net," \
                  "sas-xvqk00vicovbbiqh.db.yandex.net"
DEFAULT_DB_PORT = "6432"
DEFAULT_DB_DATABASE = "porchmark-metrics"
DEFAULT_DB_USER = "porchmark"

PASSWORD_VAULT_KEY = "porchmark-metrics-db-password"


class MarketFrontPorchmarkSaveResultArc(sdk2.Task):
    """
    Сохранение porchmark-результатов в БД
    """

    root_dir = None
    report_dir = None
    report_resource = None
    metrics = None

    class Context(sdk2.Context):
        compare_task_id = None

    class Parameters(sdk2.Task.Parameters):
        ubuntu_version = create_ubuntu_selector()
        node_version = create_node_selector()

        report_resource_id = sdk2.parameters.Integer(
            'Id ресурса MARKET_PORCHMARK_REPORT',
            required=True
        )

        with sdk2.parameters.Group('GitHub репозиторий проекта') as github_repo_block:
            app_owner = sdk2.parameters.String(
                "Github owner",
                required=True
            )

            app_repo = sdk2.parameters.String(
                "Github repo",
                required=True
            )

            app_branch = sdk2.parameters.String(
                "Тестируемая ветка",
                required=True
            )

            git_commit = sdk2.parameters.String(
                "Commit hash",
                required=True
            )

        with sdk2.parameters.Group('Porchmark') as compare_block:
            porchmark_config_path = sdk2.parameters.String(
                "Porchmark config path",
                required=True,
            )

            test_mode = sdk2.parameters.String(
                "Test mode",
                required=True,
            )

            test_page = sdk2.parameters.String(
                "Test page from porchmark.conf.js",
                required=True,
            )

        with sdk2.parameters.Group('db') as db:
            db_host = sdk2.parameters.String(
                "DB host",
                default_value=DEFAULT_DB_HOST,
                required=True,
            )

            db_port = sdk2.parameters.String(
                "DB port",
                default_value=DEFAULT_DB_PORT,
                required=True,
            )

            db_database = sdk2.parameters.String(
                "DB database",
                default_value=DEFAULT_DB_DATABASE,
                required=True,
            )

            db_user = sdk2.parameters.String(
                "DB user",
                default_value=DEFAULT_DB_USER,
                required=True,
            )

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64
        disk_space = DISK_SPACE
        environments = (
            PipEnvironment('psycopg2-binary'),
        )

    def on_enqueue(self):
        super(MarketFrontPorchmarkSaveResultArc, self).on_enqueue()
        setup_container(self)

    def on_prepare(self):
        self._prepare_dirs()
        self._prepare_report_resource()

    def on_execute(self):
        with MetatronEnvArc(self, nodejs_version=self.Parameters.node_version):
            self._parse()
            self._save()

    def _prepare_dirs(self):
        self.root_dir = str(self.path())
        self.report_dir = os.path.join(self.root_dir, 'report')

    def _prepare_report_resource(self):
        resource_id = self.Parameters.report_resource_id
        report_resource = sdk2.Resource['MARKET_PORCHMARK_REPORT'].find(id=resource_id).first()
        self.Context.compare_task_id = report_resource.task_id

        if not report_resource:
            raise TaskFailure("no MARKET_PORCHMARK_REPORT resource with id {}".format(resource_id))

        resource_data = sdk2.ResourceData(report_resource)
        logging.debug('Copy resource, path: {} to {}'.format(resource_data.path, self.report_dir))
        rich_check_call(
            ['cp', '-R', '{}'.format(resource_data.path), self.report_dir],
            task=self, alias='copy reports', cwd=self.root_dir,
        )
        self.report_resource = report_resource

    def _get_project(self):
        porchmark_config_path = self.Parameters.porchmark_config_path
        try:
            project = porchmark_config_path.split("/")[0]

            if project not in ("market", "pokupki"):
                raise TaskFailure("can't get project from porchmark_config_path: {}".format(porchmark_config_path))

            return project
        except IndexError:
            raise TaskFailure("can't get project from porchmark_config_path: {}".format(porchmark_config_path))

    def _parse(self):
        logging.info("parse porchmark report")

        from sandbox.projects.market.frontarc.constants.porchmark \
            import JSON_REPORT_FILENAME, PORCHMARK_CONFIG_FILENAME

        report_filepath = "{}/{}".format(
            self.report_dir,
            JSON_REPORT_FILENAME
        )

        porchmark_config_filepath = "{}/{}".format(
            self.report_dir,
            PORCHMARK_CONFIG_FILENAME
        )

        info = {
            "app_owner": self.Parameters.app_owner,
            "app_repo": self.Parameters.app_repo,
            "project": self._get_project(),
            "app_branch": self.Parameters.app_branch,
            "commit": self.Parameters.git_commit,
            "test_mode": self.Parameters.test_mode,
            "test_page": self.Parameters.test_page,
            "sandbox_task_id": self.Context.compare_task_id,
        }

        self.metrics = parse(report_filepath, porchmark_config_filepath, info)

        logging.debug(pformat(self.metrics))

    def _save(self):
        logging.info("save metrics to database")

        import psycopg2
        from sandbox.projects.market.frontarc.MarketFrontPorchmarkSaveReportArc.save import save_metrics

        secrets = self._get_db_secrets()

        connection = psycopg2.connect(
            database=secrets["database"],
            user=secrets["user"],
            password=secrets["password"],
            host=secrets["host"],
            port=secrets["port"],
            target_session_attrs="read-write"
        )

        save_metrics(connection, self.metrics)

        connection.close()

    def _get_db_secrets(self):
        return {
            "host": self.Parameters.db_host,
            "port": self.Parameters.db_port,
            "database": self.Parameters.db_database,
            "user": self.Parameters.db_user,
            "password": sdk2.Vault.data(PASSWORD_VAULT_KEY),
        }
