# -*- coding: utf-8 -*-

import json
from string import Template

FORMAT_VERSION = 1.0


def deep_get(d, keys, default=None):
    """
    Example:
        d = {'meta': {'status': 'OK', 'status_code': 200}}
        deep_get(d, ['meta', 'status_code'])          # => 200
        deep_get(d, ['garbage', 'status_code'])       # => None
        deep_get(d, ['meta', 'garbage'], default='-') # => '-'
    """
    assert type(keys) is list
    if d is None:
        return default
    if not keys:
        return d
    return deep_get(d.get(keys[0]), keys[1:], default)


def create_blank_metrics():
    """
    Return empty metrics dict
    """
    return {
        # "id"
        "app_owner": None,
        "app_repo": None,
        "project": None,
        "app_branch": None,
        "commit": None,
        "test_mode": None,
        "test_page": None,
        "compare_id": None,
        "sandbox_task_id": None,
        "sample_url": None,
        "test_url": None,
        "porchmark_config": None,
        "porchmark_version": None,
        "report_version": None,
        "format_version": None,
        "created_at": None,
        "updated_at": None,
        "started_at": None,
        "completed_at": None,
        "status": None,
        "status_message": None,
        "extra": None,
        # metrics
        "request_start": None,
        "response_start": None,
        "response_end": None,
        "first_paint": None,
        "first_contentful_paint": None,
        "dom_content_loaded_event_end": None,
        "load_event_end": None,
        "dom_interactive": None,
        "dom_complete": None,
        "market_ttr": None,
        "market_tti": None,
        "node_count": None,
        "image_count": None,
        "script_count": None,
        "style_count": None,
        "transfer_size": None,
        "encoded_body_size": None,
        "decoded_body_size": None,
        # diffs
        "diff_request_start": None,
        "diff_response_start": None,
        "diff_response_end": None,
        "diff_first_paint": None,
        "diff_first_contentful_paint": None,
        "diff_dom_content_loaded_event_end": None,
        "diff_load_event_end": None,
        "diff_dom_interactive": None,
        "diff_dom_complete": None,
        "diff_market_ttr": None,
        "diff_market_tti": None,
        "diff_node_count": None,
        "diff_image_count": None,
        "diff_script_count": None,
        "diff_style_count": None,
        "diff_transfer_size": None,
        "diff_encoded_body_size": None,
        "diff_decoded_body_size": None,
    }


def parse_report_file(filepath, metrics):
    """
    get data from porchmark report file
    """
    with open(filepath) as f:
        report = json.load(f)
        f.close()

        # sample_url, test_url
        sites = deep_get(report, ["sites"], [])
        sample_site = list(filter(lambda i: i["name"] == "sample", sites))
        if len(sample_site) > 0:
            metrics["sample_url"] = sample_site[0]["url"]
        test_site = list(filter(lambda i: i["name"] == "test", sites))
        if len(test_site) > 0:
            metrics["test_url"] = test_site[0]["url"]

        metrics["porchmark_version"] = deep_get(report, ["version"])
        metrics["report_version"] = deep_get(report, ["reportVersion"])
        metrics["started_at"] = deep_get(report, ["startedAt"])
        metrics["completed_at"] = deep_get(report, ["completedAt"])
        metrics["status"] = deep_get(report, ["status"])
        metrics["status_message"] = deep_get(report, ["statusMessage"])

        # metrics
        metrics["request_start"] = deep_get(report, ["data", "metrics", "requestStart"])
        metrics["response_start"] = deep_get(report, ["data", "metrics", "responseStart"])
        metrics["response_end"] = deep_get(report, ["data", "metrics", "responseEnd"])
        metrics["first_paint"] = deep_get(report, ["data", "metrics", "first-paint"])
        metrics["first_contentful_paint"] = deep_get(report, ["data", "metrics", "first-contentful-paint"])
        metrics["dom_content_loaded_event_end"] = deep_get(report, ["data", "metrics", "domContentLoadedEventEnd"])
        metrics["load_event_end"] = deep_get(report, ["data", "metrics", "loadEventEnd"])
        metrics["dom_interactive"] = deep_get(report, ["data", "metrics", "domInteractive"])
        metrics["dom_complete"] = deep_get(report, ["data", "metrics", "domComplete"])
        metrics["market_ttr"] = deep_get(report, ["data", "metrics", "marketTTR"])
        metrics["market_tti"] = deep_get(report, ["data", "metrics", "marketTTI"])
        metrics["node_count"] = deep_get(report, ["data", "metrics", "nodeCount"])
        metrics["image_count"] = deep_get(report, ["data", "metrics", "imageCount"])
        metrics["script_count"] = deep_get(report, ["data", "metrics", "imageCount"])
        metrics["style_count"] = deep_get(report, ["data", "metrics", "styleCount"])
        metrics["transfer_size"] = deep_get(report, ["data", "metrics", "transferSize"])
        metrics["encoded_body_size"] = deep_get(report, ["data", "metrics", "encodedBodySize"])
        metrics["decoded_body_size"] = deep_get(report, ["data", "metrics", "decodedBodySize"])

        # diffs
        metrics["diff_request_start"] = deep_get(report, ["data", "diffs", "requestStart"])
        metrics["diff_response_start"] = deep_get(report, ["data", "diffs", "responseStart"])
        metrics["diff_response_end"] = deep_get(report, ["data", "diffs", "responseEnd"])
        metrics["diff_first_paint"] = deep_get(report, ["data", "diffs", "first-paint"])
        metrics["diff_first_contentful_paint"] = deep_get(report, ["data", "diffs", "first-contentful-paint"])
        metrics["diff_dom_content_loaded_event_end"] = deep_get(report, ["data", "diffs", "domContentLoadedEventEnd"])
        metrics["diff_load_event_end"] = deep_get(report, ["data", "diffs", "loadEventEnd"])
        metrics["diff_dom_interactive"] = deep_get(report, ["data", "diffs", "domInteractive"])
        metrics["diff_dom_complete"] = deep_get(report, ["data", "diffs", "domComplete"])
        metrics["diff_market_ttr"] = deep_get(report, ["data", "diffs", "marketTTR"])
        metrics["diff_market_tti"] = deep_get(report, ["data", "diffs", "marketTTI"])
        metrics["diff_node_count"] = deep_get(report, ["data", "diffs", "nodeCount"])
        metrics["diff_image_count"] = deep_get(report, ["data", "diffs", "imageCount"])
        metrics["diff_script_count"] = deep_get(report, ["data", "diffs", "imageCount"])
        metrics["diff_style_count"] = deep_get(report, ["data", "diffs", "styleCount"])
        metrics["diff_transfer_size"] = deep_get(report, ["data", "diffs", "transferSize"])
        metrics["diff_encoded_body_size"] = deep_get(report, ["data", "diffs", "encodedBodySize"])
        metrics["diff_decoded_body_size"] = deep_get(report, ["data", "diffs", "decodedBodySize"])

        return metrics


def read_porchmark_config(filepath):
    with open(filepath) as f:
        config = json.load(f)
        f.close()
        return config


def parse(report_filepath, porchmark_config_filepath, info):
    metrics = create_blank_metrics()

    metrics["app_owner"] = deep_get(info, ["app_owner"])
    metrics["app_repo"] = deep_get(info, ["app_repo"])
    metrics["project"] = deep_get(info, ["project"])
    metrics["app_branch"] = deep_get(info, ["app_branch"])
    metrics["commit"] = deep_get(info, ["commit"])
    metrics["test_mode"] = deep_get(info, ["test_mode"])
    metrics["test_page"] = deep_get(info, ["test_page"])
    metrics["compare_id"] = Template(
        "$app_owner-$app_repo-$app_branch-$project-$test_mode-$test_page"
    ).substitute(info)
    metrics["sandbox_task_id"] = deep_get(info, ["sandbox_task_id"])
    # sample_url, test_url
    metrics["porchmark_config"] = read_porchmark_config(porchmark_config_filepath)
    metrics["format_version"] = FORMAT_VERSION
    # created_at by DB
    # updated_at by DB
    metrics["extra"] = deep_get(info, ["extra"])

    metrics = parse_report_file(report_filepath, metrics)

    return metrics
