# coding=utf-8

import json
import logging
import os
import re

import sandbox.common.types.misc as ctm
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.projects.common import task_env
from sandbox.projects.market.frontarc.helpers.MetatronEnvArc import MetatronEnvArc
from sandbox.projects.market.frontarc.helpers.github import get_pr_info
from sandbox.projects.market.frontarc.helpers.tsum import restart_percommit
from sandbox.projects.market.frontarc.helpers.ubuntu import create_ubuntu_selector, setup_container
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sdk2 import Task

TEST_LAUNCH_ENABLED_FIELD = "61efccc566afaa33d5746047--vklucenAvtozapuskTestov"
YESS = "Да"
GITHUB_APPLICATION_ID = "ru.yandex.github.enterprise"

JOB_MODE = {
    "by_PR": "by_PR",
    "by_issue": "by_issue",
}

class MarketFrontReLaunchTsumPerCommitMTArc(Task):
    """
    Таска для запуска и перезапуска покоммитного пайплайна

    Берёт данные из Github по пулл-реквесту и рестартит соответствующий пайплайн в ЦУМе
    """
    st = None

    class Parameters(Task.Parameters):
        ubuntu_version = create_ubuntu_selector()

        with sdk2.parameters.Group('Первый метод запуска') as github_repo_block:
            # github_payload.pull_request.head.repo.owner.login
            app_owner = sdk2.parameters.String(
                'GitHub owner',
                description='Логин владельца репозитория или название организации',
                default='market',
                required=True,
            )
            # github_payload.pull_request.head.repo.name
            app_repo = sdk2.parameters.String(
                "Репозиторий",
                default='marketfront',
                required=True,
            )
            # github_payload.pull_request.head.ref
            app_branch = sdk2.parameters.String(
                "Ветка",
                default='master',
            )

            # github_payload.pull_request.number
            pull_request_number = sdk2.parameters.String(
                'Номер пулл-реквеста',
            )

        with sdk2.parameters.Group('Второй метод запуска') as issue_key_block:
            issue_key = sdk2.parameters.String(
                'Номер тикета',
                description='В формате MARKETFRONT-12345',
            )


    class Requirements(task_env.TinyRequirements):
        dns = ctm.DnsType.DNS64
        environments = [
            PipEnvironment("yandex_tracker_client", version="1.3", custom_parameters=["--upgrade-strategy only-if-needed"]),
            PipEnvironment("startrek_client", version="2.3.0", custom_parameters=["--upgrade-strategy only-if-needed"])
        ]

    def on_enqueue(self):
        super(MarketFrontReLaunchTsumPerCommitMTArc, self).on_enqueue()
        setup_container(self)

    def on_execute(self):
        super(MarketFrontReLaunchTsumPerCommitMTArc, self).on_execute()
        with MetatronEnvArc(self), self.memoize_stage.restart_mt(max_runs=1):
            from startrek_client import Startrek
            oauth_token = os.environ["ST_OAUTH_TOKEN"]
            self.st = Startrek(useragent="robot-metatron", token=oauth_token)

            mode = self.choose_mode()
            pr_info = self.get_pr_info(mode)

            tsum_project_id = "front-market-unified"
            tsum_pipeline_id = "per-commit"
            token = os.environ["TSUM_OAUTH_TOKEN"]
            issue_key = get_issue_key(pr_info["title"])

            st_issue = self.st.issues[issue_key]
            is_launch_needed_prev_value = st_issue[TEST_LAUNCH_ENABLED_FIELD]
            if is_launch_needed_prev_value != YESS:
                st_issue.update(**{TEST_LAUNCH_ENABLED_FIELD: YESS})

            response = restart_percommit(
                auth_token=token,
                project=tsum_project_id,
                git_owner=pr_info["head"]["repo"]["owner"]["login"],
                git_repo=pr_info["head"]["repo"]["name"],
                git_branch=pr_info["head"]["ref"],
                pipeline_id=tsum_pipeline_id,
                pull_request_id=pr_info["id"],
            )

            # Почему-то покоммитные пайплайны используют в названии среды не весь хэш.
            tsum_pipeline_name = ("m-" + pr_info["head"]["sha"])[:35]

            pipe_url = "https://tsum.yandex-team.ru/pipe/projects/{}/multitestings/environments/{}".format(tsum_project_id, tsum_pipeline_name)
            queue_url = "https://tsum.yandex-team.ru/pipe/projects/per_commit/{}".format(tsum_project_id)

            try:
                if response.status_code == 400 and "The PerCommit launch already in queue." in response.json()['errors']:
                    status_message = 'Пайплайн уже ожидает запуска. <a href="{queue_url}" target="_blank">Ссылка на очередь</a>.\n' \
                                     'Ответ ЦУМа: {status_code}\n{text}'
                    self.set_info(status_message.format(queue_url=queue_url, status_code=response.status_code, text=response.text))
                    return
            except KeyError:
                pass

            if response.status_code != 200:
                self.set_info("Не удалось перезапустить пайплайн. Ответ ЦУМа: {}\n{}".format(response.status_code, response.text))
                st_issue.update(**{TEST_LAUNCH_ENABLED_FIELD: is_launch_needed_prev_value})
                raise TaskFailure("Не удалось перезапустить пайплайн.")

            status_message = 'Автозапуск тестов включён.\n' \
                             'Среда с запуском тестов: <a href="{pipe_url}" target="_blank">{pipe_url}</a>\n\n' \
                             'Очередь покоммитных: <a href="{queue_url}" target="_blank">{queue_url}</a>:'.format(pipe_url=pipe_url, queue_url=queue_url)
            self.set_info(status_message, do_escape=False)

    def choose_mode(self):
        can_find_by_pr = not (
            is_none_or_empty(self.Parameters.app_branch)
            or is_none_or_empty(self.Parameters.pull_request_number)
        )

        can_find_by_issue = not (
            is_none_or_empty(self.Parameters.issue_key)
            or not get_issue_key(self.Parameters.issue_key, "Номер тикета должен быть в формате MARKETFRONT-12345")
        )

        msg = "Должен быть указан или номер пулл-реквеста и репозиторий, или номер тикета"
        assert can_find_by_pr or can_find_by_issue, msg

        if can_find_by_pr:
            return JOB_MODE["by_PR"]

        if can_find_by_issue:
            return JOB_MODE["by_issue"]

    def get_pr_info(self, mode):
        pr_info = None
        if mode == JOB_MODE["by_issue"]:
            issue = self.st.issues[self.Parameters.issue_key]
            # https://st-api.yandex-team.ru/v2/issues/MARKETFRONT-71700/remotelinks
            linked_prs = [link.object.key for link in issue.remotelinks if link.object.application.id == GITHUB_APPLICATION_ID]
            logging.debug("Нашли {} связанных пулл-реквестов: {}".format(len(linked_prs), linked_prs))

            expected_repo = "{}/{}".format(self.Parameters.app_owner, self.Parameters.app_repo)
            for pr in linked_prs:
                # splitting "pull:market/marketfront/10116"
                [owner, repo, number] = pr.split(":")[1].split("/")
                current_pr_info = get_pr_info(owner, repo, number)
                if current_pr_info["state"] == "open" and current_pr_info["head"]["repo"]["full_name"] == expected_repo:
                    assert pr_info is None, "В тикете должен быть только один открытый пулл-реквест в marketfront"
                    pr_info = current_pr_info

            assert pr_info is not None, "Не удалось получить связанный пулл-реквест в {}".format(expected_repo)

        if mode == JOB_MODE["by_PR"]:
            pr_info = get_pr_info(self.Parameters.app_owner, self.Parameters.app_repo, self.Parameters.pull_request_number)

        # Например, https://api.github.yandex-team.ru/repos/market/marketfront/pulls/10116
        # В pr_info все строки в unicode, их надо перекидывать в utf-8
        logging.debug("PR_INFO:")
        logging.debug(pr_info)
        return pr_info


def get_tsum_pipeline_resources(pr_info):
    # В pr_info все строки в unicode, их надо перекидывать в utf-8
    issue_key = get_issue_key(pr_info["title"])
    resource_dict = {
        # PerCommitLaunchParams
        '9c82a6c0-1bfb-4817-9b09-1494d2296ecf': json.dumps(
            {
                "commitSha": pr_info["head"]["sha"],
                "commitMessage": ' ',
                "startrekTicket": issue_key,
                "startrekTickets": [issue_key],
                "repository": pr_info["head"]["repo"]["full_name"],
                "branch": pr_info["head"]["ref"],
            },
            ensure_ascii=False, encoding='utf-8'
        ),
        # PerCommitPullRequestParams
        '420fecc9-c504-43d2-8cd0-b4b483c696e2': json.dumps(
            {
                "pullRequestId": pr_info["id"],  # type: int
                "number": pr_info["number"],  # type: int
                "title": pr_info["title"],
                "headBranch": pr_info["head"]["ref"],
                "baseBranch": pr_info["base"]["ref"],
                "mergeCommitSha": pr_info["merge_commit_sha"],
                "headCommitSha": pr_info["head"]["sha"],
                "baseCommitSha": pr_info["base"]["sha"],
            },
            ensure_ascii=False, encoding='utf-8'
        ),
        # FrontIsManualLaunchConfig
        '3c0f1b17-4c8f-4745-a8f8-66c984e92dcc': json.dumps(
            {
                "manual": True
            },
            ensure_ascii=False, encoding='utf-8'
        ),
    }

    logging.debug("TSUM_PIPELINE_RESOURCES:")
    logging.debug(resource_dict)

    return resource_dict


def get_issue_key(any_str, msg="Заголовок PR должен быть в формате `MARKETFRONT-12345: ...`"):
    issue_key = re.search("^[A-Z]+-\\d+", any_str.strip())
    assert issue_key is not None, msg
    return issue_key.group(0)


def is_none_or_empty(any_str):
    return (
        any_str is None
        or len(str(any_str).strip()) == 0
    )
