# -*- coding: utf-8 -*-

import logging
import os

from sandbox import sdk2
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
from sandbox.common.errors import TaskError
from sandbox.projects.market.frontarc.helpers.ubuntu import create_ubuntu_selector, setup_container
from sandbox.projects.browser.perf.record_wpr_archive import BrowserPerfRecordWprArchive
from sandbox.projects.market.frontarc.helpers.github import get_branch
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.common.errors import TaskFailure
from sandbox.projects.sandbox_ci.managers import MetaTaskManager
from sandbox.common.utils import get_task_link
from sandbox.projects.browser.perf.record_wpr_archive import WprArchive

KILL_TIMEOUT = 3 * 60 * 60  # 3 h
SUBTASK_TIMEOUT = 3 * 60 * 60  # 3 h
DISK_SPACE = 3 * 1024  # 3 Gb
GH_OWNER = 'MARKET'

class MarketFrontRecordWprArchiveArc(sdk2.Task):
    failure_links = None

    """
    Запуск таска сбора wpr-архива для бенчмарка фронтенда
    """
    class Context(sdk2.Context):
        record_config_template = ''
        snapshot_id = ''
        s3_key = ''
        child_tasks_ids = []
        wpr_archive_resource_id = None

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = KILL_TIMEOUT
        ubuntu_version = create_ubuntu_selector()

        record_config_template_url = sdk2.parameters.ArcadiaUrl(
            'Record config template arcadia url',
            description='params={host}, {market_platform}, {spec}',
            required=True
        )

        product = sdk2.parameters.String(
            'Yin product',
            description='для бенчмарка (web-market-desktop, web-market-touch ..., https://nda.ya.ru/3VnVGq)',
            required=True
        )

        market_product = sdk2.parameters.String(
            'Market Product',
            description='white | beru',
            required=True
        )

        market_platform = sdk2.parameters.String(
            'Market Platform',
            description='desktop | touch',
            required=True
        )

        host = sdk2.parameters.String(
            'Хост стенда',
            description='для которого будет собираться wpr',
            required=True
        )

        record_config_template = sdk2.parameters.String(
            'Record config template',
            description='Если не задан получаем из Аркадии по record_config_template_url',
            required=False,
            default='',
            multiline=True
        )

        with sdk2.parameters.Group('GitHub репозиторий проекта') as github_repo_block:
            app_owner = sdk2.parameters.String(
                'GitHub owner',
                description='Логин владельца репозитория или название организации',
                default=GH_OWNER,
                required=True
            )
            app_repo = sdk2.parameters.String(
                "Репозиторий",
                default='market',
                required=True
            )
            app_branch = sdk2.parameters.String(
                "Ветка",
                default='master',
                required=True
            )

        with sdk2.parameters.Group('Параметры таска BrowserPerfRecordWprArchive') as record_wpr_block:
            snapshot_id = sdk2.parameters.String(
                'Commit hash из репозитория приложения',
                description='если не задан получаем commit по app_branch',
                required=False,
                default=''
            )

            mds_upload = sdk2.parameters.Bool(
                'Upload wpr_archive to S3MDS',
                default=True
            )

            s3_bucket = sdk2.parameters.String(
                'S3-MDS bucket',
                default='market-perf-test'
            )

            s3_url = sdk2.parameters.String(
                'S3-MDS url',
                default='https://s3.mds.yandex.net'
            )

            s3_access_key_id = sdk2.parameters.String(
                'S3-MDS access key id',
                default='pgpexz07RagGedWltp3B'
            )

            vault_owner = sdk2.parameters.String(
                'S3-MDS secret key vault owner',
                default='MARKET'
            )

            vault_name = sdk2.parameters.String(
                'S3-MDS secret key vault name',
                default='robot-metatron-s3-secretkey'
            )

            proxy_url = sdk2.parameters.String(
                'WPR http-proxy url',
                default=''
            )

            proxy_auth_vault = sdk2.parameters.String(
                'WPR proxy vault with credentials',
                default=''
            )

            record_product = sdk2.parameters.String(
                'Browser that is used for recording wpr',
                default='chromium-snapshot'
            )

            spec = sdk2.parameters.String(
                'Specificator for record_product',
                description='hash коммита браузера для бенчмарка и записи wpr',
                default='4273c619662b03f6b5b71f94b30c52ab3267c2bf'
            )

            retry = sdk2.parameters.Integer(
                'Retry recording of failed pages. Skip retries by default',
                default=0
            )

            max_failures = sdk2.parameters.Integer(
                'The value of --max-failures arg to pass to record_wpr',
                default=None
            )

            branch = sdk2.parameters.String(
                'Yin branch',
                default_value='master',
                required=True
            )

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64
        disk_space = DISK_SPACE

    def on_enqueue(self):
        super(MarketFrontRecordWprArchiveArc, self).on_enqueue()
        setup_container(self)

    def _run_subtasks(self):
        with self.memoize_stage.subtasks_ex(max_runs=1):
            record_wpr_task = self._run_record_wpr_task()

            if record_wpr_task:
                self.Context.child_tasks_ids.append(record_wpr_task.id)

            raise sdk2.WaitTask(
                self.Context.child_tasks_ids,
                ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                wait_all=True,
                timeout=SUBTASK_TIMEOUT
            )

    def _run_record_wpr_task(self):
        params = {
            'record_config': self.Context.record_config,
            'snapshot_id': self.Context.snapshot_id,
            'mds_upload': self.Parameters.mds_upload,
            's3_key': self.Context.s3_key,
            's3_bucket': self.Parameters.s3_bucket,
            's3_url': self.Parameters.s3_url,
            's3_access_key_id': self.Parameters.s3_access_key_id,
            'vault_owner': self.Parameters.vault_owner,
            'vault_name': self.Parameters.vault_name,
            'proxy_url': self.Parameters.proxy_url,
            'proxy_auth_vault': self.Parameters.proxy_auth_vault,
            'record_product': self.Parameters.record_product,
            'spec': self.Parameters.spec,
            'retry': self.Parameters.retry,
            'max_failures': self.Parameters.max_failures,
            'branch': self.Parameters.branch,
        }
        logging.info('run record wpr task')
        logging.info(params)
        subtask = BrowserPerfRecordWprArchive(self, **params)
        subtask.enqueue()
        return subtask

    def _prepare_snapshot_id(self):
        if self.Parameters.snapshot_id:
            logging.info('get snapshot_id from parameters: {}'.format(self.Parameters.snapshot_id))
            self.Context.snapshot_id = self.Parameters.snapshot_id
            return

        try:
            logging.info('get snapshot_id from github branch info: {}'.format(self.Parameters.app_branch))
            app_branch_info = get_branch(self.Parameters.app_owner, self.Parameters.app_repo, self.Parameters.app_branch)
            commit_sha = app_branch_info['commit']['sha']
            logging.info('snapshot_id = {}'.format(commit_sha))
            self.Context.snapshot_id = commit_sha
        except Exception as e:
            raise TaskError(e)

    def _prepare_s3_key(self):
        self.Context.s3_key = '/{}/wpr/{}/{}/wpr_archive.tar'.format(
            self.Parameters.market_product,
            self.Parameters.market_platform,
            self.Context.snapshot_id
        )

    def _prepare_record_config(self):
        if self.Parameters.record_config_template:
            self.Context.record_config_template = self.Parameters.record_config_template
        else:
            record_config_path = self.path('record_config_template.yml')
            Arcadia.export(self.Parameters.record_config_template_url, record_config_path)

            with open(record_config_path.as_posix(), 'r') as f:
                self.Context.record_config_template = f.read()

        self.Context.record_config = self.Context.record_config_template.format(
            host=self.Parameters.host,
            platform=self.Parameters.market_platform,
            spec=self.Parameters.spec
        )

    def _fail_on_childs_fail(self):
        with self.memoize_stage.fail_on_childs_fail(max_runs=1):
            subtasks = list(
                sdk2.Task.find(
                    parent=self,
                    children=True,
                    id=self.Context.child_tasks_ids
                )
                .limit(1)
            )

            failed_tasks = MetaTaskManager.filter_failed(subtasks)

            if failed_tasks:
                failure_msg = 'Some of child tasks failed, see message below'
                failure_links = []

                for task in failed_tasks:
                    failure_links.append('<a href="{url}" target="_blank">{text}</a>'.format(
                        url=get_task_link(task.id),
                        text=task.type
                    ))

                self.failure_links = '\n'.join(failure_links)

                raise TaskFailure(failure_msg)

    def _process_result(self):
        with self.memoize_stage.process_result(max_runs=1):
            subtasks = list(
                sdk2.Task.find(
                    parent=self,
                    children=True,
                    id=self.Context.child_tasks_ids
                ).limit(1)
            )

            if len(subtasks) == 0:
                raise TaskFailure('No child tasks')

            subtask = subtasks[0]

            wpr_archive_resources = list(
                sdk2.Resource.find(
                    type=WprArchive,
                    task=subtask
                ).limit(1)
            )

            if len(wpr_archive_resources) == 0:
                raise TaskFailure('No resource WprArchive in child task')

            wpr_archive_resource = wpr_archive_resources[0]

            self.Context.wpr_archive_resource_id = wpr_archive_resource.id

    def on_prepare(self):
        self._prepare_snapshot_id()
        self._prepare_s3_key()
        self._prepare_record_config()

    def on_execute(self):
        self._run_subtasks()
        self._fail_on_childs_fail()
        self._process_result()

    def on_failure(self, prev_status):
        if self.failure_links:
            self.set_info(self.failure_links, do_escape=False)
