# coding=utf-8

import logging
import json

from sandbox import sdk2

from sandbox.common.errors import ResourceNotFound
from sandbox.common.types import misc as ctm
from sandbox.common.types import resource as ctr
from sandbox.common.types import task as ctt
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.common import task_env
from sandbox.projects.market.frontarc.helpers.MetatronEnvArc import MetatronEnvArc
from sandbox.projects.market.frontarc.helpers.sandbox_helpers import rich_check_call, unpack_resource
from sandbox.projects.market.frontarc.helpers.startrack import ReleaseTicket
from sandbox.projects.market.frontarc.MarketFrontSimpleBashSaveHtmlReportArc\
    import MarketFrontSimpleBashSaveHtmlReportArc

DISK_SPACE = 3 * 1024  # 3 Gb

COMMENT_FORMAT = """
Статистика прогона ({context}):
#|
||Статус|Отчёт|Таск||
||{status}|{report_url}|{task_url}||
|#
<!-- <# consolidated-report-manager-short-report-regex #>||{report_url}|{task_url}|| -->
<# <!-- Screenshot-i18n-test --> #>
<!--<# <div id="consolidated-reports-comment" /> #>-->
"""

EXTENDED_COMMENT_FORMAT = """
Статистика прогона ({context}):
#|
||Статус|Passed|Failed|New||
||{status} {report_url}|{passed}|{failed}|{new}||
|#
- {report_url}
- {task_url}
<!-- <# consolidated-report-manager-short-report-regex #>||{status} {report_url}|{passed}|{failed}|{new}|| -->
<# <!-- Screenshot-i18n-test --> #>
<!--<# <div id="consolidated-reports-comment" /> #>-->
"""


class MarketFrontScreenshotTestArc(MarketFrontSimpleBashSaveHtmlReportArc):
    """
    Прогон jest-puppeteer скриншот тестов
    """

    TOOL = 'jest-puppeteer'

    class Parameters(MarketFrontSimpleBashSaveHtmlReportArc.Parameters):
        testpalm_run_issue = ReleaseTicket(
            'Релизный тикет',
            description='Пример: "MARKETVERSTKA-12345"'
        )

    class Requirements(task_env.BuildRequirements):
        dns = ctm.DnsType.DNS64
        disk_space = DISK_SPACE
        environments = [
            PipEnvironment('yandex_tracker_client', version="1.3", custom_parameters=["--upgrade-strategy only-if-needed"]),
            PipEnvironment('startrek_client', version="2.3.0", custom_parameters=["--upgrade-strategy only-if-needed"])
        ]

    @staticmethod
    def find_existing_screen_pack(host):
        return sdk2.Resource["MARKET_SCREENSHOTS_PACK"].find(
            host=host,
            state=ctr.State.READY,
            attrs={"tool": "jest-puppeteer"}
        ).first()

    def _send_startrek_report(self, status):
        oauth_token = sdk2.Vault.data('robot-metatron-st-token')

        from startrek_client import Startrek

        st = Startrek(useragent='robot-metatron', token=oauth_token)
        issue = st.issues[self.Parameters.testpalm_run_issue]

        http_report_url = self.report_url
        stat_reporter_path = '{}/stat-reporter.json'.format(self.APP_SRC_PATH)
        task_url = "((https://sandbox.yandex-team.ru/task/{}/view Таск))".format(self.id)
        report_url = "(({} Отчёт))".format(http_report_url) if http_report_url else ''

        try:
            f = open(stat_reporter_path, 'r')
            data = json.load(f)

            total = data['total']
            error = data['error']
            new = data['newTest']
            success = data['success']

            if success == total or success + new == total:
                report_status = '!!(green)PASSED!!'
            else:
                report_status = '!!(red)FAILED!!'

            comment = EXTENDED_COMMENT_FORMAT.format(
                context='Screenshot i18n',
                status=report_status,
                report_url=report_url,
                task_url=task_url,
                passed='!!(green){}!!'.format(success),
                failed='!!(red){}!!'.format(error),
                new='!!(blue){}!!'.format(new),
            )

        except OSError:
            if status == ctt.Status.FAILURE:
                report_status = '!!(red)FAILED!!'
            elif status == ctt.Status.SUCCESS:
                report_status = '!!(green)PASSED!!'
            else:
                report_status = '!!(orange)PENDING!!'

            comment = COMMENT_FORMAT.format(context='Screenshot i18n', status=report_status, report_url=report_url, task_url=task_url)

        issue.comments.create(text=comment, params={'notify': False})

        logging.debug('Statistic has been sent to {}'.format(self.Parameters.testpalm_run_issue))

    def on_prepare(self):
        super(MarketFrontScreenshotTestArc, self).on_prepare()

        with MetatronEnvArc(self, nodejs_version=self.Parameters.node_version):
            screen_pack = MarketFrontScreenshotTestArc.find_existing_screen_pack(self.Parameters.screen_pack_url)

            if screen_pack:
                logging.debug('Using screenshot pack with ID: {}'.format(str(screen_pack.id)))
            else:
                raise ResourceNotFound('Screenshot pack not found. '
                                       'Please set the correct resource "screenshots-pack" or '
                                       'rebuild the screenshots pack for your host '
                                       '(by using SandboxCiMarketFrontScreenshotTestUpdate task)')

            unpack_resource(self, screen_pack, self.APP_SRC_PATH)

    def _run_build(self):
        status = ctt.Status.SUCCESS

        try:
            logging.debug('Running {} tests on host: {}'.format(self.TOOL, self.Parameters.screen_pack_url))
            rich_check_call(
                ["make", "screenshot"],
                self, "test_screenshot_command", cwd=self.APP_SRC_PATH, env=self.check_env
            )
        except Exception as e:
            logging.debug('Executing exception: {}'.format(str(e)))
            status = ctt.Status.FAILURE
            raise
        finally:
            self._get_report(
                resource_path='html_reports',
                type='{}-report'.format(self.TOOL),
                status=status,
                root_path='index.html'
            )
            if self.Parameters.testpalm_run_issue:
                self._send_startrek_report(status)

