import json
import random
import logging
from datetime import date

from sandbox import sdk2
import sandbox.common.errors as ce
import sandbox.common.types.task as ctt
import sandbox.common.types.resource as ctr
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.tank.load_resources.resources import AMMO_FILE
from sandbox.projects.tank.load_resources.resources import STRESS_SESSION_IDS
from sandbox.projects.tank.LoadTestResults import LoadTestResults
from sandbox.projects.tank.ShootViaTankapi.detect_tank import TankFinder
from sandbox.projects.tank.ShootViaTankapi import ShootViaTankapi, link_format
from sandbox.projects.tank.GenerateStressSessionIds import GenerateStressSessionIds
from sandbox.projects.market.frontarc.MarketFrontShooterArc.generate_ammo import MarketFrontAmmoGeneratorArc
from sandbox.projects.market.frontarc.MarketFrontShooterArc.calculate_sla_message import CalculateMessageArc

QUERY = '''SELECT {fields} FROM nginx2
        WHERE date=yesterday()
         AND timestamp > (toUInt32(now()) - (60 * 60 * 48))
         AND (http_code in (200,301,302,304,307))
         AND (http_method = 'GET')
         AND vhost LIKE '{vhost}%'
        LIMIT {records_num}'''


class MarketFrontShooterArc(sdk2.Task):
    """
    Task for load tests on market front
    """

    ammo_generator = MarketFrontAmmoGeneratorArc({})
    session_cookies = []

    class Requirements(sdk2.Requirements):
        disk_space = 1024 * 5
        environments = (PipEnvironment('clickhouse_driver'),)

    class Context(sdk2.Context):
        cookies_subtask = ''
        tanks = []
        target_host = ''
        ammo_resource_id = None
        sla_json = None

    class Parameters(sdk2.Parameters):
        kill_timeout = 60 * 30

        dry_run = sdk2.parameters.Bool(
            'Dry run', description='Only ammo and config will be generated, no shooting', default=False)
        with sdk2.parameters.Group('Ammo parameters') as ammo_block:
            generate_fresh_ammo = sdk2.parameters.Bool('Generate fresh ammo?')

            vhost = sdk2.parameters.String(
                'Vhost', description='Virtual host in ammo (header Host). Jenkins parameter: VHOST')
            log_vhost = sdk2.parameters.String(
                'Log vhost', description='Filter logs by this vhost. Jenkins parameter: log_vhost')
            x_real_ip = sdk2.parameters.Bool(
                'Use X-Real-IPs from logs?', description='Jenkins parameter: X_Real_IP', default=True)
            ignore_filters = sdk2.parameters.Bool(
                'Ignore log filters for static files?', description='Jenkins parameter: ignore_filters',
                default=False)
            # log_format = sdk2.parameters.String('Log format, seems not to be used', default='TskvLog')
            with sdk2.parameters.Group('Cookies in ammo') as cookies_block:
                generate_stress_cookies = sdk2.parameters.Bool(
                    'Generate fresh stress session cookies?',
                    description='Fresh cookies will be generated from list of uids'
                )
                with generate_stress_cookies.value[True]:
                    uids = sdk2.parameters.List('UIDs for stress cookies generation',
                                                description='Jenkins parameter: UIDs',
                                                default=['67282295', '28128938', '234710816'])
                    uids_list_resource = sdk2.parameters.Resource(
                        'Or resource with list of uids separated by newline')
                    tvm_client_id = sdk2.parameters.String(
                        'Client tvm2 id for cookies generation', default='2000158')
                    tvm_secret_token = sdk2.parameters.String('Vault name with tvm2 secret for cookies generation',
                                                              default='robot-lunapark-tvm2-stressids-secret')
                    tvm_vault_owner = sdk2.parameters.String(
                        'Owner of vault item with tvm2 secret', default='LOAD')
                with generate_stress_cookies.value[False]:
                    stress_cookies_resource = sdk2.parameters.Resource(
                        'List of cookies', resource_type=STRESS_SESSION_IDS)

            with generate_fresh_ammo.value[False]:
                ammofile = sdk2.parameters.String(
                    'Ammofile link',
                    description='Link to ammofile, should be accessible from tank. Jenkins parameter: ammofile')
        with sdk2.parameters.Group('Shooting parameters') as shoot_block:
            task = sdk2.parameters.String('Task in startrek',
                                          description='Jenkins parameter: TASK', required=True, default='LOAD-272')
            with sdk2.parameters.Dict(
                'RPS schedule and type of shooting',
                description='Load schedule: define rps_schedule and test type.\n '
                            'You can leave values "default_imbalance" and "default_const" for rps_schedule, '
                            'in this case the scheme defined in config file will be used.\n '
                            'Jenkins parameters: TEST_TYPE and RPS_SCHEDULE',
                default={'default_imbalance': 'imbalance', 'default_const': 'const'}
            ) as load_schedule:
                load_schedule.values['rps_schedule'] = ''
                load_schedule.values.imbalance = 'imbalance'
                load_schedule.values.const = 'const'
            autostop = sdk2.parameters.String(
                'Autostop', description='Autostop criteria. Jenkins parameter: AUTOSTOP',
                default='http(4xx,30%,20) http(50x,10%,10) http(51x,10%,10) time(3000,15s) net(xx,10%,10)')
            nanny_service = sdk2.parameters.String(
                'Nanny service',
                description='Nanny service name with testing instances. One of these instances will be used as target. '
                            'Jenkins parameter: nanny_service'
            )
        with sdk2.parameters.Group('Report parameters') as report_block:
            st_token_name = sdk2.parameters.String(
                'Vault record name with startrek token',
                description='Vault name of record with startrek token',
                default='lunapark-startrek-token')
            check_sla = sdk2.parameters.Bool('Check SLA?', description='Jenkins parameter: CHECK_SLA')
            with check_sla.value[True]:
                sla_resource = sdk2.parameters.Resource('Json with sla')

        with sdk2.parameters.Output:
            lunapark_job_ids = sdk2.parameters.String('Lunapark job ids (CSV)', default_value='')

    def on_create(self):
        self.Parameters.sla_resource = sdk2.Resource.find(id=576356012).first()

    def generate_stress_session_ids(self):
        logging.info('Start generation of stress sessions ids')
        uids = ''
        if self.Parameters.uids:
            uids = '\n'.join(self.Parameters.uids)
        elif self.Parameters.uids_list_resource:
            resource_path = sdk2.ResourceData(self.Parameters.uids_list_resource).path
            with open(str(resource_path), 'r') as uids_list:
                uids = uids_list.read()

        if uids:
            subtask = GenerateStressSessionIds(
                self,
                description='Generate stress session ids for task#{}'.format(self.id),
                uids=uids,
                tvm_client_id=self.Parameters.tvm_client_id,
                tvm_secret_token=self.Parameters.tvm_secret_token,
                vault_owner=self.Parameters.tvm_vault_owner,
                result_label='market_frontend_shooting_{}'.format(self.id)
            ).enqueue()
            self.Context.cookies_subtask = subtask.id
            raise sdk2.WaitTask([subtask], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK)

    def make_stress_cookies(self):
        logging.info('Start making stress cookies')
        subtask = self.find().first()
        previous_cookies = None
        try:
            previous_cookies = STRESS_SESSION_IDS.find(
                task=subtask,
                state=ctr.State.READY,
                attrs=dict(label='market_frontend_shooting_common'),
            ).order(-sdk2.Resource.id).first()
        except AttributeError:
            logging.info('Previous cookies not found')
        resource = self.Parameters.stress_cookies_resource or previous_cookies

        if not resource:
            logging.warning('Unable to make stress passport sessions, so we continue with empty cookies')
        else:
            logging.info('Resource %s is found', resource.id)
            resource_path = sdk2.ResourceData(resource).path
            with open(str(resource_path), 'r') as cookies:
                for line in cookies:
                    try:
                        session_data = json.loads(line)
                        self.session_cookies.append({
                            'session': session_data['new-session']['value'],
                            'ssl_session': session_data['new-sslsession']['value']
                        })
                    except ValueError:
                        pass

    def get_data_from_clickhouse(self):
        logging.info('Start fetching data from clickhouse')

        passwd = sdk2.Vault.data('lunapark-clickhouse-health-token')

        log_fields = 'vhost,client_ip,http_method,url,http_code,user_agent,page_id'
        records_num = 30000
        if not self.Parameters.log_vhost:
            url_type = 'mobile' if 'touch' in self.Parameters.vhost else 'desktop'
            log_vhost = 'market.yandex' if url_type == 'desktop' else 'm.market.yandex'
        elif self.Parameters.log_vhost.startswith('market.yandex.'):
            log_vhost = 'market.yandex'
        elif self.Parameters.log_vhost.startswith('m.market.yandex.'):
            log_vhost = 'm.market.yandex'
        else:
            log_vhost = self.Parameters.log_vhost
        query = QUERY.format(fields=log_fields, vhost=log_vhost, records_num=records_num)

        from clickhouse_driver import Client
        clickhouse_client = Client(
            'health-house.market.yandex.net',
            user='lunapark',
            password=passwd,
            database='market'
        )
        return clickhouse_client.execute(query)

    def parse_clickhouse_results(self):
        try:
            for line in self.get_data_from_clickhouse():
                yield line
        except Exception as exc:
            logging.error('Impossible to get data from clickhouse: %s', exc)

    def generate_ammo(self, label):

        sku_resource = sdk2.Resource['AMMO_FILE'].find(
            state=ctr.State.READY,
            attrs=dict(ammo_label='market_frontend_sku_list')
        ).order(-sdk2.Resource.id).first()
        sku_path = sdk2.ResourceData(sku_resource).path
        with open(str(sku_path), 'r') as sku_file:
            sku_list = sku_file.readlines()
        logging.info('Sku list taken from resource %s, starting skus from %s',
                     sku_resource.id, sku_list[0])

        self.ammo_generator.params = {
            'vhost': self.Parameters.vhost,
            'x_real_ip': self.Parameters.x_real_ip,
            'sku_list': sku_list
        }
        ammofile = '{}/ammo'.format(self.path())

        output_file = open(ammofile, 'w')
        try:
            for line in self.parse_clickhouse_results():
                try:
                    self.ammo_generator.log_line = line
                    logging.debug('Create ammo for log line %s', line)
                    self.ammo_generator.prepare_case_header()
                    self.ammo_generator.prepare_log_headers(self.session_cookies)

                    single_req = self.ammo_generator.make_ammo()
                    if single_req and (self.Parameters.ignore_filters or self.ammo_generator.junk_filter(line)):
                        output_file.write(single_req)
                except UnicodeEncodeError:
                    logging.warning('Failed on log line %s', line)
        finally:
            output_file.close()

        resource = AMMO_FILE(self, 'Ammofile for Market front', ammofile)
        resource.ammo_label = label
        sdk2.ResourceData(resource).ready()
        return resource.id

    def define_target(self):
        target = ''
        tank_finder = TankFinder(
            target_nanny_service=self.Parameters.nanny_service,
            target_host=target,
            gencfg_group='SAS_MARKET_TANKS'
        )
        tank_finder.collect_info()
        try:
            random_target = random.choice(tank_finder.targets)
        except IndexError:
            raise ce.TaskError('Target from %s is not found', self.Parameters.nanny_service)
        target_host = '{}:{}'.format(random_target.host, random_target.port)
        tanks = ['{}:{}'.format(tank.host, tank.port) for tank in tank_finder.tanks]
        return target_host, tanks

    def run_shooting_task(self, config_name, rps_schedule):
        cmd_params = ' -o '.join([
            '-o phantom.address={}'.format(self.Context.target_host),
            'meta.jenkinsbuild=https://sandbox.yandex-team.ru/task/{}'.format(self.id),
            'meta.task={}'.format(self.Parameters.task),
            'autostop.autostop="{}"'.format(self.Parameters.autostop),
        ])
        if 'default' not in rps_schedule:
            cmd_params += ' -o phantom.rps_schedule="{}"'.format(rps_schedule)

        shoot_params = {
            'description': 'Started by task #{}'.format(self.id),
            'config_source': 'arcadia',
            'config_arc_path': 'sandbox/projects/market/frontarc/MarketFrontShooter/configs/{}'.format(config_name),
            'config_add_parameters': cmd_params,
            'use_monitoring': True,
            'monitoring_source': 'arcadia',
            'monitoring_arc_path': 'sandbox/projects/market/frontarc/MarketFrontShooter/configs/monitoring.xml',
            'ammo_source': 'resource',
            'ammo_resource': self.Context.ammo_resource_id,
            'tanks': self.Context.tanks
        }
        self.Context.current_shooting_params = shoot_params

        if self.Parameters.ammofile:
            shoot_params['ammo_source'] = 'url'
            shoot_params['ammo_source_url'] = self.Parameters.ammofile
            del shoot_params['ammo_resource']

        logging.info('Shooting params are: %s', shoot_params)

        if not self.Parameters.dry_run:
            logging.info('Start shooting')
            shoot_subtask = ShootViaTankapi(
                self,
                **shoot_params
            ).enqueue()
            raise sdk2.WaitTask([shoot_subtask], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK)

    def report_results(self):
        shoot_subtasks = self.find(ShootViaTankapi, status=ctt.Status.Group.FINISH)
        report_subtasks = []
        job_ids = []

        for task in shoot_subtasks:
            job_ids.append(task.Parameters.lunapark_job_id)
            lunapark_link = task.Parameters.lunapark_link

            if 'const' in task.Parameters.config_arc_path:
                load_type = 'const'
                report_type = 'simple_const_link'
            else:
                load_type = 'line'
                report_type = 'simple_line_link'

            if lunapark_link:
                message = 'Sandbox shooting task: https://sandbox.yandex-team.ru/task/{}\n'.format(self.id)

                info = link_format(lunapark_link, lunapark_link, description='Lunapark link: ')
                self.set_info(info, do_escape=False)

                if 'const' in load_type and self.Context.sla_json and self.Parameters.check_sla:
                    try:
                        sla_comparison = CalculateMessageArc(lunapark_link.split('/')[-1], self.Context.sla_json)
                        sla_comparison.create_message()
                        message += sla_comparison.message
                    except RuntimeError as exc:
                        logging.warning('Sla calculation failed because of %s', exc)
                report_task = LoadTestResults(
                    self,
                    description='Report for shooting {}'.format(lunapark_link),
                    shoot_id=lunapark_link.split('/')[-1],
                    report_type=report_type,
                    send_comment=True,
                    ticket_id=self.Parameters.task,
                    st_token_name=self.Parameters.st_token_name,
                    additional_message=message
                ).enqueue()
                report_subtasks.append(report_task)

        self.Parameters.lunapark_job_ids = ','.join(job_ids)

        if report_subtasks:
            raise sdk2.WaitTask(report_subtasks, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK)

    def _check_subtasks_status(self):
        sub_tasks = self.find()
        task_errors = ''
        for task in sub_tasks:
            if task.status not in ctt.Status.Group.SUCCEED:
                task_errors += 'Subtask {} {} failed with status {}\n'.format(task.type, task.id, task.status)
        if task_errors:
            raise ce.TaskFailure(task_errors)

    def on_execute(self):

        if self.Parameters.generate_stress_cookies:
            if not self.Context.cookies_subtask:
                self.generate_stress_session_ids()
        if not self.session_cookies:
            self.make_stress_cookies()

        with self.memoize_stage.generate_ammo:
            ammo_label = '{}_{}'.format(self.Parameters.vhost, str(date.today()))
            try:
                self.Context.ammo_resource_id = AMMO_FILE.find(
                    state=ctr.State.READY,
                    attrs=dict(ammo_label=ammo_label),
                ).order(-sdk2.Resource.id).first().id
                logging.info('Resource %s with ammo is found', self.Context.ammo_resource_id)
            except AttributeError:
                logging.info('Resource with ammo not found')
            if self.Parameters.generate_fresh_ammo or not self.Context.ammo_resource_id:
                self.Context.ammo_resource_id = self.generate_ammo(ammo_label)

        with self.memoize_stage.setup_shooting:
            self.Context.target_host, self.Context.tanks = self.define_target()
            self.Context.shoots_left = self.Parameters.load_schedule

        if self.Context.shoots_left:
            load_scheme, load_type = self.Context.shoots_left.popitem()
            self.run_shooting_task('load_{}.yaml'.format(load_type), load_scheme)

        with self.memoize_stage.report_results:
            if self.Parameters.check_sla and not self.Context.sla_json:
                sla_resource_path = sdk2.ResourceData(self.Parameters.sla_resource).path
                with open(str(sla_resource_path), 'r') as sla_file:
                    sla = sla_file.read()
                    self.Context.sla_json = json.loads(sla)
            self.report_results()

        self._check_subtasks_status()
