# coding=utf-8

import os
import logging

from sandbox import sdk2

from sandbox.projects.market.front.resources.yammy_resources import MarketFrontYammyBuildPackageArtefact
from sandbox.projects.market.frontarc.MarketFrontYammyBaseArc import MarketFrontYammyBaseArc
from sandbox.projects.market.frontarc.helpers.sandbox_helpers import rich_check_call, get_resource_http_proxy_link
from sandbox.projects.market.frontarc.helpers.yammy.requirements import YammyBigRequirementsArc


class MarketFrontYammyBuildPackageArc(MarketFrontYammyBaseArc):
    """
    Таск сборки пакетов
    """

    class Requirements(YammyBigRequirementsArc):
        pass

    class Parameters(MarketFrontYammyBaseArc.Parameters):
        with MarketFrontYammyBaseArc.Parameters.yammy() as yammy:
            yammy_build_meta = MarketFrontYammyBaseArc.Parameters.yammy_build_meta(required=True)
            yammy_prebuilt = MarketFrontYammyBaseArc.Parameters.yammy_prebuilt(required=True)

            yammy_packages = sdk2.parameters.List(
                "Название пакетов для сборки в порядке их сборки",
                value_type=sdk2.parameters.String
            )

            yammy_wait_builds = sdk2.parameters.Dict(
                "Список тасков для ожидания сборки",
                value_type=sdk2.parameters.Integer
            )

            yammy_wait_artefacts = sdk2.parameters.Dict(
                "Список артефактов для установки из тасков (comma-separated)",
                value_type=sdk2.parameters.String
            )

        with sdk2.parameters.Output:
            build_results = sdk2.parameters.Dict(
                "Результаты сборки пакетов",
                value_type=sdk2.parameters.Integer
            )

    def setup_env(self):
        super(MarketFrontYammyBuildPackageArc, self).setup_env()
        os.environ['YENV'] = 'production'

    def on_enqueue(self):
        super(MarketFrontYammyBuildPackageArc, self).on_enqueue()
        tags = [pkg_name.split("#")[0].replace(':', '-').replace('@', '').replace('/', '_') for pkg_name in
                self.Parameters.yammy_packages]
        self.prepend_tags('build', *tags)

    def _install_prebuilt_packages(self):
        logging.debug('Installing prebuilt packages')

        for task_name in self.Parameters.yammy_wait_artefacts.keys():
            task = sdk2.Task[self.Parameters.yammy_wait_builds[task_name]]
            task_artifacts = self.Parameters.yammy_wait_artefacts[task_name].split(',')

            for artifact_name in task_artifacts:
                artifact_id = task.Parameters.build_results[artifact_name]
                self.install_package_artefact(artifact_name, artifact_id)

    def bootstrap(self):
        if not self.Context.tasks_ready:
            self.wait_tasks(self.Parameters.yammy_wait_builds.values(), title="dependencies")

        self.Context.tasks_ready = True

        super(MarketFrontYammyBuildPackageArc, self).bootstrap()

        self._install_prebuilt_packages()

    def run_task(self):
        results = dict()

        for pkg_name in self.Parameters.yammy_packages:
            results[pkg_name] = self._build_package(pkg_name)

        with self.timer("result"):
            self.Parameters.build_results = results

    def _build_package(self, pkg_build_name):
        logging.info("Running build {}".format(pkg_build_name))

        safe_pkg_name = pkg_build_name.replace('/', '__').replace('#', '--').replace(':', '-')

        chunks = pkg_build_name.split("#")
        pkg_name = chunks[0]
        build_name = "build" if len(chunks) < 2 else chunks[1]

        with self.timer("{}:yammy:build".format(':'.join(chunks))):
            rich_check_call(
                ["yammy", "ci", "build"] + chunks,
                self, "build.{}".format(safe_pkg_name), cwd=self.APP_SRC_DIR
            )

        logging.info("Publishing build artefact for {}".format(pkg_build_name))

        params = dict(
            package=pkg_name,
            commit=self.build_meta["packages"][pkg_name]["tip"]
        )

        if len(chunks) > 1:
            params['build'] = build_name

        res = MarketFrontYammyBuildPackageArtefact(
            self, "Build artefact for {}, build {}".format(pkg_name, build_name),
            self.publish_path(safe_pkg_name, "pkg-{}.tar.gz".format(build_name.replace(":", "-"))),
            **params
        )

        data = sdk2.ResourceData(res)

        with self.timer("{}:yammy:pack-pkg".format(':'.join(chunks))):
            rich_check_call(
                ["yammy", "ci", "pack-pkg", pkg_name, str(data.path)],
                self, "build.{}.pack".format(safe_pkg_name), cwd=self.APP_SRC_DIR
            )

        with self.timer("{}:ready".format(':'.join(chunks))):
            data.ready()

        self.set_info(
            ' <span class="status status_assigned"'
            ' style="font-size: 12px; text-transform: none;vertical-align: text-top;">{}</span>'
            ' <span class="status"'
            'style="font-size: 12px; text-transform: none;vertical-align: text-top;">{}</span>'
            ' <a class="status status_draft" href="{}">build log</a>'
            ' <a class="status status_draft" href="{}">pack log</a>'
            ' <a class="status status_success" href="//{}/resource/{}/view">resource:{}</a>'.format(
                pkg_name, build_name,
                "{}/build.{}.out.log".format(get_resource_http_proxy_link(self.log_resource), safe_pkg_name),
                "{}/build.{}.pack.out.log".format(get_resource_http_proxy_link(self.log_resource), safe_pkg_name),
                self.server.host,
                res.id,
                res.id,
            ), do_escape=False)

        return res.id
