# coding=utf-8
import json
import logging
import time
import os

from sandbox import sdk2

from sandbox.sandboxsdk.environments import PipEnvironment

from sandbox.projects.market.frontarc.helpers.MetatronEnvArc import MetatronEnvArc
from sandbox.projects.market.frontarc.helpers.sandbox_helpers import rich_get_output, rich_check_call
from sandbox.projects.market.frontarc.helpers.yammy.requirements import YammySmallRequirementsArc
from sandbox.projects.market.frontarc.MarketFrontYammyBaseArc import MarketFrontYammyBaseArc
from sandbox.projects.market.frontarc.MarketFrontEnvyArc import MarketFrontEnvyArc
from sandbox.projects.market.resources import MARKET_FRONT_ENVY_SUMMARY


class MarketFrontYammyEnvyArc(MarketFrontYammyBaseArc):
    """
    Таск обновления пакетов
    """

    _startrek = None

    class Requirements(YammySmallRequirementsArc):
        environments = [
            PipEnvironment("yandex_tracker_client", version="1.3", custom_parameters=["--upgrade-strategy only-if-needed"]),
            PipEnvironment("startrek_client", version="2.3.0", custom_parameters=["--upgrade-strategy only-if-needed"])
        ]

    @property
    def startrek(self):
        if not self._startrek:
            from startrek_client import Startrek

            oauth_token = sdk2.Vault.data("robot-metatron-st-token")
            self._startrek = Startrek(useragent="robot-metatron", token=oauth_token)

        return self._startrek

    def on_enqueue(self):
        super(MarketFrontYammyEnvyArc, self).on_enqueue()
        self.prepend_tags('envy')

    def _create_ticket(self, version):
        summary = 'mandrel@{} release'.format(version)

        logging.info("Creating head ticket: {}".format(summary))

        ticket = self.startrek.issues.create(
            queue='MARKETFRONTECH',
            summary=summary,
            tags=['платформа_фронта', 'mandrel_release']
        )

        self.set_info(
            "Created ticket <a href='https://st.yandex-team.ru/{}'>{}: {}</a>".format(ticket.key, ticket.key, summary),
            do_escape=False
        )

        return ticket

    def _run_envy(self):
        with MetatronEnvArc(task=self, nodejs_version=self.Parameters.node_version):
            root_dir = os.path.join(self.arcadia, 'market/front/monomarket')

            packages = json.loads(rich_get_output(
                ["yammy", "version", "--json"],
                self, "yammy.version.log", cwd=self.APP_SRC_DIR,
            ).strip())

            # У ЦУМа левый сертификат
            os.environ["NODE_TLS_REJECT_UNAUTHORIZED"] = "0"

            summary_res = MARKET_FRONT_ENVY_SUMMARY(self, "Envy summary", "summary.json")
            data = sdk2.ResourceData(summary_res)

            params = [
                "-t", self._create_ticket(packages["@yandex-market/mandrel"]).key,
                "--contour", "platform",
                "--component", "@Common",
                "--summary", str(data.path),
            ]

            for (package, ver) in packages.items():
                params.append('-i')
                params.append('{}@{}'.format(package, ver))

            rich_check_call(
                command=["yammy", "run", "envy", "update", '--'] + params,
                task=self,
                alias="run_envy",
                cwd=root_dir
            )

            data.ready()

            return summary_res

    def run_task(self):
        summary_res = self._run_envy()
        summary_data = sdk2.ResourceData(summary_res)
        summary = json.loads(summary_data.path.read_bytes())

        logging.info("Envy summary: {}".format(summary))

        for (service, service_update) in summary.items():
            ticket = service_update['ticket']
            changelog = []

            for (package, update) in service_update['packages'].items():
                package_log = rich_get_output(
                    ["yammy", "envy-log", package, update["from"], update["to"]],
                    self, "changelog.{}.{}--{}".format(
                        package.replace("@", "").replace("/", "-"),
                        update["from"].replace(":", "-").replace("/", "-"),
                        update["to"],
                    ), cwd=self.APP_SRC_DIR)

                changelog.append("==+%%{}%% {} -> {}==\n{}".format(
                    package, update["from"], update["to"], package_log
                ))

            error = None

            for i in range(0, 3):
                if error:
                    time.sleep(5)
                    error = None

                logging.info("Updating description: {}, retry: {}".format(ticket, i))

                try:
                    issue = self.startrek.issues[ticket]
                    issue.update(description="\n\n".join(changelog))
                    break
                except Exception as e:
                    error = e

            if error:
                raise error
