# coding=utf-8

import time
import re
import logging

from sandbox import sdk2
import sandbox.common.types.task as ctt

from sandbox.projects.market.frontarc.helpers.ArcanumApi import ReviewRequestFields
from sandbox.projects.market.frontarc.MarketFrontYammyBaseArc import MarketFrontYammyBaseArc
from sandbox.projects.market.frontarc.helpers.sandbox_helpers import rich_check_call
from sandbox.projects.market.frontarc.helpers.yammy.errors import MergeChecksFailedException, PrWrongStatusException, \
    TimeoutException
from sandbox.projects.market.frontarc.helpers.yammy.requirements import YammySmallRequirementsArc

BAD_STATUSES = ("conflicts", "errors", "merge_failed", "discarding", "discarded", "fatal_error")


class MarketFrontYammyMergeArc(MarketFrontYammyBaseArc):
    pr_number = None

    class Parameters(MarketFrontYammyBaseArc.Parameters):
        yammy_build_meta = None

        with sdk2.parameters.Output:
            intermediate_pr_number = sdk2.parameters.Integer("Номер пул-реквеста, через который произошло вливание")

    class Requirements(YammySmallRequirementsArc):
        semaphores = ctt.Semaphores(acquires=[ctt.Semaphores.Acquire(
            name='yammy_merge_task',
            weight=1,
            capacity=1,
        )])

    @sdk2.header()
    def header(self):
        header = super(MarketFrontYammyMergeArc, self).header()

        if not self.Parameters.intermediate_pr_number:
            return header

        return header + ' <a class="status status_assigned" href="https://a.yandex-team.ru/review/{}">{}</a>' \
            .format(self.Parameters.intermediate_pr_number, "Intermediate PR")

    def _wait_for_status(self, pr, good_status, bad_status=BAD_STATUSES, timeout=300):
        logging.info("Waiting for status {}".format(good_status))

        start_time = time.time()

        while True:
            info = self.arcanum_api.get_pull_request_info(pr, ["status"])
            status = info["status"]

            if status in good_status:
                break

            if bad_status and status in bad_status:
                logging.error("Pr in wrong status: {}".format(status))
                raise PrWrongStatusException()

            if time.time() - start_time > timeout:
                raise TimeoutException()

            time.sleep(10)

    def _check_initial_pr(self):
        pr_info = self.arcanum_api.get_review_request_info(
            self.Parameters.pr_number,
            [ReviewRequestFields.checks]
        )

        bad_checks = []
        for check in pr_info["checks"]:
            if check["system"] == "YAMMY" and check["type"] == "Use merge task":
                continue

            if check["required"] and not check["satisfied"]:
                bad_checks.append(check)

        if len(bad_checks) > 0:
            logging.error("Some checks failed: {}".format(bad_checks))
            raise MergeChecksFailedException()

    def _perform_after_merge(self):
        logging.info("Preparing temporary local branch")
        self.arc_checkout(re.sub(r'^users/[^/]*/', "", self.Parameters.head_branch), create_branch=True, force=True)
        self.arc_rebase(rebase_onto="trunk")

        logging.info("Running after-merge scenario")
        rich_check_call(["yarn", "run", "after-merge"], self, "after-merge", cwd=self.APP_SRC_DIR)

        logging.info("Commiting changes")
        self.arc_add(all_changes=True)
        rich_check_call(["arc", "commit", "--no-verify", "-m", "NPM versions up"], self, "commit", cwd=self.APP_SRC_DIR)

    def _create_intermediate_pr(self):
        logging.info("Preparing temporary merge branch")
        branch = "users/robot-metatron/{}/merge-{}".format(
            re.sub(r'^users/', "", self.Parameters.head_branch),
            self.id
        )

        self.arc_checkout(branch, create_branch=True, track=True)
        self.arc_push(branch)

        logging.info("Creating pull request")
        info = self.arcanum_api.get_pull_request_info(self.Parameters.pr_number, ["summary"])

        self.pr_number = self.arc_pr_create("[MERGE] " + info["summary"].encode("utf8"), publish=True, auto=True)

        self.Parameters.intermediate_pr_number = self.pr_number
        self.set_info(
            ('Intermediate pull request created: <a class="status status_success" target="_blank"'
             ' href="https://a.yandex-team.ru/review/{pr}">{pr}</a>').format(pr=self.pr_number),
            do_escape=False
        )

        self._wait_for_status(self.pr_number, ["open"])

    def _merge_intermediate_pr(self):
        logging.info("Merging pull request")
        self.arcanum_api.patch_review_request_policies(self.pr_number, dict(policies=dict(auto_merge="force")))
        self._wait_for_status(self.pr_number, ["merged"])

    def _close_initial_pr(self):
        logging.info("Closing inital PR")
        self.arcanum_api.post_pull_request_comment(self.Parameters.pr_number, dict(
            content="Pull request was merged at: https://a.yandex-team.ru/review/{}".format(self.pr_number),
            draft=False,
            issueStatus="NOT_ISSUE",
        ))

        self.arcanum_api.put_review_request_state(self.Parameters.pr_number, dict(state="closed"))
        self._wait_for_status(self.Parameters.pr_number, ["discarded"], ("fatal_error", "errors"))

    def on_prepare(self):
        super(MarketFrontYammyMergeArc, self).on_prepare()
        self._check_initial_pr()

    def run_task(self):
        self._perform_after_merge()

        self._create_intermediate_pr()
        self._merge_intermediate_pr()

        sdk2.Requirements.semaphores.release()
        self._close_initial_pr()
