# coding=utf-8

import logging

from sandbox import sdk2

from sandbox.projects.market.frontarc.MarketFrontYammyBaseArc import MarketFrontYammyBaseArc
from sandbox.projects.market.frontarc.helpers.sandbox_helpers import rich_check_call


class MarketFrontYammySpokServiceTemplateArc(MarketFrontYammyBaseArc):
    pr_number = None

    class Parameters(MarketFrontYammyBaseArc.Parameters):
        head_branch = sdk2.parameters.String("Тестируемая ветка", required=False, default="trunk")
        root_path = sdk2.parameters.String("Путь до приложения от корня arcanum", required=True,
                                           default="market/front/monomarket",
                                           description="Слэши и пробелы в начале и в конце будут убраны автоматически")
        task_description = None
        commit_hash = None
        diff_set_id = None
        yammy_base_ref = None
        yammy_build_meta = None
        yammy_prebuilt = None
        yammy_build_digest = None

        with sdk2.parameters.Group("Общие параметры") as common_params:
            ticket = sdk2.parameters.String("Тикет", required=True)
            auto_merge = sdk2.parameters.Bool("Автоматический мёрж", default=False)

        with sdk2.parameters.Group("Параметры шаблона") as template_params:
            template_name = sdk2.parameters.String("Тип шаблона", default_value="@yandex-market/template-empty")

        with sdk2.parameters.Group("Параметры сервиса") as service_params:
            service_name = sdk2.parameters.String("Название сервиса", required=True)

        with sdk2.parameters.Output:
            pr_number = sdk2.parameters.Integer("Номер пул-реквеста")

    @property
    def ticket(self):
        return self.Parameters.ticket

    @property
    def branch(self):
        return "users/{}/{}-{}".format("robot-metatron", self.ticket, self.id)

    @property
    def commit_message(self):
        return "{}: Создание сервиса {}".format(
            self.ticket,
            self.Parameters.service_name
        )

    def on_enqueue(self):
        super(MarketFrontYammySpokServiceTemplateArc, self).on_enqueue()
        self.prepend_tags('spok', 'spok-create')

    def apply_template(self):
        logging.info("Applying template")

        with self.timer("template:apply"):
            rich_check_call(
                ["yammy", "ci", "apply-template", self.Parameters.template_name, self.Parameters.service_name],
                self, "apply-template", cwd=self.APP_SRC_DIR,
            )

        with self.timer("template:install"):
            rich_check_call(
                ["yammy", "install"],
                self, "install-template", cwd=self.APP_SRC_DIR,
            )

        rich_check_call(
            ["arc", "checkout", "HEAD", ".yarnrc"],
            self, "yarnrc", cwd=self.APP_SRC_DIR,
        )

    def commit_template(self):
        logging.info("Creating commit")

        self.arc_checkout(branch=self.branch, create_branch=True)

        with self.timer("arcadia:add"):
            logging.debug("Arc: add")
            self.arc_add(all_changes=True)

        with self.timer("arcadia:commit"):
            logging.debug("Arc: commit")
            self.arc_commit(message=self.commit_message, all_changes=True)

    def create_pr(self):
        logging.info("Creating pull request")

        with self.timer("arcadia:push"):
            logging.debug("Arc: push")
            self.arc_push(upstream=self.branch)

        with self.timer("arcadia:pr"):
            logging.debug("Arc: pr")
            self.pr_number = self.arc_pr_create(message=self.commit_message, publish=True, auto=True)

        self.Parameters.pr_number = self.pr_number
        self.set_info(
            'PR created: <a class="status status_success" href="https://a.yandex-team.ru/review/{pr}">{pr}</a>'.format(
                pr=self.pr_number,
            ), do_escape=False)

        return self.pr_number

    def merge(self):
        logging.info("Merging pull request")

        with self.timer("arcadia:merge"):
            self.arcanum_api.patch_review_request_policies(self.pr_number, dict(policies=dict(auto_merge="force")))

    def run_task(self):
        self.apply_template()
        self.commit_template()
        self.create_pr()

        if self.Parameters.auto_merge:
            self.merge()
