# coding=utf-8

import os

from sandbox import sdk2

from sandbox.projects.market.frontarc.helpers.yammy.errors import TestFailure
from sandbox.projects.market.front.resources.yammy_resources import MarketFrontYammyTestArtefact
from sandbox.projects.market.frontarc.MarketFrontYammyPackageTaskArc import MarketFrontYammyPackageTaskArc
from sandbox.projects.market.frontarc.helpers.sandbox_helpers import get_resource_http_proxy_link
from sandbox.projects.market.frontarc.helpers.yammy.requirements import YammyBigRequirementsArc
from sandbox.projects.market.frontarc.helpers.ArcanumApi import CheckStatus

STANDARD_REPORT_NAMES = [
    "index.html",
    "index.htm",
    "index.md",
    "report.html",
    "report.htm",
    "report.txt",
    "report.md",
    "stdout.txt",
]


class MarketFrontYammyTestArc(MarketFrontYammyPackageTaskArc):
    """
    Таск сборки пакетов
    """

    alias = "test"

    class Requirements(YammyBigRequirementsArc):
        pass

    class Parameters(MarketFrontYammyPackageTaskArc.Parameters):
        with sdk2.parameters.Group('Настройки отчётов') as report:
            report_name = sdk2.parameters.String("Название файла отчёта", required=False)
            summary_name = sdk2.parameters.String("Название файла саммари", required=False)

        with sdk2.parameters.Output:
            result = sdk2.parameters.Resource("Отчёт о тестировании")

        with sdk2.parameters.Group("Перезапуск тестов") as test_restart_params:
            test_rerun_failed = sdk2.parameters.Bool("Перезапустить упавшие тесты")

    @property
    def _test_stage(self):
        return self.Parameters.yammy_command_name.split(':')[2]

    @property
    def _test_name(self):
        return self.Parameters.yammy_command_name.split(':')[3]

    def update_status(self, state, description, type=None, url=None):
        if self.Parameters.test_rerun_failed or self._test_stage == 'deploy' or state in (
        CheckStatus.error, CheckStatus.failure):
            super(MarketFrontYammyTestArc, self).update_status(state, description, type=None, url=None)

    def on_enqueue(self):
        self.prepend_tags(
            'test-stage-{}'.format(self._test_stage),
            'test-name-{}'.format(self._test_name),
        )
        super(MarketFrontYammyTestArc, self).on_enqueue()

    def setup_env(self):
        super(MarketFrontYammyTestArc, self).setup_env()
        os.environ["REPORT_DIR"] = self.output_dir

    def _create_report(self, test_report_file, error):
        with self.timer("report:create"):
            res = MarketFrontYammyTestArtefact(
                self,
                "Package {} test {} results".format(self.Parameters.yammy_package, self.Parameters.yammy_command_title),
                self.output_dir,
                commit=self.head_commit,
                ref=self.head_ref,
                test_stage=self._test_stage,
                test_name=self.Parameters.yammy_command_name,
                report_file=test_report_file
            )
            data = sdk2.ResourceData(res)

        with self.timer("report:ready"):
            data.ready()

        status = "failure" if error else "success"

        self.set_info(
            '<b>Test result:</b> <span class="status status_{}">{}</span>'
            ' <a class="status status_draft" href="{}">Report</a>'
            ' <a class="status status_draft" href="//{}/resource/{}/view">Resource: {}</a>'.format(
                status, status,
                "{}/{}".format(get_resource_http_proxy_link(res), test_report_file),
                self.server.host, res.id, res.id,
            ),
            do_escape=False,
        )

        with self.timer("result"):
            self.Parameters.result = res

    def run_task(self):
        (stdout, error) = self.run_command(error_class=TestFailure)

        with self.timer("report:prepare"):
            # ---------------------------------------

            test_report_file = None

            if self.Parameters.report_name:
                test_report_file = self.Parameters.report_name
            else:
                files = os.listdir(self.output_dir)
                for name in STANDARD_REPORT_NAMES:
                    if name in files:
                        test_report_file = name
                        break

            # ---------------------------------------

            if self.Parameters.summary_name:
                test_summary_path = self.output_file(str(self.Parameters.summary_name))
            else:
                test_summary_path = self.output_file('summary.html')

            if test_summary_path and os.path.exists(test_summary_path):
                with open(test_summary_path, 'r') as f:
                    summary = f.read()
            else:
                summary = stdout

        # ---------------------------------------

        if test_report_file and os.path.exists(self.output_file(test_report_file)):
            self._create_report(test_report_file, error)

        self.set_info("<b>Test summary:</b>\n<pre>{}</pre>".format(summary), do_escape=False)

        # ---------------------------------------

        self.append_tags('test-{}'.format("failure" if error else "success"))

        if error:
            raise error
