# coding=utf-8
import os
import pipes
import tempfile

import sandbox.common.types.misc as ctm
import sandbox.sdk2.path as spath
from sandbox import sdk2
from sandbox.projects.common import task_env
from sandbox.projects.market.resources import MARKET_APP

from sandbox.projects.market.frontarc.helpers.MetatronEnvArc import MetatronEnvArc
from sandbox.projects.market.frontarc.helpers.github import clone_repo
from sandbox.projects.market.frontarc.helpers.sandbox_helpers import rich_check_call
from sandbox.projects.market.frontarc.helpers.node import create_node_selector
from sandbox.projects.market.frontarc.helpers.ubuntu import create_ubuntu_selector, setup_container


class MarketNotaBuildStaticArc(sdk2.Task):
    """
    Task for MARKET_NOTA_CLAIM_STATIC build

    https://github.yandex-team.ru/market/nota
    """

    root_dir = '.'

    class Requirements(task_env.TinyRequirements):
        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):
        ubuntu_version = create_ubuntu_selector()
        node_version = create_node_selector()

        app_branch = sdk2.parameters.String(
            "Git ветка проекта",
            default_value="master"
        )

        comment = sdk2.parameters.String(
            "Comment for diff",
            required=False
        )

        dupload = sdk2.parameters.Bool(
            "dupload?",
            default_value=False
        )

    def on_enqueue(self):
        super(MarketNotaBuildStaticArc, self).on_enqueue()
        setup_container(self)

    @property
    def app_owner(self):
        return "market"

    @property
    def app_repo_name(self):
        return "nota"

    @property
    def app_branch(self):
        return self.Parameters.app_branch

    @property
    def app_repo_url(self):
        return "https://github.yandex-team.ru/{owner}/{name}.git".format(owner=self.app_owner, name=self.app_repo_name)

    def on_prepare(self):
        with MetatronEnvArc(self, nodejs_version=self.Parameters.node_version):
            self._prepare_dirs()
            self._clone_repo()

    def on_execute(self):
        with MetatronEnvArc(self, nodejs_version=self.Parameters.node_version):
            self._make()
            self._make_package()
            self._create_app_pack()
            if self.Parameters.dupload:
                self._dupload()

    def _prepare_dirs(self):
        self.root_dir = tempfile.mkdtemp()
        self.app_src_path = os.path.join(self.root_dir, self.app_repo_name)

    def _clone_repo(self):
        clone_repo(
            pipes.quote(self.app_owner),
            pipes.quote(self.app_repo_name),
            pipes.quote(self.app_branch),
            self.app_src_path
        )

    def _make(self):
        rich_check_call(
            ["make"],
            task=self, alias="make", cwd=self.app_src_path,
        )

    def _make_package(self):
        rich_check_call(
            ["make", "package"],
            task=self, alias="make_package", cwd=self.app_src_path,
        )

    def _create_app_pack(self):
        pack_app_archive_path = tempfile.mktemp(suffix=".tar.gz", prefix=self.app_repo_name)

        rich_check_call(
            ["tar", "-C", os.path.join(self.app_src_path, '..'), "-czf", pack_app_archive_path, self.Parameters.app_repo],
            task=self, alias="create_app_pack"
        )

        resource = MARKET_APP(
            self, "nota", "{}.tar.gz".format(self.app_repo_name),
            app_repo=self.app_repo_url,
            app_branch=self.Parameters.app_branch,
        )

        app_res = sdk2.ResourceData(resource)
        app_res.path.write_bytes(spath.Path(pack_app_archive_path).read_bytes())
        app_res.ready()

    def _dupload(self):
        rich_check_call(
            ["dupload", "--nomail"],
            task=self, alias="dupload"
        )
