# coding=utf-8

import logging
import json

import sandbox.common.types.misc as ctm
from sandbox import sdk2
from sandbox.sdk2 import Task, Vault
from sandbox.projects.sandbox_ci.utils.request import send_request
from sandbox.projects.market.frontarc.helpers.ubuntu import create_ubuntu_selector, setup_container
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.market.frontarc.helpers.MetatronEnvArc import ST_OAUTH_TOKEN_VAULT_KEY
from sandbox.projects.common import task_env


class MarketPlannerStartMTArc(Task):
    """
    Таска для запуска и перезапуска мультитестинга планера.
    """

    TSUM_TOKEN_VAULT_KEY = "robot-metatron-tsum-token"
    TSUM_MT_PAGE_URL = "https://tsum.yandex-team.ru/pipe/projects/{project}/multitestings/environments/{name}"
    TSUM_MT_API_URL = "https://tsum.yandex-team.ru/api/multitestings/project/{project}/environments/{name}/createOrRestart"

    class Requirements(task_env.TinyRequirements):
        dns = ctm.DnsType.DNS64
        environments = [
            PipEnvironment("yandex_tracker_client", version="1.3", custom_parameters=["--upgrade-strategy only-if-needed"]),
            PipEnvironment("startrek_client", version="2.3.0", custom_parameters=["--upgrade-strategy only-if-needed"])
        ]

    class Parameters(Task.Parameters):
        ubuntu_version = create_ubuntu_selector()

        ticket_id = sdk2.parameters.String(
            "Тикет",
            required=True
        )

        arcadia_url = sdk2.parameters.ArcadiaUrl(
            "Arcadia repository url",
            default_value="arcadia-arc:/#trunk",
            required=True
        )

        tsum_project = sdk2.parameters.String(
            "Название проекта",
            default_value="market-planner",
            required=True
        )

        tsum_pipeline_type = sdk2.parameters.String(
            "Тип пайплайна",
            default_value="mt-market-planner",
            required=True
        )

    def on_enqueue(self):
        super(MarketPlannerStartMTArc, self).on_enqueue()
        setup_container(self)

    @property
    def _tsum_mt_name(self):
        return self.Parameters.ticket_id.lower()

    @property
    def _tsum_api_url(self):
        return self.TSUM_MT_API_URL.format(
            project=self.Parameters.tsum_project,
            name=self._tsum_mt_name
        )

    @property
    def _tsum_page_url(self):
        return self.TSUM_MT_PAGE_URL.format(
            project=self.Parameters.tsum_project,
            name=self._tsum_mt_name
        )

    def _send_startrek_report(self, message):
        oauth_token = sdk2.Vault.data(ST_OAUTH_TOKEN_VAULT_KEY)

        from startrek_client import Startrek
        st = Startrek(useragent="robot-metatron", token=oauth_token)
        issue = st.issues[self.Parameters.ticket_id]
        issue.comments.create(text=message)

    def _create_or_restart_mt(self):
        token = Vault.data(self.TSUM_TOKEN_VAULT_KEY)

        headers = {
            "Authorization": token,
            "Content-Type": "application/json"
        }

        data = {
            "name": self._tsum_mt_name,
            "type": "USE_EXISTING_PIPELINE",
            "pipelineId": self.Parameters.tsum_pipeline_type,
            "resources": {
                "32fa4237-4246-46a8-84b8-9df3efdd8301": {
                    "key": self.Parameters.ticket_id
                },
                "c549477a-867a-483f-947b-5fd64feef256": {
                    "ref": self.Parameters.arcadia_url
                }
            }
        }

        res = send_request("post", self._tsum_api_url, headers=headers, data=json.dumps(data))

        res.raise_for_status()

        try:
            assert res.json()['name'] == self._tsum_mt_name

            # TODO: Добавить ссылки на балансеры
            self._send_startrek_report("Ссылка на среду: {}".format(self._tsum_page_url))
        except Exception:
            logging.error(u"Bad TSUM response: {}".format(res.text))
            self._send_startrek_report("Ошибка при запуске среды: {}".format(res.text))

            raise Exception("Something is wrong, see logs")

    def on_execute(self):
        self._create_or_restart_mt()
