# coding=utf-8

import os
import logging

from sandbox import sdk2
from sandbox import common
from sandbox.sandboxsdk import errors
from sandbox.common.types import task as ctt
from sandbox.projects.market.frontarc.helpers.github import change_status, GitHubStatusArc, GitHubStatusDescriptionArc
from sandbox.projects.market.resources import MARKET_FRONT_APP_BUNDLE_STATS, MARKET_FRONT_STATOSCOPE_REPORT
from sandbox.projects.market.frontarc.helpers.sandbox_helpers import rich_check_call, get_resource_http_proxy_link
from sandbox.projects.market.frontarc.MarketFrontBundleStatsArc import MarketFrontBundleStatsArc
from sandbox.projects.market.frontarc.helpers.path import raise_if_path_not_exists, makedirs_if_not_exists
from sandbox.projects.market.frontarc.helpers.MetatronEnvArc import MetatronEnvArc
from sandbox.common.types import resource as ctr
from sandbox.projects.sandbox_ci.utils.context import GitRetryWrapper

class MarketStatoscopeReportArc(MarketFrontBundleStatsArc):
    """
    Таска для создания ресурса с результатами сравнения стаитсики по webpack бандлам ветки и мастера.
    Для сравнения используется утилита https://statoscope.tech/
    """

    STATOSCOPE_REPORT_DIR = "statoscope"
    BASE_STATS_DIR = "base_stats"

    class Parameters(MarketFrontBundleStatsArc.Parameters):
        report_path = sdk2.parameters.String(
            "Путь до отчета от корня приложения",
            required=True
        )

        make_report_command = sdk2.parameters.String(
            "Команда для генерации статоскоп репорта",
            description="Строка c плейсхолдерами, например"
                        "'npx statoscope generate -i {base_stats_path} -i {actual_stats_path} -o {report_path}'",
            required=True
        )

        github_context = sdk2.parameters.String(
            "Контекст в github (flow coverage, eslint warnings, etc.)"
        )

        commit_hash = sdk2.parameters.String(
            "Хэш коммита, которому проставляется статус"
        )

    @property
    def _statoscope_report_path(self):
        return os.path.join(self._app_root_path, self.Parameters.report_path)

    def _get_base_stats_path(self):
        logging.info('Finding base stats resource')

        base_stats_resource = MARKET_FRONT_APP_BUNDLE_STATS.find(
            state=ctr.State.READY,
            attrs=dict(
                app_id=self.Parameters.app_id
            )
        ).order(-sdk2.Resource.id).first()

        if base_stats_resource:
            makedirs_if_not_exists(self.BASE_STATS_DIR)

            resource_data = sdk2.ResourceData(base_stats_resource)

            logging.info('Unpacking base stats resource')

            rich_check_call(
                ["tar", "-xzf", "{}".format(resource_data.path), "-C", self.BASE_STATS_DIR],
                task=self, alias="unpack_base_stats"
            )

            return os.path.join(self._root_path, self.BASE_STATS_DIR, self.Parameters.stats_filename)
        else:
            return None

    def _create_statoscope_report(self):
        logging.info('Creating statoscope report')

        raise_if_path_not_exists(
            self._stats_path,
            "Can't run statoscope, {} not exists".format(self._stats_path)
        )

        actual_stats_path = self._stats_path
        base_stats_path = self._get_base_stats_path()

        if not base_stats_path:
            raise Exception("Can't run statoscope, base stats not exists")

        logging.info('Preparing statoscope report dirs')

        makedirs_if_not_exists(self.STATOSCOPE_REPORT_DIR)

        base_stats_path_named = os.path.join(self._root_path, self.STATOSCOPE_REPORT_DIR, 'base.json')
        actual_stats_path_named = os.path.join(self._root_path, self.STATOSCOPE_REPORT_DIR, 'actual.json')

        os.rename(actual_stats_path, actual_stats_path_named)
        os.rename(base_stats_path, base_stats_path_named)

        gh_status = GitHubStatusArc.SUCCESS

        try:
            logging.info('Runing statoscope command')

            rich_check_call(
                ["bash", "-c", self.Parameters.make_report_command.format(
                    base_stats_path=base_stats_path_named,
                    actual_stats_path=actual_stats_path_named,
                    report_path=self._statoscope_report_path
                )],
                self, "create_statoscope_report", cwd=self._app_repo_path
            )
        except:
            gh_status = GitHubStatusArc.ERROR
        finally:
            report_resource = self._create_statoscope_report_resource()
            self._change_gh_status(gh_status, get_resource_http_proxy_link(report_resource))

    def _create_statoscope_report_resource(self):
        logging.info('Creating statoscope report resource')

        raise_if_path_not_exists(
            self._statoscope_report_path,
            "Can't create statoscope resource, {} not exists".format(self._statoscope_report_path)
        )

        resource = MARKET_FRONT_STATOSCOPE_REPORT(
            self, "Statoscope report", self._statoscope_report_path
        )
        resource_data = sdk2.ResourceData(resource)
        resource_data.ready()

        return resource

    def _change_gh_status(self, status, report_url=None):
        if self.Parameters.commit_hash and self.Parameters.github_context:
            change_status(
                owner=self.Parameters.github_owner,
                repo=self.Parameters.github_repo,
                context="[Sandbox CI] {}".format(self.Parameters.github_context),
                sha=self.Parameters.commit_hash,
                url=report_url or common.utils.get_task_link(self.id),
                state=status,
                description=GitHubStatusDescriptionArc[status]
            )

    def on_prepare(self):
        with MetatronEnvArc(self, nodejs_version=self.Parameters.node_version), GitRetryWrapper():
            self._change_gh_status(GitHubStatusArc.PENDING)

    def on_break(self, prev_status, status):
        with MetatronEnvArc(self, nodejs_version=self.Parameters.node_version), GitRetryWrapper():
            self._change_gh_status(GitHubStatusArc.FAILURE)

    def on_failure(self, prev_status):
        with MetatronEnvArc(self, nodejs_version=self.Parameters.node_version), GitRetryWrapper():
            self._change_gh_status(GitHubStatusArc.FAILURE)

    def on_execute(self):
        super(MarketFrontBundleStatsArc, self).on_execute()

        with MetatronEnvArc(self, nodejs_version=self.Parameters.node_version), GitRetryWrapper():
            self._clone_repo()
            self._create_stats()
            self._create_statoscope_report()
