# -*- coding: utf-8 -*-

"""
Файл-хэлпер для Api Arcanum
@see https://arcanum.yandex-team.ru/api/swagger/index.html
@warning Api считается условно внутренним. Использовать на свой страх и риск
"""

import logging
import requests
from requests.exceptions import HTTPError

from functools import wraps


class ArcanumApiException(Exception):
    pass


class Fields(object):
    @classmethod
    def get_static_fields(cls):
        """"
            Получает список статических полей класса
        """
        return {field for field in vars(cls) if not field.startswith('__') or callable(field)}


class CheckStatus(Fields):
    """
        Возможный статус проверки ПР
    """
    unknown = "unknown"
    pending = "pending"
    success = "success"
    failure = "failure"
    error = "error"
    cancelled = "cancelled"
    skipped = "skipped"


class PullRequestFields(Fields):
    """
        Список полей, которые описывают ПР. Требуется явно перечислять необходимые для запроса в API
    """
    created_at = "created_at"
    updated_at = "updated_at"
    closed_at = "closed_at"
    merge_allowed = "merge_allowed"
    auto_merge = "auto_merge"
    status = "status"
    id = "id"
    url = "url"
    author = "author"
    summary = "summary"
    description = "description"
    vcs = "vcs"


class ReviewRequestFields(Fields):
    """
        Список полей, которые описывают запрос на ревью.
        Разница по сравнению с ПР в терминологии. Одно публичное (пр), другое не очень (рр)
        pull_request_id == review_request_id
    """
    id = "id"
    url = "url"
    author = "author"
    outstaff = "outstaff"
    summary = "summary"
    description = "description"
    vcs = "vcs"
    state = "state"
    checks = "checks"
    assignees = "assignees"
    approvers = "approvers"
    reviewers = "reviewers"
    review = "review"
    issues = "issues"
    commentators = "commentators"
    subscriptions = "subscriptions"
    unread = "unread"
    external_id = "external_id"
    full_status = "full_status"
    merge_commits = "merge_commits"
    checks_satisfied = "checks_satisfied"
    wip_16e8e55d142_policies = "wip_16e8e55d142_policies"
    policies = "policies"
    merge_allowed = "merge_allowed"
    has_block_merge = "has_block_merge"
    active_diff_set = "active_diff_set"
    diff_sets = "diff_sets"
    subscriber_roles = "subscriber_roles"
    ya_commit_tasks = "ya_commit_tasks"
    ya_rebase_tasks = "ya_rebase_tasks"
    created_at = "created_at"
    updated_at = "updated_at"
    closed_at = "closed_at"
    ci_settings = "ci_settings"
    latest_ci_checks = "latest_ci_checks"
    labels = "labels"
    commit_description = "commit_description"


class DiffSetFields(Fields):
    """
        Список полей diffset. Требуется явно перечислять необходимые для запроса в API
    """
    id = "id"
    gsid = "gsid"
    description = "description"
    zipatch = "zipatch"
    status = "status"
    conflicts = "conflicts"
    owners = "owners"
    patch_url = "patch_url"
    patch_vcs_ids = "patch_vcs_ids"
    patch_stats = "patch_stats"
    out_of_arcadia_bounds = "out_of_arcadia_bounds"
    created_at = "created_at"
    published_at = "published_at"
    discarded_at = "discarded_at"
    has_conflicts = "has_conflicts"
    legacy_revision = "legacy_revision"
    arc_branch_heads = "arc_branch_heads"
    legacy_in_arcadia = "legacy_in_arcadia"


def check_token(fn):
    """
    Декоратор, проверяющий, что token не потерялся
    :param fn:
    :return:
    """

    @wraps(fn)
    def wrapper(*args, **kwargs):
        if 'token' not in kwargs:
            raise TypeError("Token must be set to make request in '{}'".format(fn.__name__))

        return fn(*args, **kwargs)

    return wrapper


def check_fields(fields):
    """
    Фабрика декораторов, валидирующих один уровень правильности query-параметра fields
    :param fields:
    :return:
    """
    field_set = fields.get_static_fields()

    def actual_decorator(fn):
        @wraps(fn)
        def wrapper(*args, **kwargs):
            if 'fields' not in kwargs or kwargs['fields'] is None:
                return fn(*args, **kwargs)

            t = type(kwargs['fields'])

            if t is not list:
                raise TypeError("Argument 'fields' should be either list or None in method {}".format(fn.__name__))

            for field in kwargs['fields']:
                _field = field if type(field) == str else field[0]
                if _field not in field_set:
                    raise TypeError("Field {} is unknown to arcanum api in method {}".format(_field, fn.__name__))

            return fn(*args, **kwargs)

        return wrapper

    return actual_decorator


class ArcanumApi(object):
    """
        Класс для общения с Api Arcanum
        Все методы представлены в двух вариантах:
        - Статический - необходимо руками передать OUATH-токен
        - Инстанса - можно один раз при вызове конструктора указать токен, далее использовать без указания токена

        Часть get-методов дополнительно запрашивает аргумент fields
        fields вида ['id', 'status', ['diff_set', ['id', 'owner']]] в запросе превратится в id,status,diff_set(id, owner)

        Конфигурация проверок привязывается к review-request (RR, РР) или что примерно тоже самое pull-request (PR, ПР)
        Статус проверки привязывается к дифф-сету

        Все методы возвращают объект `response` из библиотеки `requests`. Т.е. доступны методы, text, json и прочие
    """

    def __init__(self, oauth_token, ):
        self.token = oauth_token

    __API_ADDRESS = "https://arcanum.yandex-team.ru/api/{version}/{path}"

    __METHODS = dict(
        get=requests.get,
        post=requests.post,
        patch=requests.patch,
        put=requests.put,
    )

    @staticmethod
    def _request(method, *args, **kwargs):
        response = ArcanumApi.__METHODS[method](*args, **kwargs)
        raw = response.text

        try:
            response.raise_for_status()
        except HTTPError as e:
            logging.error("Response: {}".format(raw))
            raise

        # В части ручек ответ не приходит вообще.
        # Минимальный размер json'а 2 символа, проверяем что нам есть что декодировать.
        if len(raw) > 1:
            result = response.json()
        else:
            result = dict(data=True)

        if "errors" in result and len(result["errors"]) > 0:
            logging.error("Acranum Errors: {}".format(result["errors"]))
            raise ArcanumApiException(result["errors"][0]["message"])

        return result["data"]

    @staticmethod
    def _format_url(path, version='v1'):
        return ArcanumApi.__API_ADDRESS.format(version=version, path=path)

    @staticmethod
    def _get_auth_headers(token):
        return {
            'Content-Type': 'application/json',
            'Authorization': "OAuth {}".format(token)
        }

    @staticmethod
    def _get_fields(field_list):
        """
            Уплощает список field в строковый параметр для запроса
            :param field_list: Например, ['id', 'status', ['diffsets', ['id', 'status', 'owner']]]
            :return: Например, {'fields': 'id,status,diffsets(id,status,owner)'}
        """
        if field_list is None:
            return {}

        def flatten(field):
            if type(field) is list or type(field) is tuple:
                return "{}({})".format(field[0], ",".join(field[1]))
            return field

        return {
            'fields': ','.join([flatten(field) for field in field_list])
        }

    @staticmethod
    @check_token
    @check_fields(PullRequestFields)
    def _get_pull_request_info(pull_request_id, token=None, fields=None):
        """
            Возвращает описание пулл-реквеста. Опционально можно указать выводимые поля для ПР
            type: (int, string, [string: PullRequestFields]) -> object
        """
        return ArcanumApi._request(
            "get",
            ArcanumApi._format_url("pull-requests/{}".format(pull_request_id)),
            headers=ArcanumApi._get_auth_headers(token),
            params=ArcanumApi._get_fields(fields)
        )

    def get_pull_request_info(self, pull_request_id, fields=None):
        """
            Возвращает описание пулл-реквеста. Опционально можно указать выводимые поля для ПР
            type: (int, [string: PullRequestFields]) -> object
        """
        return self._get_pull_request_info(pull_request_id, token=self.token, fields=fields)

    @staticmethod
    @check_token
    @check_fields(ReviewRequestFields)
    def _get_review_request_info(pull_request_id, token=None, fields=None):
        """
            Возвращает описание ревью-реквеста. Опционально можно указать выводимые поля для РР
            type: (int, string, [string: ReviewRequestFields]) -> object
        """
        return ArcanumApi._request(
            "get",
            ArcanumApi._format_url("review-requests/{}".format(pull_request_id)),
            headers=ArcanumApi._get_auth_headers(token),
            params=ArcanumApi._get_fields(fields)
        )

    def get_review_request_info(self, pull_request_id, fields=None):
        """
            Возвращает описание ревью-реквеста. Опционально можно указать выводимые поля для РР
            type: (int, [string: ReviewRequestFields]) -> object
        """
        return self._get_review_request_info(pull_request_id, token=self.token, fields=fields)

    @staticmethod
    @check_token
    def _post_pull_request_comment(pull_request_id, content, token=None):
        """
            Отправляет комментарий в ПР
            type: (int, string, string) -> object
        """
        return ArcanumApi._request(
            "post",
            ArcanumApi._format_url("pull-requests/{}/comments".format(pull_request_id)),
            json=content,
            headers=ArcanumApi._get_auth_headers(token)
        )

    def post_pull_request_comment(self, pull_request_id, content):
        """
            Возвращает описание пулл-реквеста
            type: (int, string) -> object
        """
        return self._post_pull_request_comment(pull_request_id, content, token=self.token)

    @staticmethod
    @check_token
    @check_fields(DiffSetFields)
    def _get_review_request_diff_set_list(pull_request_id, token=None, fields=None):
        """
            Возвращает список диффсетов для ревью-реквеста. Опционально можно указать выводимые поля для диффсета
            type: (int, string, [string: DiffSetFields]) -> object
        """
        return ArcanumApi._request(
            "get",
            ArcanumApi._format_url("review-requests/{}/diff-sets".format(pull_request_id)),
            headers=ArcanumApi._get_auth_headers(token),
            params=ArcanumApi._get_fields(fields)
        )

    def get_review_request_diff_set_list(self, pull_request_id, fields=None):
        """
            Возвращает список диффсетов для ревью-реквеста. Опционально можно указать выводимые поля для диффсета
            type: (int, [string: DiffSetFields]) -> object
        """
        return ArcanumApi._get_review_request_diff_set_list(pull_request_id, token=self.token, fields=fields)

    @staticmethod
    @check_token
    @check_fields(DiffSetFields)
    def _update_review_request_description(pull_request_id, description, token=None):
        """
            Обновляет описание пулреквеста
        """
        return requests.put(
            ArcanumApi._format_url("review-requests/{}/description".format(pull_request_id)),
            headers=ArcanumApi._get_auth_headers(token),
            json={"description": description},
        )

    def update_review_request_description(self, pull_request_id, description):
        """
            Обновляет описание пулреквеста
        """
        return ArcanumApi._update_review_request_description(pull_request_id, description=description, token=self.token)

    @staticmethod
    @check_token
    @check_fields(DiffSetFields)
    def _get_review_request_diff_set(pull_request_id, diff_set_id, token=None, fields=None):
        """
            Возвращает описание диффсета. Опционально можно указать выводимые поля для диффсета
            type: (int, int, string, [string: DiffSetFields]) -> object
        """
        return ArcanumApi._request(
            "get",
            ArcanumApi._format_url("review-requests/{}/diff-sets/{}".format(pull_request_id, diff_set_id)),
            headers=ArcanumApi._get_auth_headers(token),
            params=ArcanumApi._get_fields(fields)
        )

    def get_review_request_diff_set(self, pull_request_id, diff_set_id, token=None, fields=None):
        """
            Возвращает описание диффсета. Опционально можно указать выводимые поля для диффсета
            type: (int, int, [string: DiffSetFields]) -> object
        """
        return ArcanumApi._get_review_request_diff_set(pull_request_id, diff_set_id, token=self.token, fields=None)

    @staticmethod
    @check_token
    def _post_review_request_check_requirement(pull_request_id, content, token=None):
        """
            Создаёт проверку для РР. Статус проверки привязан к дифф-сету и не задаётся через эту ручку
            :param content.system: Группировка в интерфейсе арканума
            :param content.type: Название проверки
            :param content.enabled:
            type: (int, {system: string, type: string, enabled: boolean}, string) -> object
        """
        return ArcanumApi._request(
            "post",
            ArcanumApi._format_url("review-requests/{}/check-requirements".format(pull_request_id)),
            json=content,
            headers=ArcanumApi._get_auth_headers(token)
        )

    def post_review_request_check_requirement(self, pull_request_id, content):
        """
            Создаёт проверку для РР. Статус проверки привязан к дифф-сету и не задаётся через эту ручку
            :param content.system: Группировка в интерфейсе арканума
            :param content.type: Название проверки
            :param content.enabled:
            type: (int, {system: string, type: string, enabled: boolean}, string) -> object
        """
        return self._post_review_request_check_requirement(pull_request_id, content, token=self.token)

    @staticmethod
    @check_token
    def _patch_review_request_policies(pull_request_id, content, token=None):
        """
            Изменяет политики пул-реквеста
            :param content.auto_merge: Политика мёржа. Enum: disabled, on_satisfied_requirements, force
        """
        return ArcanumApi._request(
            "patch",
            ArcanumApi._format_url("review-requests/{}/policies".format(pull_request_id)),
            json=content,
            headers=ArcanumApi._get_auth_headers(token)
        )

    def patch_review_request_policies(self, pull_request_id, content):
        """
            Изменяет политики пул-реквеста
            :param content.auto_merge: Политика мёржа. Enum: disabled, on_satisfied_requirements, force
        """
        return self._patch_review_request_policies(pull_request_id, content, token=self.token)

    @staticmethod
    @check_token
    def _put_review_request_state(pull_request_id, content, token=None):
        """
            Изменяет статус пул-реквеста
            :param content.state: Статус пул-реквеста. Enum: open, closed
        """
        return ArcanumApi._request(
            "put",
            ArcanumApi._format_url("review-requests/{}/state".format(pull_request_id)),
            json=content,
            headers=ArcanumApi._get_auth_headers(token)
        )

    def put_review_request_state(self, pull_request_id, content):
        """
            Изменяет статус пул-реквеста
            :param content.state: Статус пул-реквеста. Enum: open, closed
        """
        return self._put_review_request_state(pull_request_id, content, token=self.token)

    @staticmethod
    @check_token
    def _get_review_request_diff_set_checks(pull_request_id, diff_set_id, token=None):
        """
            Возвращает список проверок диффсета
            type: (int, int, string) -> object
        """
        return ArcanumApi._request(
            "get",
            ArcanumApi._format_url("review-requests/{}/diff-sets/{}/checks".format(pull_request_id, diff_set_id)),
            headers=ArcanumApi._get_auth_headers(token)
        )

    def get_review_request_diff_set_checks(self, pull_request_id, diff_set_id):
        """
            Возвращает список проверок диффсета
            type: (int, int) -> object
        """
        return ArcanumApi._get_review_request_diff_set_checks(pull_request_id, diff_set_id, token=self.token)

    @staticmethod
    @check_token
    def _post_review_request_check(pull_request_id, diff_set_id, content, token=None):
        """
            Проставляет статус проверки диффсету
            :param pull_request_id: id ПР или РР
            :param diff_set_id: id diff_set
            :param content: Состояние проверки
            :param content['system']: Группировка в интерфейсе арканума
            :param content['type']: Название проверки
            :param content['status']: Статус проверки. Тип CheckStatus
            :param content['description']: Описание проверки. Тип Строка
            :param content['systemCheckUri'] гиперссылка, для поля type
            type: (int, int, {system: string, type: string, enabled: boolean}, string) -> object
        """
        return ArcanumApi._request(
            "post",
            ArcanumApi._format_url("review-requests/{}/diff-sets/{}/checks".format(pull_request_id, diff_set_id)),
            json=content,
            headers=ArcanumApi._get_auth_headers(token)
        )

    def post_review_request_check(self, pull_request_id, diff_set_id, content):
        """
            Проставляет статус проверки диффсету
            :param pull_request_id: id ПР или РР
            :param diff_set_id: id diff_set
            :param content: Состояние проверки
            :param content['system']: Группировка в интерфейсе арканума
            :param content['type']: Название проверки
            :param content['status']: Статус проверки. Тип CheckStatus
            :param content['systemCheckUri'] гиперссылка, для поля type
            type: (int, int, {system: string, type: string, enabled: boolean}, string) -> object
        """
        return self._post_review_request_check(pull_request_id, diff_set_id, content, token=self.token)
