# coding=utf-8

import logging
import re

from sandbox import sdk2
from sandbox.projects.common.vcs.arc import Arc, ResetMode

ARC_TOKEN_VAULT_KEY = "robot-metatron-arc-oauth"


class MarketFrontArcBase(object):
    arc = None
    mount_point = None
    arcadia = None

    def arc_mount(self, store_path=None, object_store_path=None):
        arc_token = sdk2.Vault.data(ARC_TOKEN_VAULT_KEY)
        self.arc = Arc(arc_oauth_token=arc_token)
        self.mount_point = self.arc.mount_path(None, "trunk", fetch_all=False, store_path=store_path, object_store_path=object_store_path)
        self.arcadia = self.mount_point.__enter__()

    def arc_unmount(self):
        if self.mount_point and self.mount_point.mounted:
            self.mount_point.unmount()
            self.mount_point = None

    def arc_rev_parse(self, ref):
        return self.arc.rev_parse(self.arcadia, ref)

    def arc_checkout_smart(
        self,
        pr=None,
        commit=None,
        branch=None,
        iteration=None,
        published=False,
        default_branch_name="market-front-branch"
    ):
        """
        В зависимости от входных параметров устанавливает определённый стейт аркадии в контейнере.

        Приоритет состояния файлов - PR > commit > branch.
        Название ветки в любом случае будет соовтетствовать указанному в branch.
        При необходимости создастся новая ветка с именем branch, название ветки из PR игнорируется.

        Если нужно счекаутить ветку, а не PR - то передавать надо PR=0 и commit=0 (или не передавать их вовсе)
        """

        if commit and re.match(r'^[0-9]{,8}$', commit):
            commit = "r" + commit

        if pr:
            logging.info("Checkout pr: {}#{}".format(pr, iteration))
            self.arc_pr_checkout(pr, iteration=iteration, published=published)

            if branch and self.arc.info(self.arcadia).get('branch') != branch:
                logging.info("Creating branch: {} from PR {}".format(branch, pr))
                self.arc_checkout(branch=branch, create_branch=True, track=False, force=True)
        elif branch == "trunk":
            logging.info("Checkout trunk")
            self.arc_checkout(branch="trunk")

            if commit:
                self.arc_reset(mode=ResetMode.HARD, force=True, branch=commit)
        else:
            branch_name = branch if branch else default_branch_name
            start_point = commit if commit else branch

            logging.info("Checkout branch: {} from {}".format(branch_name, start_point))
            self.arc_checkout(
                branch=branch_name,
                create_branch=bool(commit and branch),
                start_point=start_point,
                track=False,
                force=True,
            )

    def arc_checkout_default(self):
        self.arc_checkout_smart(
            pr=self.Parameters.pr_number,
            commit=self.Parameters.commit_hash,
            branch=self.Parameters.head_branch,
            iteration=None,
            published=False
        )

    def arc_push(self, upstream=None, force=False, refspecs=None):
        return self.arc.push(
            self.arcadia,
            upstream=upstream,
            force=force,
            refspecs=refspecs
        )

    def arc_export(self, repo_path, export_path, rev="trunk"):
        return self.arc.export(self.arcadia, rev, repo_path, export_path)

    def arc_checkout(self, branch="trunk", create_branch=False, start_point=None, track=False, force=False):
        return self.arc.checkout(
            self.arcadia,
            branch=branch,
            create_branch=create_branch,
            start_point=start_point,
            track=track,
            force=force
        )

    def arc_reset(self, mode=ResetMode.MIXED, branch=None, path=None, force=False):
        self.arc.reset(self.arcadia, mode, branch, path, force)

    def arc_commit(self, message, all_changes=False):
        return self.arc.commit(self.arcadia, message=message, all_changes=all_changes)

    def arc_add(self, path=None, all_changes=False):
        return self.arc.add(self.arcadia, path=path, all_changes=all_changes)

    def arc_cherry_pick(self, commit=None, add_commit_name=False, allow_empty=False):
        return self.arc.cherry_pick(
            self.arcadia,
            commit=commit,
            add_commit_name=add_commit_name,
            allow_empty=allow_empty
        )

    def arc_rebase(self, upstream=None, rebase_onto=None):
        return self.arc.rebase(self.arcadia, upstream=upstream, rebase_onto=rebase_onto)

    def arc_pull(self, rebase=False):
        return self.arc.pull(self.arcadia, rebase=rebase)

    def arc_fetch(self, branch=None):
        return self.arc.fetch(self.arcadia, branch=branch)

    def arc_pr_create(self, message, to=None, publish=False, auto=False, no_commits=False):
        if not message:
            raise ValueError("Message is not defined")

        stdout = self.arc.pr_create(
            self.arcadia,
            to=to,
            message=message,
            publish=publish,
            auto=auto,
            no_commits=no_commits
        )

        match_id = re.search(r'a\.yandex-team\.ru/review/(\d+)', stdout)
        if match_id:
            return int(match_id.group(1))

    def arc_pr_checkout(self, id, iteration=None, published=False):
        return self.arc.pr_checkout(self.arcadia, id=id, iteration=iteration, published=published)

    def arc_pr_status(self, as_dict=False):
        return self.arc.pr_status(self.arcadia, as_dict=as_dict)

    def arc_status(self, as_dict=False):
        return self.arc.status(self.arcadia, as_dict=as_dict)

    def arc_branch(self, branch=None, remote_branch=None, all=False, as_dict=False):
        return self.arc.branch(self.arcadia, branch=branch, remote_branch=remote_branch, all=all, as_dict=as_dict)

    def arc_merge_base(self, base="trunk", head="HEAD"):
        return self.arc.get_merge_base(self.arcadia, base_branch=base, branch=head)

    def arc_show(self, commit=None, path=None, as_dict=False, name_status=False):
        return self.arc.show(self.arcadia, commit=commit, path=path, as_dict=as_dict, name_status=name_status)

    def arc_log(
        self,
        path=None,
        start_commit=None,
        end_commit=None,
        max_count=0,
        author="",
        as_dict=False,
        walk=True,
        name_only=False,
    ):
        return self.arc.log(
            self.arcadia,
            path=path,
            start_commit=start_commit,
            end_commit=end_commit,
            max_count=max_count,
            author=author,
            as_dict=as_dict,
            walk=walk,
            name_only=name_only
        )

    def arc_describe(
        self,
        commits=None,
        all=False,
        exact_match=False,
        match=None,
        exclude=None,
        always=False,
        dirty=None,
        svn=False
    ):
        return self.arc.describe(
            self.arcadia,
            commits=commits,
            all=all,
            exact_match=exact_match,
            match=match,
            exclude=exclude,
            always=always,
            dirty=dirty,
            svn=svn
        )
