# coding=utf-8
import logging
import json
import os
import hashlib

import sandbox.sdk2.path as spath
from sandbox.projects.market.frontarc.helpers.props import create_string_selector
from sandbox import sdk2

logger = logging.getLogger(__name__)
level = 'error'
project = 'market_autotests_hermione'
language = 'nodejs'

TEST_TYPE_RELEASE = 'release'
TEST_TYPE_FEATURE = 'feature'
TEST_TYPE_MASTER = 'master'


def create_metrics_params():
    class MetricsParametersArc(sdk2.Task.Parameters):
        with sdk2.parameters.Group('Metrics') as metrics_block:
            upload_metrics = sdk2.parameters.Bool(
                'Сохранять статистику и ошибки в ресурсы',
                default=True,
            )
            send_metrics = sdk2.parameters.Bool(
                'Выгружать статистику и ошибки в logbroker',
                default=False,
            )
            logbroker_endpoint = sdk2.parameters.String(
                'Хост логброкера для записи статистики',
                default="logbroker.yandex.net",
            )
            logbroker_port = sdk2.parameters.Integer(
                'Порт логброкера для записи статистики',
                default=2135,
            )
            logbroker_topic = sdk2.parameters.String(
                'Топик логброкера для записи статистики',
                default="market-front-infra/stable/autotest-stats",
            )
            logbroker_errors_topic = sdk2.parameters.String(
                'Топик логброкера для записи ошибок',
                default="market-front-infra/market-infra-error-booster",
            )

            metrics_tests_type = sdk2.parameters.String(
                'Тип тестов',
                choices=[
                    ('e2e', 'e2e'),
                    ('unit', 'unit'),
                ],
                default='e2e',
            )

            metrics_project = sdk2.parameters.String("Проект",
                                                     description="whitemarket-desktop или bussiness-touch, etc...")

            metrics_branch_type = sdk2.parameters.String(
                'Тип ветки',
                choices=[
                    ('pr', 'pr'),
                    ('release', 'release'),
                    ('trunk', 'trunk'),
                ],
                default='pr',
            )

            metrics_env = sdk2.parameters.String(
                'Среда',
                choices=[
                    ('development', 'development'),
                    ('testing', 'testing'),
                    ('prestable', 'prestable'),
                    ('production', 'production'),
                ],
                default='testing',
            )

            metrics_tech = sdk2.parameters.List("Технологии",
                                                description="hermione, geminy, testament, etc...")
            metrics_tags = sdk2.parameters.List("Доп. теги")

    return MetricsParametersArc


def select_send_metrics_parameters(task_parameters):
    return {
        'upload_metrics': task_parameters.upload_metrics,
        'send_metrics': task_parameters.send_metrics,
        'logbroker_endpoint': task_parameters.logbroker_endpoint,
        'logbroker_port': task_parameters.logbroker_port,
        'logbroker_topic': task_parameters.logbroker_topic,
    }


def select_additional_metrics_parameters(task_parameters):
    return {
        'metrics_tests_type': task_parameters.metrics_tests_type,
        'metrics_project': task_parameters.metrics_project,
        'metrics_branch_type': task_parameters.metrics_branch_type,
        'metrics_env': task_parameters.metrics_env,
        'metrics_tech': task_parameters.metrics_tech,
        'metrics_tags': task_parameters.metrics_tags
    }


def collect_cases_stats(resource_path, ticket, repo, branch, platform, task_id, is_retry, metrics_params):
    allure_cases_dir_path = os.path.join(resource_path, 'data/test-cases')

    cases_stats = []
    for test_case in os.listdir(allure_cases_dir_path):
        test_case_data_raw = spath.Path(allure_cases_dir_path, test_case).read_bytes()
        test_case_data = json.loads(test_case_data_raw)

        test_name = test_case_data.get('name')
        md5_hash = hashlib.md5()
        md5_hash.update(test_name)
        test_id = md5_hash.hexdigest()[:7]

        item = {
            'type': metrics_params.get('metrics_tests_type'),
            'env': metrics_params.get('metrics_env'),
            'ticket': ticket,
            'repo': repo,
            'branch': branch,
            'platform': platform,
            'project': metrics_params.get('metrics_project'),
            'sandboxTaskId': task_id,
            'testId': test_id,
            'testName': test_name,
            'isRetry': 1 if test_case_data.get('retry') or is_retry else 0,
            'status': test_case_data.get('status'),
            'timestamp': int(test_case_data.get('time', {}).get('start') / 1000),
            'duration': test_case_data.get('time', {}).get('duration'),
            'tags': map(lambda t: "tech_%s" % t, metrics_params.get('metrics_tech')) +
                    metrics_params.get('metrics_tags')
        }

        if 'kadavr' in test_case_data.get('parameterValues'):
            item.get('tags').append('kadavr')

        cases_stats.append(item)

    return cases_stats


def collect_cases_errors(resource_path, ticket, repo, branch, platform, task_id, metrics_params):
    allure_cases_dir_path = os.path.join(resource_path, 'data/test-cases')
    allure_attachments_dir_path = os.path.join(resource_path, 'data/attachments')

    cases_errors = []
    for test_case in os.listdir(allure_cases_dir_path):
        test_case_data_raw = spath.Path(allure_cases_dir_path, test_case).read_bytes()
        test_case_data = json.loads(test_case_data_raw)

        status = test_case_data.get('status')

        if status != 'broken' and status != 'failed':
            continue

        kadavr_session_id_file_name = None
        kadavr_session_id = None

        for step in test_case_data.get('testStage', {}).get('steps', []):
            for attach in step.get('attachments', []):
                if attach.get('name') == 'kadavr_session_id':
                    kadavr_session_id_file_name = attach.get('source')
                    break
            if kadavr_session_id_file_name is not None:
                break

        if kadavr_session_id_file_name is not None:
            for attach_file_name in os.listdir(allure_attachments_dir_path):
                if kadavr_session_id_file_name == attach_file_name:
                    attach_file_raw = spath.Path(allure_attachments_dir_path, attach_file_name).read_bytes()
                    kadavr_session_id = attach_file_raw
                    break

        item = {
            'env': metrics_params.get('metrics_env'),
            'stack': test_case_data.get('statusTrace'),
            'level': level,
            'file': test_case_data.get('source'),
            'message': test_case_data.get('statusMessage'),
            'project': project,
            'service': platform,
            'block': test_case_data.get('name'),
            'language': language,
            'timestamp': int(test_case_data.get('time', {}).get('stop')),
            'additional': {
                'ticket': ticket,
                'repo': repo,
                'branch': branch,
                'sandboxTaskId': str(task_id),
                'barnchType': get_test_type(branch),
                'project': metrics_params.get('metrics_project'),
                'testType': metrics_params.get('metrics_tests_type')
            }
        }

        if kadavr_session_id is not None:
            item.get('additional')['kadavrSessionId'] = kadavr_session_id

        for label in test_case_data.get('labels', []):
            item.get('additional')[label.get('name')] = label.get('value')

        for label in map(lambda t: "tag_tech_%s" % t, metrics_params.get('metrics_tech')) + \
                     map(lambda t: "tag_%s" % t, metrics_params.get('metrics_tags')):
            item.get('additional')[label] = 'true'

        cases_errors.append(item)

    return cases_errors


def get_test_type(branch):
    test_type = TEST_TYPE_FEATURE

    if branch == 'master':
        test_type = TEST_TYPE_MASTER

    if 'release/' in branch:
        test_type = TEST_TYPE_RELEASE

    return test_type
