# coding=utf-8
import os
import hashlib
import logging
import re
import boto3

import mimetypes

mime = mimetypes.MimeTypes()
mime.encodings_map['.br'] = 'br'

# full s3 path will be e.g. '/s3/market-static/white-desktop/'
S3_BUCKET_NAME = "market-static"
S3_REPO_TO_DIR = {
    "marketfront_market_desktop": "white-desktop",
    "marketfront_market_touch": "white-touch",
    "marketfront_business_desktop": "black-desktop",
    "affiliate": "affiliate",
    "vendors": "vendors",
    "partner": "partnernode",
    "partnernode-cms": "partnernode-cms",
    "marketcorp": "marketcorp",  # нужен вообще?
    "tpl-front": "market-tpl",
    "export-static": "export-static"
}

class ObjectAlreadyExistsErrorArc(Exception):
    def __init__(self, bucket_name, object_key, expected_etag, actual_etag):
        super(ObjectAlreadyExistsErrorArc, self) \
            .__init__("S3: Object at key '{}' already existst in bucket {};"
                      "Expected etag is '{}'; actual etag is '{}'".format(
                        object_key,
                        bucket_name,
                        expected_etag,
                        actual_etag
        ))


def get_connection():
    return boto3.resource(
        "s3",
        endpoint_url="https://s3.mds.yandex.net",
        aws_access_key_id=os.environ["AWS_ACCESS_KEY_ID"],
        aws_secret_access_key=os.environ["AWS_SECRET_ACCESS_KEY"]
    )


def _get_client(resource):
    return resource.meta.client


def _md5(fname):
    hash_md5 = hashlib.md5()
    with open(fname, "rb") as f:
        for chunk in iter(lambda: f.read(4096), b""):
            hash_md5.update(chunk)
    return hash_md5.hexdigest()


def check_same_file(conn, bucket_name, file_path, s3_path, ignore_mismatch=False):
    """
    Возвращает True, если файл есть, и md5 совпадает
    False, если файла нет
    Бросает ObjectAlreadyExistsError, если файл есть и md5 отличачется
    """
    client = _get_client(conn)

    try:
        s3_key = client.head_object(Bucket=bucket_name, Key=s3_path)
    except:
        return False

    print(s3_key)
    expected_etag = '"{}"'.format(_md5(file_path))
    result_etag = s3_key["ETag"]

    if expected_etag != result_etag:
        if ignore_mismatch:
            logging.warn("Etag mismatch! File: {}. Expected: {}. Actual: {}".format(
                file_path, expected_etag, result_etag
            ))
        else:
            raise ObjectAlreadyExistsErrorArc(bucket_name, s3_path, expected_etag, result_etag)

    return True


def upload_file(conn, bucket_name, file_path, s3_path):
    bucket = conn.Bucket(bucket_name)

    (mimeType, encType) = mime.guess_type(file_path)

    extra_args = {}
    if mimeType is not None:
        extra_args["ContentType"] = mimeType
    if encType is not None:
        extra_args["ContentEncoding"] = encType

    bucket.upload_file(file_path, s3_path, ExtraArgs=extra_args)

    logging.info("S3: Uploaded object '{}' into bucket '{}' with content-type '{}' and content-encoding '{}'".format(
        s3_path, bucket_name, mimeType, encType
    ))


def safe_upload(conn, bucket_name, file_path, s3_path, ignore_mismatch=False):
    same_file_exists = check_same_file(conn, bucket_name, file_path, s3_path, ignore_mismatch)

    if not same_file_exists:
        upload_file(conn, bucket_name, file_path, s3_path)


def s3_path_by_repo(app_repo, platform=None, dirname=None):
    # get last part of repository url
    repo = re.findall(r"/([^/]*)$", app_repo)[-1]
    # for cases it was specified with `.git`
    repo_name = repo.split(".")[0]

    key = repo_name

    if platform and dirname:
        key = '_'.join([repo_name, dirname, platform])
    elif platform:
        key = '_'.join([repo_name, platform])

    return S3_REPO_TO_DIR[key]
