# coding=utf-8
import json
import logging
import os
import ast

import sandbox.common.types.resource as ctr
from sandbox.common.types import task as ctt
from sandbox import common
from sandbox import sdk2
from sandbox.projects.sandbox.sandbox_lxc_image import RichTextTaskFailure
from sandbox.projects.sandbox_ci.managers import Reports
from sandbox.sdk2.helpers import subprocess as sp


# FIXME: https://st.yandex-team.ru/SANDBOX-4063
def get_resource_http_proxy_link(res):
    return common.rest.Client().resource[res.id].read()["http"]["proxy"]


class NotPlatformBuildErrorArc(RuntimeError):
    pass


def rich_check_report(proc, log, task, alias, check_return_code=True):
    with log:
        logfile = "{}.out.log".format(alias)
        proc.communicate()

        if proc.returncode == 0:
            return True

        error_msg = "Command {!r} failed, see the log below. <b><a href='{}'>{}</a></b>".format(
            alias,
            "/".join((get_resource_http_proxy_link(task.log_resource), logfile)),
            logfile
        )

        if not check_return_code:
            task.set_info(error_msg, do_escape=False)
            return True

        raise RichTextTaskFailure(task, error_msg)


def rich_check_start(command, task, alias, env=os.environ, cwd=None):
    log = sdk2.helpers.ProcessLog(task, alias)
    proc = sp.Popen(command, stdout=log.stdout, stderr=log.stdout, env=env, cwd=cwd)
    return (proc, log)


def rich_check_call(command, task, alias, env=os.environ, cwd=None, check_return_code=True):
    (proc, log) = rich_check_start(command, task, alias, env, cwd)
    return rich_check_report(proc, log, task, alias, check_return_code)


def rich_get_output(command, task, alias, env=os.environ, cwd=None):
    logfile = "{}.out.log".format(alias)

    with sdk2.helpers.ProcessLog(task, alias) as log:
        proc = sp.Popen(command, stdout=sp.PIPE, stderr=log.stdout, env=env, cwd=cwd)
        (stdout, stderr) = proc.communicate()

        if proc.returncode == 0:
            return stdout

        raise RichTextTaskFailure(
            task,
            "Command {!r} failed, see the log below. <b><a href='{}'>{}</a></b>".format(
                alias,
                "/".join((get_resource_http_proxy_link(task.log_resource), logfile)),
                logfile),
        )


def make_browser_semaphore_parameters(prefix=''):
    class BrowserSemaphoreParametersArc(sdk2.Task.Parameters):
        sdk2.helpers.set_parameter(
            '{}sem_capacity'.format(prefix),
            sdk2.parameters.Integer(
                'Capacity',
                default=1000,
                description='Количество браузеров, доступных для одновременной работы',
            )
        )
        sdk2.helpers.set_parameter(
            '{}sem_weight'.format(prefix),
            sdk2.parameters.String(
                'Weight',
                description='"Вес" семафора; '
                            'Будет использовано, если в genisys/параметре browsers не указан sessionsPerBrowser'
            )
        )
        sdk2.helpers.set_parameter(
            '{}browser_name'.format(prefix),
            sdk2.parameters.String(
                'Имя браузера',
                description='Фактическое название семафора будет '
                            '"market-autotests-browser_<browser_name>_<browser_version>"; '
                            'Будет использовано, если не указано в genisys/параметре browsers'
            )
        )
        sdk2.helpers.set_parameter(
            '{}browser_version'.format(prefix),
            sdk2.parameters.String(
                'Версия браузера',
                description='Фактическое название семафора будет '
                            '"market-autotests-browser_<browser_name>_<browser_version>"; '
                            'Будет использовано, если не указано в genisys/параметре browsers'
            )
        )

    return BrowserSemaphoreParametersArc


def make_browser_capabilities_parameters(prefix=''):
    class BrowserCapabilitiesParametersArc(sdk2.Task.Parameters):
        sdk2.helpers.set_parameter(
            '{}browser_id'.format(prefix),
            sdk2.parameters.String(
                'ID',
                default='',
                description='ID браузера для гермионы, использутся в переменной окружения'
            )
        )

        sdk2.helpers.set_parameter(
            '{}browser_name'.format(prefix),
            sdk2.parameters.String(
                'Browser name',
                default='',
                description='Фактическое название семафора будет "market-autotests-browser_<name>_<version>"'
            )
        )

        sdk2.helpers.set_parameter(
            '{}browser_version'.format(prefix),
            sdk2.parameters.String(
                'Version',
                default='',
                description='Фактическое название семафора будет "market-autotests-browser_<name>_<version>"'
            )
        )

        sdk2.helpers.set_parameter(
            '{}sessions_per_browser'.format(prefix),
            sdk2.parameters.Integer(
                'Sessions per browser',
                description='Сколько браузерных сессий будет запущено. Так же вес запущенной таски для семафора'
            )
        )

        sdk2.helpers.set_parameter(
            '{}other_desired_capabilities'.format(prefix),
            sdk2.parameters.Dict('Other desired capabilities')
        )

    return BrowserCapabilitiesParametersArc


def unpack_resource(task, resource, unpack_path):
    resource_data = sdk2.ResourceData(resource)

    logging.debug('Unpacking resource, path: {} to {}'.format(resource_data.path, unpack_path))

    if not os.path.exists(unpack_path):
        os.makedirs(unpack_path)

    rich_check_call(
        ["tar", "--overwrite", "-xzf", "{}".format(resource_data.path), "-C", unpack_path],
        task=task, alias="unpack_app", cwd=task.root_dir
    )


def report_data(resource):
    status = resource['attributes']['status']
    # see https://st.yandex-team.ru/FEI-4998
    if resource['state'] == ctr.State.BROKEN:
        status = ctr.State.BROKEN

    return {
        'status': status,
        'url': resource['http']['proxy'],
        'report_type': resource['attributes']['type'],
        'root_path': resource['attributes'].get('root_path', None),
        'description': resource['attributes'].get('report_description', None),
    }


def format_header(status, url, report_type, root_path, description):
    return {
        'Status': Reports.status_badge(status),
        'Report': Reports.description(url, report_type, root_path, description),
    }


def prepare_browsers_semaphores(task):
    DEFAULT_BROWSER_SEMAPHORE_CAPACITY = 2000
    DEFAULT_BROWSER_SEMAPHORE_WEIGHT = 1

    if not task.Parameters.use_browser_semaphore:
        return

    actual_browsers = task._get_browsers()

    semaphore_weight = 0
    if actual_browsers:
        for browser_id, capabilities in actual_browsers.items():
            actual_capabilities = capabilities

            if isinstance(capabilities, str) or isinstance(capabilities, unicode):
                actual_capabilities = ast.literal_eval(capabilities)

            semaphore_weight += actual_capabilities.get('sessionsPerBrowser', DEFAULT_BROWSER_SEMAPHORE_WEIGHT)

    task.Requirements.semaphores = ctt.Semaphores(
        acquires=[
            ctt.Semaphores.Acquire(
                name="market-autotests-browser",
                capacity=DEFAULT_BROWSER_SEMAPHORE_CAPACITY,
                weight=semaphore_weight or DEFAULT_BROWSER_SEMAPHORE_WEIGHT
            )
        ],
        release=(
            ctt.Status.Group.BREAK, ctt.Status.Group.FINISH
        )
    )


def prepare_ginny_user_config(task, family='hermione'):
    """Если в параметрах переданы браузеры, то создаем конфиг и проставляем соответствующую переменную окружения.
    Сделано для возможности запускаться с полностью кастомным браузером, т.к. browsers из юзер-конфига полностью
    перетирают секцию browsers из проектного конфига.
    """
    GINNY_USER_CONFIG_PATH = 'ginny.user.conf.js'

    browsers = task._get_browsers()
    actual_browsers = {}

    for browser_id, capabilities in browsers.items():
        if isinstance(capabilities, str) or isinstance(capabilities, unicode):
            actual_browsers[browser_id] = ast.literal_eval(capabilities)
        else:
            actual_browsers[browser_id] = capabilities

    if not actual_browsers:
        logging.info("Can't find browsers customization, won't create ginny user config")
        return

    user_config = 'module.exports = {}'.format(
        json.dumps({family: {'browsers': actual_browsers}})
    )
    user_config_path = os.path.join(str(task.app_src_path), GINNY_USER_CONFIG_PATH)
    logging.info("Ginny user config: path: {}\nconfig: {}\n".format(user_config_path, user_config))

    with open(user_config_path, 'w') as f:
        f.write(user_config)

    os.environ['GINNY_USER_CONFIG_PATH'] = user_config_path
