# -*- coding: utf-8 -*-
import json
import re
from sandbox import sdk2


class FixVersion(sdk2.parameters.String):
    @classmethod
    def cast(cls, value):
        value = super(FixVersion, cls).cast(value)
        if value is None or not value and not cls.required:
            return value

        if re.match("^\d{4}\.\d+(?:\.\d+)?[md]?$", value) is None:
            raise ValueError("Fix version must be in format `YYYY.R.PS`")
        return value


class ReleaseTicket(sdk2.parameters.String):
    @classmethod
    def cast(cls, value):
        value = super(ReleaseTicket, cls).cast(value)
        if not cls.required and not value:
            return value

        msg = "Release ticket must be in format `QUEUENAME-ID`"
        assert re.match("^[A-Z]+-\d+$", value) is not None, msg
        return value


def build_test_statistic(tool, pack, data, report_url, task_url):
    hidden_id_value = '{}-{}-report'.format(tool, pack)
    caption = 'Статистика прогона {}-тестов, пак %%{}%%:'.format(tool, pack)
    footer = '\n- (({} Отчет))\n- (({} Таск))\n{}'.format(report_url, task_url, hidden_id(hidden_id_value))
    table = """{}#|{}{}|#{}
<!--<# <div id="consolidated-reports-comment" /> #>-->"""
    head = '**||Browser|Status|Tests|Passed|Failed|Broken|Skipped|Retries|Duration||**'
    row = '||{browser}|{status}|{tests}|{passed}|{failed}|{broken}|{skipped}|{retries}|{duration}||'
    body = ''
    cell = '!!({}){}!!'

    for run in data:
        duration = divmod(run['duration'] // 1000, 60)
        [hours, minutes] = divmod(duration[0], 60)
        seconds = duration[1]

        body += row.format(
            browser=run['browser'],
            status=cell.format('red' if run['status'] == 'failed' else 'green', run['status']),
            tests=cell.format('blue', run['tests']),
            passed=cell.format('green', run['passed']),
            failed=cell.format('red', run['failed']),
            broken=cell.format('yellow', run.get('broken', '-')),
            skipped=cell.format('gray', run['skipped']),
            retries=cell.format('yellow', run['retries']),
            duration='{h}:{m}:{s}'.format(h=hours, m=minutes, s=seconds)
        )

    return table.format(caption, head, body, footer)


def build_result_cms(data, task_url, report_url):
    is_success = data['success'] is True
    is_not_empty = 'schemaName' in data and 'schemaRevisionId' in data

    if is_not_empty:
        footer = '\nРевизия: %%{}#{}%%, (({} таск)).'.format(data['schemaName'], data['schemaRevisionId'], task_url)
    else:
        footer = '\n(({} Таск.))'.format(task_url)
    footer += '\n{}'.format(hidden_id('cms-report'))

    caption = '===== Результат сохранения JSON-схемы в CMS: {}'
    caption_text_success = '!!(green)УСПЕХ!!!'
    caption_text_error = '!!(red)ФИАСКО!!!'
    caption_text_empty = '!!(yellow)НЕЧЕГО СОХРАНЯТЬ!!!'

    caption_text = caption_text_error if not is_success else\
        caption_text_success if is_not_empty else\
        caption_text_empty

    result = caption.format(caption_text)

    if not data["success"] and "errorMessage" in data:
        result += '\nОшибка: %%{}%%'.format(data["errorMessage"])
        result += footer

        return result

    if is_not_empty:
        result += '\n(({} Сохранённые типы)) (см. поле %%nodeTypes%%)'.format(report_url)

    result += _build_result_cms_table(data['errors'], 'Надо исправить ошибки во flow-типах:')
    result += _build_result_cms_table(data['warnings'], 'Обрати внимание на предупреждения:')
    result += footer

    return result


def _build_result_cms_table(data, caption):
    table = '{}\n#|{}{}|#'
    table_caption = '\n===== {}\n'
    table_head = '**||Схема|Сообщение||**\n'
    table_body = ''
    table_row = '||%%{reference}%%|{message}||\n'

    if not len(data):
        return ''

    for item in data:
        ref = item['ref']

        table_body += table_row.format(
            reference=ref,
            message=item['message']
        )

    result = table.format(
        table_caption.format(caption),
        table_head,
        table_body
    )

    return result


def colored_text(color, text):
    return '!!({color}){text}!!'.format(color=color, text=text)


def link_text(ref, text):
    return '(({ref} {text}))'.format(ref=ref, text=text)


def cut_text(cut_header, hidden_text):
    return '<{{{cut_header}\n%%{hidden_text}%%}}>'.format(cut_header=cut_header, hidden_text=hidden_text)


def hidden_id(value):
    return '<# <div id="{id}"/> #>'.format(id=value)
