# -*- coding: utf-8 -*-

"""
@author: mvshmakovmv@yandex-team.ru
Модуль сборки бота Hroom в ресурс
Подробная информация: https://wiki.yandex-team.ru/market/marketlife/hr/hroom/
"""

from sandbox import sdk2
import sandbox.sdk2.path as spath

from sandbox.sdk2.vcs.git import Git
import os.path
import re
import pipes
import tempfile
import sandbox.common.types.misc as ctm
from sandbox.projects.common.nanny import nanny

from sandbox.projects.market.front.helpers.sandbox_helpers import rich_check_call
from sandbox.projects.market.front.helpers.node import create_node_selector
from sandbox.projects.market.front.helpers.ubuntu import create_ubuntu_selector, setup_container
from sandbox.projects.market.front.helpers.MetatronEnv import MetatronEnv

from sandbox.projects.market.resources import MARKET_HROOM_APP

HROOM_REPO = "git://github.yandex-team.ru/market/hroom.git"


class MarketBuildHroom(nanny.ReleaseToNannyTask2, sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group("Приложение") as app:
            app_repo = sdk2.parameters.String(
                "Репозиторий",
                default_value=HROOM_REPO,
                required=True
            )
            app_branch = sdk2.parameters.String(
                "Ветка",
                default_value="master",
                required=True
            )
            app_revision = sdk2.parameters.String(
                "Git revision",
                required=False,
            )
            ubuntu_version = create_ubuntu_selector()
            node_version = create_node_selector()
            prune = sdk2.parameters.Bool("Делать prune зависимостей", default=True, required=True)

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64

    def on_enqueue(self):
        super(MarketBuildHroom, self).on_enqueue()
        setup_container(self)

    @property
    def _app_repo_name(self):
        app_repo_match = re.search(".+/([^/.]+?)(:?.git)?$", str(self.Parameters.app_repo))

        assert app_repo_match is not None

        return app_repo_match.group(1)

    def on_execute(self):
        super(MarketBuildHroom, self).on_execute()

        with MetatronEnv(self, nodejs_version=self.Parameters.node_version):
            root_dir = tempfile.mkdtemp()
            app_src_path = os.path.join(root_dir, self._app_repo_name)

            self._clone_repo(app_src_path)
            self._build_app(app_src_path)
            self._create_app_pack(app_src_path)

    def _clone_repo(self, app_src_path):
        git = Git(pipes.quote(self.Parameters.app_repo))
        git.clone(str(app_src_path), pipes.quote(self.Parameters.app_branch), commit=self.Parameters.app_revision)

    def _build_app(self, app_src_path):
        rich_check_call(
            ["npm", "run", "ci:build"],
            task=self, alias="npm_build", cwd=app_src_path
        )

        self._prune_deps(app_src_path)

    def _prune_deps(self, app_src_path):
        if self.Parameters.prune:
            rich_check_call(
                ["npm", "prune", "--production"],
                task=self, alias="npm_prune", cwd=app_src_path
            )

    def _create_app_pack(self, app_src_path):
        pack_app_archive_path = tempfile.mktemp(suffix=".tar.gz", prefix=self._app_repo_name)

        rich_check_call(
            ["tar", "-C", os.path.join(app_src_path, '..'), "-czf", pack_app_archive_path, self._app_repo_name],
            task=self, alias="create_app_pack"
        )

        self._app_create_resource(pack_app_archive_path)

    def _app_create_resource(self, pack_app_archive_path):
        app_res = sdk2.ResourceData(MARKET_HROOM_APP(
            self, "Hroom tarball", "{}.tar.gz".format(self._app_repo_name)
        ))

        app_res.path.write_bytes(spath.Path(pack_app_archive_path).read_bytes())
        app_res.ready()
