import os
import logging
import json

from sandbox import sdk2
from sandbox.common.rest import Client as SandboxClient
from sandbox.common.auth import OAuth as SandboxOAuth
from sandbox.projects.common import binary_task
from sandbox.projects.common.arcadia import sdk as arcadiasdk

from . import juggler
from datetime import datetime, timedelta

logger = logging.getLogger()

TAG = "MARKET_UNIVERSAL_SCHEDULER"

JUGGLER_HOSTS = {
    "production": "mi-datacamp-united",
    "testing": "mi-datacamp-united-testing",
}


class SchedulerInfo:
    def __init__(self, actual_cfg=None, upcoming_cfg=None, juggler_cfg=None, task_name=None):
        self.actual_cfg = actual_cfg
        self.upcoming_cfg = upcoming_cfg
        self.juggler_cfg = juggler_cfg
        self._task_name = task_name

    @property
    def task_name(self):
        if self._task_name:
            return self._task_name
        if self.actual_cfg:
            return get_task_name(self.actual_cfg['task'])
        if self.upcoming_cfg:
            return get_task_name(self.upcoming_cfg['task'])

        return None

    def merge(self, other):
        if other.actual_cfg:
            self.actual_cfg = other.actual_cfg
        if other.upcoming_cfg:
            self.upcoming_cfg = other.upcoming_cfg
        if other.juggler_cfg:
            self.juggler_cfg = other.juggler_cfg

    def set_start_time_for_task(self):
        if self.upcoming_cfg is None:
            return
        if self.actual_cfg and self.actual_cfg['schedule'].get('start_time'):
            self.upcoming_cfg['schedule']['start_time'] = self.actual_cfg['schedule']['start_time']
        else:
            self.upcoming_cfg['schedule']['start_time'] = (datetime.utcnow() + timedelta(minutes=1)).strftime('%Y-%m-%dT%H:%M:%S')

    @staticmethod
    def from_cfg(task_name, gen_params, env, author):
        if gen_params.get("type") == "ROUTINES_TASK":
            scheduler_params = generate_routines_scheduler_params(gen_params['sandbox'], env, author)
        elif gen_params.get("type") == "SAAS_PUSH":
            scheduler_params = generate_saas_push_scheduler_params(gen_params['sandbox'], env, author)
        else:
            scheduler_params = gen_params["sandbox"]
            scheduler_params['author'] = author

        if 'tags' not in scheduler_params['task']:
            scheduler_params['task']['tags'] = []

        scheduler_params['task']['tags'].extend([env.upper(), TAG, task_name.upper()])

        return SchedulerInfo(upcoming_cfg=scheduler_params, juggler_cfg=gen_params.get('juggler'))


def get_current_schedulers_from_sandbox(sandbox_client, author, env):
    search_params = {
        "limit": 100,
        "author": author,
        "tags": [TAG, env.upper()],
        "all_tags": "true",
    }

    ids = [scheduler['id'] for scheduler in sandbox_client.scheduler.read(search_params)["items"]]

    # for migration
    if author in ('robot-mrkt-dc-r-tst', 'robot-mrkt-dc-r-prod'):
        search_params['author'] = 'robot-mrkt-idx-sb'
        ids.extend([scheduler['id'] for scheduler in sandbox_client.scheduler.read(search_params)["items"]])

    return [SchedulerInfo(actual_cfg=sandbox_client.scheduler[i].read()) for i in ids]


def generate_schedulers_from_config(cfg, env, author, filter=None):
    from copy import deepcopy
    from deepmerge import always_merger

    schedulers = []
    for name, params in cfg['tasks'].items():
        if filter is not None and name != filter:
            continue

        merged_params = deepcopy(cfg.get('default', {}))
        always_merger.merge(merged_params, params)

        schedulers.append(SchedulerInfo.from_cfg(name, merged_params, env, author))

    return schedulers


def merge_schedulers_info(actual_schedulers, upcoming_schedulers):
    result = actual_schedulers

    for upcoming in upcoming_schedulers:
        found_in_current = False
        for current in result:
            if upcoming.task_name == current.task_name:
                current.upcoming_cfg = upcoming.upcoming_cfg
                current.juggler_cfg = upcoming.juggler_cfg
                found_in_current = True
                break

        if not found_in_current:
            result.append(upcoming)

    for scheduler_info in result:
        scheduler_info.set_start_time_for_task()

    return result


def generate_routines_scheduler_params(rcfg, env, author):
    cfg = {
        'owner': rcfg.get('owner', 'MARKET-IDX'),
        'author': author,
        'schedule': {
            'sequential_run': rcfg.get('sequential_run', False),
            'repetition': {
                'interval': rcfg['interval'],
            },
        },
        'task': {
            'type': 'MARKET_RUN_UNIVERSAL_BUNDLE',
            'owner': rcfg.get('owner', 'MARKET-IDX'),
            'kill_timeout': rcfg.get('kill_timeout', 10800),
            'priority': {'class': 'SERVICE', 'subclass': 'HIGH'},
            'custom_fields': [
                {'name': 'bundle_name', 'value': 'routines-sandbox'},
                {'name': 'task_name', 'value': 'routines-{}'.format(rcfg["task_name"].lower())},
                {'name': 'environment', 'value': env},
                {'name': 'cmd', 'value': [
                    '{{cwd}}/bin/routines',
                    '-c',
                    '{{cwd}}/conf/routines/common.ini,{{cwd}}/conf/routines/conf-available/{{environment}}.white.ini,{{its}}',
                    '--task',
                    rcfg['task_name'],
                    '--dc',
                    '{{dc}}',
                ]},
                {'name': 'use_its_config', 'value': 'True'},
                {'name': 'its_path', 'value': '{{cwd}}/conf/routines/its.ini'},
                {'name': 'its_url', 'value': 'https://its.yandex-team.ru/v1/values/market/datacamp/routines/{}-sandbox/market_datacamp_settings/'.format(env)},
                {'name': 'its_secret_name', 'value': 'nanny-oauth'},
                {'name': 'send_metrics_to_solomon', 'value': 'True'},
                {'name': 'solomon_project', 'value': 'market.datacamp'},
                {'name': 'solomon_cluster', 'value': env},
                {'name': 'solomon_service', 'value': 'routines'},
                {'name': 'solomon_secret_name', 'value': 'solomon-oauth'},
                {'name': 'solomon_extra_labels', 'value': {
                    'task': rcfg["task_name"]
                }}
            ],
        },
    }
    if 'binary_executor_release_type' in rcfg:
        cfg['task']['custom_fields'].append({'name': 'binary_executor_release_type', 'value': rcfg['binary_executor_release_type']})

    if rcfg.get('fetch_datagetter', False):
        cfg['task']['custom_fields'].extend([
            {'name': 'use_resource', 'value': 'True'},
            {'name': 'resources', 'value': {
                'MARKET_DATA_SHOPSDAT': '{{cwd}}/data-getter-mbi/shopsdat'
            }}
        ])

    if rcfg.get('extra_resources', []):
        cfg['task']['custom_fields'].extend([
            {
                'name': 'use_resource',
                'value': 'True'
            },
            {
                'name': 'resources',
                'value': {data['resource']: data['dest'] for data in rcfg['extra_resources']}
            }
        ])

    if 'requirements' in rcfg:
        if 'requirements' not in cfg['task']:
            cfg['task']['requirements'] = {}
        for k, v in rcfg['requirements'].items():
            cfg['task']['requirements'][k] = v

    return cfg


def generate_saas_push_scheduler_params(rcfg, env, author):
    cfg = {
        'owner': rcfg.get('owner', 'MARKET-IDX'),
        'author': author,
        'schedule': {
            'sequential_run': rcfg.get('sequential_run', False),
            'repetition': {
                'interval': rcfg['interval'],
            },
        },
        'task': {
            'type': 'MARKET_RUN_UNIVERSAL_BUNDLE',
            'owner': rcfg.get('owner', 'MARKET-IDX'),
            'kill_timeout': rcfg.get('kill_timeout', 10800),
            'priority': {'class': 'SERVICE', 'subclass': 'HIGH'},
            'custom_fields': [
                {'name': 'bundle_name', 'value': 'saas-push'},
                {'name': 'task_name', 'value': rcfg["task_name"].lower()},
                {'name': 'environment', 'value': env},
                {'name': 'cmd', 'value': [
                    '{{cwd}}/bin/saas_push',
                    'yt',
                    '-c',
                    rcfg['config'],
                    '--yt-proxy',
                    'arnold',
                    '--yt-table-path',
                    rcfg['yt_table_path'],
                    '--alias',
                    rcfg['alias'],
                    '--threads-count',
                    rcfg['threads_count'],
                    '--rate-limit',
                    rcfg['rate_limit']
                ]},
                {'name': 'additional_env', 'value' : {
                    'YT_TOKEN': rcfg['YT_TOKEN'],
                    'TVM_SECRET': rcfg['TVM_SECRET']
                }},
            ],
        },
    }
    if 'binary_executor_release_type' in rcfg:
        cfg['task']['custom_fields'].append({'name': 'binary_executor_release_type', 'value': rcfg['binary_executor_release_type']})

    return cfg



def get_task_name(task_params):
    for j in task_params['custom_fields']:
        if j["name"] == "task_name":
            return j["value"]


def generate_schedulers(sandbox_client, juggler_client, schedulers_info, env, dry_run=False):
    for scheduler_info in schedulers_info:
        if scheduler_info.upcoming_cfg is None:
            logger.info('Going to remove scheduler name=%s, id=%s', scheduler_info.task_name, scheduler_info.actual_cfg['id'])
            if dry_run:
                continue

            sandbox_client.scheduler[scheduler_info.actual_cfg['id']].delete()
            if juggler_client:
                juggler_client.remove_check('{}-status'.format(scheduler_info.task_name), JUGGLER_HOSTS[env])

            continue

        if scheduler_info.actual_cfg is not None and scheduler_info.upcoming_cfg is not None:
            logger.info('Going to update scheduler name=%s, id=%s, upcoming state:\n%s', scheduler_info.task_name, scheduler_info.actual_cfg['id'], json.dumps(scheduler_info.upcoming_cfg, indent=2))
            if dry_run:
                continue

            sandbox_client.scheduler[scheduler_info.actual_cfg['id']].update(scheduler_info.upcoming_cfg)
            if juggler_client:
                juggler_client.add_or_update('{}-status'.format(scheduler_info.task_name), JUGGLER_HOSTS[env], scheduler_info.juggler_cfg['ttl'])

            continue

        if scheduler_info.upcoming_cfg:
            logger.info('Going to create scheduler name=%s, upcoming state:\n%s', scheduler_info.task_name, json.dumps(scheduler_info.upcoming_cfg, indent=2))
            if dry_run:
                continue

            req = {
                'data': scheduler_info.upcoming_cfg,
                'task_type': scheduler_info.upcoming_cfg['task']['type']
            }
            scheduler_id = sandbox_client.scheduler(req)["id"]
            sandbox_client.batch.schedulers.start.update(scheduler_id)
            if juggler_client:
                juggler_client.add_or_update('{}-status'.format(scheduler_info.task_name), JUGGLER_HOSTS[env], scheduler_info.juggler_cfg['ttl'])


class GenIdxUniversalSchedulers(binary_task.LastBinaryTaskKosherReleaseMixin, sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        environment = sdk2.parameters.String(
            'Environment',
            required=True,
            choices=[(x, x) for x in ('testing', 'production')],
            default_value='testing'
        )
        author = sdk2.parameters.String(
            'Schedulers author',
            required=True,
            default_value='robot-mrkt-idx-sb'
        )

        arcadia_url = sdk2.parameters.String(
            'Arcadia url',
            required=False,
            default_value=sdk2.svn.Arcadia.ARCADIA_TRUNK_URL
        )
        config_path = sdk2.parameters.String(
            'Schedulers params(path in arcadia)',
            required=True
        )

        generate_juggler_checks = sdk2.parameters.Bool(
            'Generate checks in juggler',
            default_value=False
        )

        sandbox_api_token = sdk2.parameters.YavSecret(
            'Sandbox OAuth token'
        )

        juggler_api_token = sdk2.parameters.YavSecret(
            'Juggler OAuth token'
        )

        dry_run = sdk2.parameters.Bool(
            'Run without modifying state, just log actions',
            default_value=False
        )

        ext_params = binary_task.binary_release_parameters(stable=True)

    def on_save(self):
        super(GenIdxUniversalSchedulers, self).on_save()

        if self.Parameters.author == 'robot-mrkt-idx-sb':
            self.Parameters.juggler_api_token = sdk2.yav.Secret('sec-01g4hx5hphh4ng7s9fwxe8s6e4', 'ver-01g4hx5hpt9722trpqmd1kka7m', 'juggler-token')
            self.Parameters.sandbox_api_token = sdk2.yav.Secret('sec-01g4hx5hphh4ng7s9fwxe8s6e4', 'ver-01g4hx5hpt9722trpqmd1kka7m', 'sandbox-token')
            if self.Parameters.environment == 'testing':
                self.Parameters.author = 'robot-mrkt-dc-r-tst'
            else:
                self.Parameters.author = 'robot-mrkt-dc-r-prod'

    def on_execute(self):
        with arcadiasdk.mount_arc_path(self.Parameters.arcadia_url) as aarcadia:
            config_path = os.path.join(aarcadia, self.Parameters.config_path)

            with open(config_path, "r") as f:
                cfg = json.loads(f.read())

        env = self.Parameters.environment

        sandbox_client = SandboxClient(auth=SandboxOAuth(token=self.Parameters.sandbox_api_token.value()))

        juggler_client = None
        if self.Parameters.generate_juggler_checks:
            juggler_client = juggler.JugglerAPI(self.Parameters.juggler_api_token.value(), env)

        current_schedulers = get_current_schedulers_from_sandbox(sandbox_client, self.Parameters.author, env)
        upcoming_schedulers = generate_schedulers_from_config(cfg, env, self.Parameters.author, None)

        schedulers_info = merge_schedulers_info(current_schedulers, upcoming_schedulers)
        generate_schedulers(
            sandbox_client, juggler_client, schedulers_info, env, self.Parameters.dry_run
        )
