import logging
import json
import datetime
import calendar
from sandbox import sdk2
from sandbox.projects.common.binary_task import LastBinaryTaskRelease, binary_release_parameters
from sandbox.projects.common.task_env import TinyRequirements
from sandbox.projects.market.idx.JugglerStatsUpdater.config import JUGGLER_STATS_TO_SOLOMON, JUGGLER_ESCALATIONS_TO_SOLOMON


def to_timestamp(dt):
    return int(calendar.timegm(dt.utctimetuple()))


class JugglerStatsUpdater(LastBinaryTaskRelease, sdk2.Task):
    class Requirements(TinyRequirements):
        pass

    class Parameters(sdk2.Parameters):
        solomon_token = sdk2.parameters.YavSecret(
            "YAV secret for solomon token (with optional version)",
            default="sec-01dsfcm45eskn30121gqrb99zq#solomon-oauth",
        )
        binary_params = binary_release_parameters(stable=True)
        send_status_stats = sdk2.parameters.Bool("Send OK/WARN/ERROR metrics to solomon", default=True)
        send_escalations_stats = sdk2.parameters.Bool("Send number of escalations happened yesterday", default=False)

    def on_execute(self):
        from market.idx.pylibrary.juggler import JugglerClient

        client = JugglerClient()

        if self.Parameters.send_escalations_stats:
            self._send_escalations_stats(client)

        if self.Parameters.send_status_stats:
            self._send_status_stats(client)

    def _send_escalations_stats(self, client):
        from library.python.monlib.encoder import dumps

        yesterday = datetime.datetime.today() - datetime.timedelta(days=1)
        since = datetime.datetime(year=yesterday.year, month=yesterday.month, day=yesterday.day)
        until = since + datetime.timedelta(days=1)
        since_ts = to_timestamp(since)
        until_ts = to_timestamp(until)
        for config in JUGGLER_ESCALATIONS_TO_SOLOMON:
            registry = self._gather_escalations(
                client,
                since=since_ts,
                until=until_ts,
                filters=config["juggler"]["filters"],
                common_labels=config["solomon"]["common_labels"],
                metric_labels=config["solomon"]["metric_labels"],
            )
            metric_timestamp = since + (until - since) / 2

            data = dumps(registry, format="spack", timestamp=metric_timestamp)
            self._send_metrics(
                data,
                project=config["solomon"]["project"],
                cluster=config["solomon"]["cluster"],
                service=config["solomon"]["service"],
            )

    @staticmethod
    def _dumps_with_ts(registry, ts):

        return

    @staticmethod
    def _gather_escalations(client, since, until, filters, common_labels, metric_labels):
        from library.python.monlib.metric_registry import MetricRegistry

        logging.info("will count escalations since {} until {}".format(since, until))
        result = client.get_escalations_log(
            filters=filters,
            only_running=False,
            page=0,
            page_size=100,
        )

        registry = MetricRegistry(common_labels)
        escalations = registry.int_gauge(metric_labels)
        counter = 0
        for escalation in result["escalations"]:
            if escalation["start_time"] >= since and escalation["start_time"] < until:
                counter += 1
        escalations.set(counter)

        return registry

    def _send_status_stats(self, client):
        from library.python.monlib.encoder import dumps

        failed_dashboards = []
        for config in JUGGLER_STATS_TO_SOLOMON:
            name = config.get("name", "UNDEFINED")
            logging.info("processing dashboard {}".format(name))
            try:
                registry = self._gather_dashboard_metrics(
                    client,
                    project=config["juggler"]["project"],
                    filters=config["juggler"]["filters"],
                    common_labels=config["solomon"],
                )
                data = dumps(registry, format="spack")
                self._send_metrics(
                    data,
                    project=config["solomon"]["project"],
                    cluster=config["solomon"]["cluster"],
                    service=config["solomon"]["service"],
                )
            except:
                logging.exception("failed to process dashboard {}".format(name))
                failed_dashboards.append(name)

        if failed_dashboards:
            raise RuntimeError("failed to update stats for dashboards: {}".format(', '.join(failed_dashboards)))

    @staticmethod
    def _gather_dashboard_metrics(client, project, filters, common_labels):
        from library.python.monlib.metric_registry import MetricRegistry

        checks_state = client.get_checks_state(
            filters=filters,
            include_mutes=True,
            limit=2,
            project=project,
            sort={"field": "DEFAULT", "order": "DESC"},
        )

        registry = MetricRegistry(common_labels)
        for status in checks_state["statuses"]:
            c = registry.int_gauge({"status": status["status"]})
            c.set(status["count"])

        return registry

    def _send_metrics(self, data, project, cluster, service):
        import requests

        solomon_token = self.Parameters.solomon_token.data()[self.Parameters.solomon_token.default_key]
        headers = {'Content-Type': 'application/x-solomon-spack', 'Authorization': 'OAuth {}'.format(solomon_token)}
        response = requests.post(
            'https://solomon.yandex.net/api/v2/push?project={}&cluster={}&service={}'.format(project, cluster, service),
            data=data,
            headers=headers,
        )
        self._log_response(response)

    @staticmethod
    def _log_response(response):
        import requests

        if response.status_code != requests.codes.ok:
            logging.error('Failed to push metrics: {} {}'.format(response.status_code, response.text))
        else:
            try:
                logging.info('solomon response: {}'.format(json.dumps(response.json(), indent=2)))
            except requests.exceptions.JSONDecodeError:
                logging.info('solomon response: {}'.format(response.text))
