# coding: utf8

from sandbox import common
from sandbox import sdk2
from sandbox.common.types import client as ctc
from sandbox.projects.common import binary_task, solomon
from sandbox.projects.common.juggler import jclient
from sandbox.projects.market.idx.MarketRunUniversalBundle import helpers
from sandbox.projects.market.resources import MARKET_IDX_UNIVERSAL_BUNDLE


from sandbox.common.types.task import Status as TaskStatus


import json
import logging
import os
import psutil
import requests
from time import time
import six.moves.configparser as ConfigParser


PRODUCTION = "production"
TESTING = "testing"
AVAILABLE_ENVS = (PRODUCTION, TESTING)

RELEASE_STATUSES = {
    PRODUCTION: "stable",
    TESTING: "testing",
}

JUGGLER_HOSTS = {
    PRODUCTION: "mi-datacamp-united",
    TESTING: "mi-datacamp-united-testing",
}


logger = logging.getLogger(__name__)


class Timer(object):
    def __init__(self, task, name):
        self.task = task
        self.name = name

    def __enter__(self):
        self.task.timer_start(self.name)

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.task.timer_end(self.name)


class TimeHelpers(object):
    @property
    def timers(self):
        if not self.Context.timers:
            self.Context.timers = dict()

        return self.Context.timers

    def timer(self, name):
        return Timer(self, name)

    def timer_start(self, name):
        self.timers[name] = dict(start=time())

    def timer_end(self, name):
        if name not in self.timers:
            return

        self.timers[name]['end'] = time()


class MarketRunUniversalBundle(binary_task.LastBinaryTaskKosherReleaseMixin, sdk2.Task, TimeHelpers):
    class Parameters(sdk2.Task.Parameters):
        task_name = sdk2.parameters.String("Task name")
        bundle_name = sdk2.parameters.String("Bundle name", required=True)
        cmd = sdk2.parameters.List("Command line arguments", required=True)
        secret = sdk2.parameters.YavSecret("Yav token", required=False)
        additional_env = sdk2.parameters.Dict("Additional environment variables", required=False, description='''
            Environment variables can be set with
                yav:secret_id:version_id[optional]:key or value
            In the first case key value of secret(secret_id, version_id) will be taken,
            in second case input value will be set.
        ''')

        with sdk2.parameters.Group("Overrides") as overrides_group:
            override_basic_params = sdk2.parameters.Bool(
                "Override basic params",
                default_value=False
            )

            with override_basic_params.value[True]:
                overridden_bundle_resource = sdk2.parameters.Resource("Bundle resource")
                override_juggler_host = sdk2.parameters.String("Juggler host")
                override_juggler_service = sdk2.parameters.String("Juggler service")

        with sdk2.parameters.Group("Task params") as task_params_group:
            environment = sdk2.parameters.String(
                "Environment",
                required=True,
                choices=[(x, x) for x in AVAILABLE_ENVS],
                default_value=TESTING
            )
            use_its_config = sdk2.parameters.Bool("Use fast config from ITS", default=False)
            with use_its_config.value[True]:
                its_url = sdk2.parameters.String("URL")
                its_path = sdk2.parameters.String("Path")
                its_secret_name = sdk2.parameters.String("ITS secret name")

        with sdk2.parameters.Group("Extra resources") as resource_group:
            use_resource = sdk2.parameters.Bool("Use additional resource", default=False)
            with use_resource.value[True]:
                resources = sdk2.parameters.Dict("Resources and mount points", description="To find by id enter 'id:<resource_id>'. To find by type just enter type")

        with sdk2.parameters.Group("Solomon metrics") as metrics_to_solomon:
            send_metrics_to_solomon = sdk2.parameters.Bool("Send task metrics to solomon", default=False)

            with send_metrics_to_solomon.value[True]:
                solomon_project = sdk2.parameters.String("Solomon project", required=True)
                solomon_cluster = sdk2.parameters.String("Cluster name", required=True)
                solomon_service = sdk2.parameters.String("Service", required=True)
                solomon_secret_name = sdk2.parameters.String("Solomon token", required=True)
                solomon_extra_labels = sdk2.parameters.Dict("Extra labels", required=False)

        ext_params = binary_task.binary_release_parameters(stable=True)

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        disk_space = 1024
        ram = 1024
        client_tags = (ctc.Tag.MULTISLOT | ctc.Tag.GENERIC)

        class Caches(sdk2.Requirements.Caches):
            pass

    @sdk2.header()
    def header(self):
        try:
            if self.status not in TaskStatus.Group.FINISH | TaskStatus.Group.BREAK:
                return ''
            logs = sdk2.Resource.find(
                type='TASK_LOGS',
                task_id=self.id,
                state=common.types.resource.State.READY
            ).first()

            if logs is None:
                return '<div style="font-size: 15px;"> <p> Логи не обнаружены </p> </div>'

            result = '<div style="font-size: 15px;">'
            path = 'https://proxy.sandbox.yandex-team.ru/{0}'.format(logs.id)
            result += """<p> Логи подготовки sb окружения:
                            <a href="{0}/debug.log">debug.log</a>,
                            <a href="{0}/common.log">common.log</a>
                        </p>""".format(path)

            result += '<p> Логи запуска бинарного файла: '
            if 'market_task.out.log' in self.Context.log_files:
                result += '<a href="{0}/market_task.out.log">market_task.out.log</a>, '.format(path)
            if 'market_task.err.log' in self.Context.log_files:
                result += '<a href="{0}/market_task.err.log">market_task.err.log</a>'.format(path)
            result += '</p>'

            result += '<p> DC Specific Logs: '
            if 'task_hahn' in self.Context.log_files:
                result += '<a href="{0}/task_hahn">task_hahn</a>, '.format(path)
            if 'task_arnold' in self.Context.log_files:
                result += '<a href="{0}/task_arnold">task_arnold</a>'.format(path)
            result += '</p> </div>'
            return result

        except Exception as ex:
            return getattr(ex, 'message', repr(ex))

    def get_bundle_resource(self):
        if self.Parameters.override_basic_params and self.Parameters.overridden_bundle_resource:
            return self.Parameters.overridden_bundle_resource
        else:
            return helpers.get_last_released_resource(
                MARKET_IDX_UNIVERSAL_BUNDLE,
                RELEASE_STATUSES[self.Parameters.environment],
                attrs={
                    'resource_name': self.Parameters.bundle_name
                }
            )

    def make_solomon_labels(self):
        labels = {
            'project': self.Parameters.solomon_project,
            'cluster': self.Parameters.solomon_cluster,
            'service': self.Parameters.solomon_service,
        }

        return labels

    def get_secret(self):
        if self.Parameters.environment == TESTING:
            secret = sdk2.yav.Secret("sec-01dsfcej668h548vxvxgh8vw41")
        else:
            secret = sdk2.yav.Secret("sec-01dsfcm45eskn30121gqrb99zq")

        if self.Parameters.secret:
            secret = self.Parameters.secret

        return secret

    def build_sensors(self):
        self.Context.sensors = {}
        for name, interval in self.timers.items():
            self.Context.sensors[name] = int(interval.get('end', time()) - interval['start'])

    def send_to_solomon(self):
        try:
            self.build_sensors()
            sensors = solomon.create_sensors(self.Context.sensors)
            if os.path.isfile('app-metrics.json'):
                sensors.extend(solomon.create_sensors_from_file('app-metrics.json'))
            solomon.push_to_solomon_v2(self.get_secret().data()[self.Parameters.solomon_secret_name],
                                       self.make_solomon_labels(),
                                       sensors,
                                       self.Parameters.solomon_extra_labels)
        except:
            logger.exception('Could not send metrics to solomon')

    def fetch_extra_resources(self):
        if self.Parameters.use_resource:
            for resource, resource_mount_point in dict(self.Parameters.resources).items():
                if resource.startswith("id:"):
                    resource_id = resource.lstrip("id:")
                    resource_path = sdk2.ResourceData(sdk2.Resource[resource_id]).path.as_posix()
                else:
                    last_released_resource = helpers.get_last_released_resource(str(resource), RELEASE_STATUSES[self.Parameters.environment])
                    resource_path = helpers.download_resource(last_released_resource)

                mount_point = self.render_cmd([resource_mount_point])[0]

                mount_dirname = os.path.dirname(mount_point)
                if not os.path.exists(mount_dirname):
                    os.makedirs(mount_dirname)

                logging.info('Resource path: %s', resource_path)
                if ".tar" in resource_path:
                    helpers.untar_path(resource_path, mount_point)
                else:
                    os.symlink(resource_path, mount_point)

    def on_prepare(self):
        with self.timer("on_prepare_time"):
            secret = self.get_secret()

            self.prepare_secrets(secret)

            bundle = self.get_bundle_resource()
            with self.timer("bundle_download_time"):
                bundle_path = helpers.download_resource(bundle)

            with self.timer("bundle_untar_time"):
                helpers.untar_path(bundle_path)

            self.set_info(bundle)
            self.set_info(os.path.basename(bundle_path))

            self.Context.bundle = dict(
                type=str(bundle.type),
                id=bundle.id,
            )

            with self.timer("resource_download_time"):
                self.fetch_extra_resources()

            if self.Parameters.use_its_config:
                self.save_config(self.make_ini(self.get_json_config(self.get_current_its(secret))))

    def collect_log_files(self):
        self.Context.log_files = list()
        for log_name in ("task_hahn", "task_arnold", "market_task.out.log", "market_task.err.log"):
            if self.log_path(log_name).exists() and self.log_path(log_name).stat().st_size != 0:
                self.Context.log_files.append(log_name)

    def on_execute(self):
        self.timer_end("time_before_execute")
        with self.timer("on_execute_time"):
            binary_task.LastBinaryTaskRelease.on_execute(self)

            with sdk2.helpers.ProcessRegistry as reg, sdk2.helpers.ProcessLog(self, logger="market_task") as pl:
                cmd = self.render_cmd(self.Parameters.cmd)
                env = self.get_env()
                process = sdk2.helpers.subprocess.Popen(cmd, shell=False, stdout=pl.stdout, stderr=pl.stderr, env=env)
                registered_pids = []
                while process.returncode is None:
                    process.poll()
                    for child in psutil.Process().children(recursive=True):
                        if child.pid not in registered_pids:
                            try:
                                if child.cmdline():
                                    reg.register(child.pid, child.cmdline())
                                    registered_pids.append(child.pid)
                            except psutil.NoSuchProcess:
                                logger.warn('Could not fetch process info for pid=%s', child.pid)
                if process.returncode != 0:
                    raise RuntimeError('Command {} failed with exit code {}'.format(cmd, process.returncode))

    def on_success(self, prev_status):
        status = helpers.calc_alert_status("OK", self.get_previous_run_status())
        self.send_juggler_event(status=status, description="Last run https://sandbox.yandex-team.ru/task/{} is successfull".format(sdk2.Task.current.id))
        super(MarketRunUniversalBundle, self).on_success(prev_status)

    def on_break(self, prev_status, status):
        status = helpers.calc_alert_status("CRIT", self.get_previous_run_status())
        self.send_juggler_event(status=status, description="Last run https://sandbox.yandex-team.ru/task/{} is failed".format(sdk2.Task.current.id))
        self.collect_log_files()
        if self.Parameters.send_metrics_to_solomon:
            self.send_to_solomon()
        super(MarketRunUniversalBundle, self).on_break(prev_status, status)

    def on_failure(self, prev_status):
        status = helpers.calc_alert_status("CRIT", self.get_previous_run_status())
        self.send_juggler_event(status=status, description="Last run https://sandbox.yandex-team.ru/task/{} is failed".format(sdk2.Task.current.id))
        super(MarketRunUniversalBundle, self).on_failure(prev_status)

    def on_finish(self, prev_status, status):
        self.collect_log_files()
        if self.Parameters.send_metrics_to_solomon:
            self.send_to_solomon()

    def on_enqueue(self):
        self.timer_start("time_before_execute")

    def get_env(self):
        env = {}
        for key, value in self.Parameters.additional_env.items():
            values = value.split(':')
            if values[0] == "yav":
                if len(values) == 3:
                    secret_value = sdk2.yav.Secret(secret=values[1], default_key=values[2]).value()
                    env[key] = str(secret_value)
                else:
                    secret_value = sdk2.yav.Secret(secret=values[1], version=values[2], default_key=values[3]).value()
                    env[key] = str(secret_value)
            else:
                env[key] = value
        return env

    def prepare_secrets(self, secret):
        try:
            os.mkdir("properties.d")
        except OSError:
            logger.info("Directory 'properties.d/' already exists")
        for key, value in secret.data().items():
            token_path = os.path.join("properties.d", key)
            with open(token_path, 'w') as fout:
                fout.write(value)
            os.chmod(token_path, 0o600)

    def render_cmd(self, cmd_items):
        context = {
            "environment": self.Parameters.environment,
            "task_name": self.Parameters.task_name,
            "cwd": os.getcwd(),
            "logdir": str(self.log_path()),
            "dc": common.config.Registry().this.dc,
            "its": getattr(self, "abs_its_path", None),
        }
        return [helpers.render_template_str(item, **context) for item in cmd_items]

    def get_current_its(self, secret):
        import retry

        @retry.retry(tries=3, delay=1, backoff=2, logger=logger)
        def do_get():
            url = self.Parameters.its_url
            logger.info("Try get current its: " + url)
            token = secret.data()[self.Parameters.its_secret_name]
            headers = {'Authorization': 'OAuth ' + token}
            response = requests.get(url, headers=headers)
            response.raise_for_status()
            return response.json()

        return do_get()

    def get_json_config(self, js):
        return json.loads(js["user_value"])["values"]

    def make_ini(self, js):
        config = ConfigParser.ConfigParser()
        for section_name in js:
            config.add_section(section_name)
            section = js[section_name]
            for key, value in section.items():
                if value is None:
                    continue
                if isinstance(value, list):
                    config.set(section_name, key, str(",".join(str(v) for v in value)))
                else:
                    config.set(section_name, key, str(value))
        return config

    def save_config(self, cfg):
        self.abs_its_path = self.render_cmd([self.Parameters.its_path])[0]
        with open(self.abs_its_path, "w") as f:
            cfg.write(f)

    def send_juggler_event(self, status, description):
        juggler_host = self.get_juggler_host()
        juggler_service = self.get_juggler_service()

        if not juggler_host or not juggler_service:
            return

        jclient.send_events_to_juggler(juggler_host, juggler_service, status, description)
        self.Context.juggler_status = status
        self.set_info("Send juggler event. Host='{}', service='{}', status='{}', description='{}'".format(
            juggler_host,
            juggler_service,
            status,
            description
        ))

    def get_juggler_host(self):
        if self.Parameters.override_basic_params:
            return self.Parameters.override_juggler_host
        else:
            return JUGGLER_HOSTS[self.Parameters.environment]

    def get_juggler_service(self):
        if self.Parameters.override_basic_params:
            return self.Parameters.override_juggler_service
        else:
            return "{}-status".format(self.Parameters.task_name)

    def get_previous_run_status(self):
        if self.scheduler is None or self.scheduler < 0:
            return None

        # Пока что считаем, что одновренеменно внутри шедулера работает только 1 таска
        last_runs = list(sdk2.Task.find(scheduler=self.scheduler).order(-sdk2.Task.id).limit(2))
        if len(last_runs) < 2:
            return None
        _, last_run = last_runs

        if last_run is None:
            return None
        return last_run.Context.juggler_status
