# coding: utf8

import jinja2
import logging
import os
import tarfile

from sandbox import sdk2
from sandbox.projects.common import resource_selectors


logger = logging.getLogger(__name__)


def __get_last_released_resource_fallback(resource_type, resource_release_status, attrs):
    logger.info("Fallback: looking for resources %s with release status %s", resource_type, resource_release_status)

    attrs = attrs or {}
    attrs['released'] = resource_release_status
    if isinstance(resource_type, str):
        resource = sdk2.Resource.find(type=resource_type, state='READY', attrs=attrs).order(-sdk2.Resource.id).first()
    else:
        resource = resource_type.find(state='READY', attrs=attrs).order(-sdk2.Resource.id).first()

    return resource


def get_last_released_resource(resource_type, release_status, attrs=None):
    """
    Returns last released resource with given type, and with one of the release statuses.
    Resources sorted by release date.
    Raises exception if can't find resource.
    """
    resource_type_name = resource_type if isinstance(resource_type, str) else resource_type.name
    logger.info("Looking for %s with release_status=%s and attrs=%s", resource_type_name, release_status, attrs)
    resource_id, _ = resource_selectors.by_last_released_task(
        resource_type=resource_type_name,
        attrs=attrs,
        stage=release_status
    )

    if resource_id:
        return sdk2.Resource[resource_id]

    # Trying fallback method with searching resources without task info
    resource = __get_last_released_resource_fallback(resource_type, release_status, attrs)
    assert resource, "Can't find {} resource with release_status={}".format(resource_type_name, release_status)
    return resource


def download_resource(resource):
    """
    Returns path to downloaded resource
    """
    logger.info("Downloading resource %s", resource)
    return sdk2.ResourceData(resource).path.as_posix()


def untar_path(path, path_to="."):
    """
    Unarchives given tar archive to the current work directory
    """
    logger.info("Untaring %s", path)
    tarfile.open(path, "r").extractall(path_to)


def render_template_str(template, **context):
    """
    Render jinja2 template with given context
    """
    return jinja2.Template(template, trim_blocks=True, lstrip_blocks=True, undefined=jinja2.StrictUndefined).render(**context)


def render_template(path, **context):
    """
    Render jinja2 template file with given context
    """
    dirname, filename = os.path.split(os.path.abspath(path))
    loader = jinja2.FileSystemLoader(dirname)
    environment = jinja2.Environment(loader=loader, undefined=jinja2.StrictUndefined, trim_blocks=True, lstrip_blocks=True)
    template = environment.get_template(filename)
    rendered_template = template.render(**context)

    logger.info("Rendered %s:\n%s", path, rendered_template)

    with open(path, "w") as f:
        f.write(rendered_template)


def get_environment_tag(environment):
    return "MARKET/{}".format(environment).upper()


def calc_alert_status(upcoming_status, prev_status=None):
    if upcoming_status in ('OK', 'WARN') or prev_status is None:
        return upcoming_status

    # остается только upcoming_status=crit
    if prev_status == 'OK':
        return 'WARN'

    return upcoming_status
