# coding=utf-8
import logging
import os
import time
from sandbox import sdk2
from sandbox.projects.common import vcs
import sandbox.projects.common.constants as consts


WAIT_TIME = 15


class MarketApplicationArcProducerBase(sdk2.Task):
    arc_token = None

    class Parameters(sdk2.Task.Parameters):
        application_name = sdk2.parameters.String("Application name (ex: market-team-application)", required=True)
        commit_message = sdk2.parameters.String("Commit message", required=True)
        arcadia_user = sdk2.parameters.String('Arcadia user', required=True, default='robot-market-infra')

        arc_secret_owner = sdk2.parameters.String("Secret owner (for SB Vault)", default="MARKET", required=False)
        arc_secret_id = sdk2.parameters.String("Vault secret ID", default="", required=False)
        arc_secret_key = sdk2.parameters.String("Secret key", default="ARC_TOKEN", required=True)

    def try_get_arc_token(self):
        key_owner = self.Parameters.arc_secret_owner or ''
        token_key = self.Parameters.arc_secret_key or ''
        token_id = self.Parameters.arc_secret_id or ''

        if token_id is not "":
            arc_token_data = sdk2.yav.Secret(token_id).data()
            self.arc_token = arc_token_data[token_key]

        if self.arc_token is None and key_owner is not "":
            self.arc_token = sdk2.Vault.data(key_owner, token_key)

    def on_execute(self):
        self.try_get_arc_token()

        if self.arc_token:
            self._run_task()
        else:
            raise RuntimeError('Arc token doesn\'t exist')

    @property
    def base_branch_name(self):
        return 'MarketApplicationArcProducerBase'

    @property
    def _full_branch_name(self):
        full_revision = "\n".join(self.arc.describe(self.arcadia_root, svn=True))
        revision = full_revision.split("-")[0].replace("r", "")
        return "{}-{}-{}".format(self.base_branch_name, self.Parameters.application_name, revision)

    @property
    def _ya_bin(self):
        return os.path.join(self.arcadia_root, consts.YMAKE_BUILD_SYSTEM)

    def produce_application(self):
        return False

    def _run_task(self):
        self.arc = vcs.arc.Arc(arc_oauth_token=self.arc_token)

        with self.arc.mount_path("", changeset="trunk", fetch_all=False) as arcadia_root:
            self.arcadia_root = arcadia_root

            should_commit = self.produce_application()

            if should_commit:
                self.push_changes()

    def _get_ya_env(self):
        env = os.environ.copy()

        env['YA_USER'] = self.Parameters.arcadia_user
        env['ARC_USER'] = self.Parameters.arcadia_user
        env['YA_TOKEN'] = self.arc_token
        env['ARC_TOKEN'] = self.arc_token

        return env

    def create_pr(self, message):
        logging.debug("Creating pr")

        pr_create_command = [
            self._ya_bin,
            "pr", "create",
            "--force", "--push",
            "--auto", "--wait", "--json",
            "-m", message
        ]

        with sdk2.helpers.ProcessLog(self, logger="ya_pr_create_log") as pl:
            sdk2.helpers.process.subprocess.check_call(
                pr_create_command,
                cwd=self.arcadia_root,
                stdout=pl.stdout,
                stderr=pl.stderr,
                env=self._get_ya_env()
            )

    def merge_pr(self):
        logging.debug("Merge pr without checks")

        pr_merge_command = [self._ya_bin, "pr", "merge", "--force", "--now"]

        with sdk2.helpers.ProcessLog(self, logger="ya_pr_merge_log") as pl:
            sdk2.helpers.process.subprocess.check_call(
                pr_merge_command,
                cwd=self.arcadia_root,
                stdout=pl.stdout,
                stderr=pl.stderr,
                env=self._get_ya_env()
            )

    def push_changes(self):
        logging.info('Committing to arcadia with message: %s', self.Parameters.commit_message)

        self.arc.checkout(self.arcadia_root, branch=self._full_branch_name, create_branch=True, force=True)
        self.arc.add(self.arcadia_root, all_changes=True)
        self.arc.commit(self.arcadia_root, self.Parameters.commit_message, all_changes=True)

        self.create_pr(self.Parameters.commit_message)
        time.sleep(WAIT_TIME)
        self.merge_pr()

