#!/usr/bin/python
# -*- coding: utf8 -*-

"""

@author Nikolay Firov <a href="mailto:firov@yandex-team.ru"></a>
@date 13.03.18

"""

import logging
from sandbox import sdk2
from sandbox.sdk2.vcs.svn import Arcadia, SvnError
import os
import re

from sandbox.projects.market.devexp.helpers.arc_vcs import MarketArcVcs
from sandbox.projects.common.vcs.arc import ResetMode

from sandbox.projects.market.infra import (
    TsumHotfixScript
)

from sandbox.sdk2 import (
    parameters,
    ResourceData
)


class MarketArcadiaHotfix(sdk2.Task, MarketArcVcs):
    arc_token = None

    class Parameters(sdk2.Task.Parameters):
        base_revision = sdk2.parameters.String('Base revision', default='')
        revs = sdk2.parameters.String('Revisions to pick', default='')
        branch = sdk2.parameters.String('Branch name', default='')
        src = sdk2.parameters.String('Source path', default='/trunk/arcadia')
        dst = sdk2.parameters.String('Destination path', default='/branches/junk/market/infra')
        user = sdk2.parameters.String('Arcadia user', default='zomb-sandbox-rw')
        message = sdk2.parameters.String('Branch commit message', default='Hotfix branch')

        use_arc_branch = sdk2.parameters.Bool('Use Arcadia branch', default=False)
        arc_user = sdk2.parameters.String('Arcadia user', required=True, default='robot-market-infra')
        arc_secret_owner = sdk2.parameters.String("Secret owner (for SB Vault)", default="MARKET", required=False)
        arc_secret_id = sdk2.parameters.String("Vault secret ID", default="", required=False)
        arc_secret_key = sdk2.parameters.String("Secret key", default="ARC_TOKEN", required=True)

        with parameters.Output:
            script_resource = parameters.Resource(
                'hotfix.sh',
                resource_type=TsumHotfixScript,
            )

    class BranchNotReadyException(Exception):
        def __init__(self, value):
            self.value = value

        def __str__(self):
            return repr(self.value)

    def try_get_arc_token(self):
        key_owner = self.Parameters.arc_secret_owner or ''
        token_key = self.Parameters.arc_secret_key or ''
        token_id = self.Parameters.arc_secret_id or ''

        if token_id is not "":
            arc_token_data = sdk2.yav.Secret(token_id).data()
            self.arc_token = arc_token_data[token_key]

        if self.arc_token is None and key_owner is not "":
            self.arc_token = sdk2.Vault.data(key_owner, token_key)

    def on_execute(self):
        logging.info('Preparing hotfix branch {} -> {}'.format(self.Parameters.base_revision, self.Parameters.revs))

        if self.Parameters.use_arc_branch:
            self.on_execute_arc()
            return

        source_path = 'arcadia:/arc{}'.format(self.Parameters.src)
        dst_path = 'arcadia:/arc{}/{}'.format(self.Parameters.dst, self.Parameters.branch)
        checkout_path = str(self.path("sources"))
        revisions = self.Parameters.revs.split(',')
        script_commands = ''
        arc_global_path = 'svn+ssh://arcadia.yandex.ru/arc'
        source_merge_path = '{}{}'.format(arc_global_path, self.Parameters.src)

        self.create_branch(dst_path, source_path)

        logging.info('Arcadia.log(url="{}", revision_from={}, revision_to="HEAD", limit=10)'
                     .format(dst_path, self.Parameters.base_revision))
        branch_log = Arcadia.log(
            url=dst_path, revision_from=self.Parameters.base_revision, revision_to='HEAD', limit=10
        )

        if len(branch_log) > 2:
            logging.info('Hotfix branch has already been patched.')
            return

        modified_dirs, msgs = self.get_modified_dirs_and_messages(source_path, revisions)

        logging.info(
            'Arcadia.checkout("{}", path={}, depth="immediates")'.format(dst_path, checkout_path)
        )

        local_arcadia_path = Arcadia.checkout(dst_path, path=checkout_path, depth='immediates')
        script_commands += 'svn checkout --depth immediates {}/{}/{}\n' \
            .format(arc_global_path, self.Parameters.dst, self.Parameters.branch)
        script_commands += 'cd {}\n'.format(self.Parameters.branch)

        script_commands += self.fetch_source_tree_and_get_fetch_command(local_arcadia_path, modified_dirs)

        logging.info('Starting merge')
        merge_error_exception = None
        merge_error_revision = None

        for idx, revision in enumerate(revisions):
            if merge_error_exception is not None:
                script_commands += self.get_merge_commands(source_merge_path, idx, msgs, revision)
                continue

            try:
                logging.info('Arcadia.merge("{}", "{}", {})'.format(source_path, local_arcadia_path, revision))
                Arcadia.merge(source_path, local_arcadia_path, revision)

                logging.info(
                    'Arcadia.commit("{}", "{}", "{}")'.format(local_arcadia_path, msgs[idx], self.Parameters.user)
                )
                output = Arcadia.commit(local_arcadia_path, msgs[idx], self.Parameters.user)

                if "Committed revision" not in output:
                    merge_error_revision = revision
                    merge_error_exception = MarketArcadiaHotfix.BranchNotReadyException(
                        'Invalid arcadia.commit output'
                    )

                    logging.info('Invalid arcadia.commit output:' + output)
                    continue

                logging.info('Commit completed:' + output)

                # we need to update to eliminate mixed-revision working copy error
                logging.info('Arcadia.update("{}"'.format(local_arcadia_path))
                Arcadia.update(local_arcadia_path)
            except SvnError as exception:
                script_commands += self.get_merge_commands(source_merge_path, idx, msgs, revision)
                merge_error_revision = revision
                merge_error_exception = exception

        if merge_error_exception is not None:
            self.create_script_resource(script_commands)

            raise MarketArcadiaHotfix.BranchNotReadyException(
                'Failed to merge revision: {}\n{}'.format(merge_error_revision, merge_error_exception)
            )

        for idx, revision in enumerate(revisions):
            script_commands += self.get_merge_commands(source_merge_path, idx, msgs, revision)

        self.create_script_resource(script_commands)

        logging.info('Hotfix branch ready')

    @staticmethod
    def get_merge_commands(source_merge_path, idx, msgs, revision):
        commands = 'svn merge -c {} {}\n'.format(revision, source_merge_path)
        commands += 'svn commit -m "{}"\n'.format(msgs[idx])
        commands += 'svn update\n'
        return commands

    @staticmethod
    def get_modified_dirs_and_messages(source_path, revisions):
        modified_dirs = set()
        msgs = []

        for revision in revisions:
            logging.info('Arcadia.log(url="{}", revision_from={})'.format(source_path, revision))
            [revision_log] = Arcadia.log(url=source_path, revision_from=revision, limit=1)
            msg = revision_log['msg'].encode('utf-8', 'ignore')
            msgs.append('cherry-pick ' + revision + ': ' + re.sub(r'REVIEW:', 'RVW:', msg))

            for path in revision_log['paths']:
                modified_dirs.add(os.path.dirname(path[1]))

        return modified_dirs, msgs

    def fetch_source_tree_and_get_fetch_command(self, local_arcadia_path, modified_dirs):
        script_commands = ""
        updated_dirs = set()
        for modified_dir in modified_dirs:
            parts = filter(None, modified_dir.split(os.sep))

            current_path = ''
            for part in parts[len(filter(None, self.Parameters.src.split(os.sep))):]:
                current_path = os.path.join(current_path, part)
                dir_to_update = os.path.join(local_arcadia_path, current_path)

                if not os.path.exists(dir_to_update):
                    logging.info('Directory {} does not exist. Ignoring'.format(dir_to_update))
                    continue

                if dir_to_update in updated_dirs:
                    continue

                updated_dirs.add(dir_to_update)

                logging.info('Arcadia.update("{}", set_depth="immediates"'.format(dir_to_update))
                Arcadia.update(dir_to_update, set_depth='immediates')
                script_commands += "svn update --set-depth immediates {}\n".format(current_path)

        logging.info('Arcadia.update("{}"'.format(local_arcadia_path))
        Arcadia.update(local_arcadia_path)

        script_commands += "svn update\n"

        return script_commands

    def create_branch(self, dst_path, source_path):
        if Arcadia.check(dst_path):
            return

        logging.info(
            'Arcadia.copy(src="{}", dst="{}", revision="{}", user="{}")'.format(
                source_path, dst_path, self.Parameters.base_revision, self.Parameters.user
            )
        )

        Arcadia.copy(
            src=source_path, dst=dst_path, revision=self.Parameters.base_revision,
            user=self.Parameters.user, message=self.Parameters.message
        )

    def create_script_resource(self, script):
        self.Parameters.script_resource = TsumHotfixScript(
            self, 'hotfix.sh', 'hotfix.sh'
        )

        app_res = ResourceData(self.Parameters.script_resource)
        app_res.path.write_bytes(script)
        app_res.ready()

        logging.info('Hotfix script saved to resources')

    def on_execute_arc(self):
        self.try_get_arc_token()

        revisions = self.Parameters.revs.split(',')
        script_commands = ''

        logging.info('Arc mount')
        self.arc_mount(arc_token=self.arc_token)

        branch_exists = self.arc_fetch(branch=self.Parameters.branch)
        if branch_exists:
            self.arc_reset(branch=self.Parameters.branch, mode=ResetMode.HARD, force=True)
            own_history_dict = self.arc_log(
                start_commit="r{}".format(self.Parameters.base_revision),
                end_commit='HEAD',
                max_count=10,
                as_dict=True
            )
            own_history_length = len(own_history_dict)

            if (own_history_length > 0):
                logging.info(
                    'Hotfix branch {} has already been patched. There are {} own commit(s) in this branch'.format(
                        self.Parameters.branch,
                        own_history_length
                    )
                )
                commit_messages = '\nList of own commits:\n'
                for entity in own_history_dict:
                    if (entity['author'] is not None):
                        commit_messages += 'author: {} | '.format(entity['author'])
                    if (entity['commit'] is not None):
                        commit_messages += 'commit: {} | '.format(entity['commit'])
                    if (entity['message'] is not None):
                        commit_messages += 'message:\n{}\n'.format(entity['message'])
                    commit_messages += '\n'
                logging.info(commit_messages)
            self.arc_unmount()
            return

        logging.info('Arc checkout: base_revision={}, '.format(self.Parameters.base_revision))
        self.arc_reset(branch="r{}".format(self.Parameters.base_revision), mode=ResetMode.HARD, force=True)

        logging.info('Arc new branch: creating new branch "{}"'.format(self.Parameters.branch))
        self.arc_checkout(create_branch=True, branch=self.Parameters.branch, track=False)

        logging.info('Starting merge')
        merge_error_exception = None
        merge_error_revision = None

        for revision in revisions:
            try:
                logging.info('Arc cherry-pick: picking commit {}'.format(revision))
                output = self.arc_cherry_pick(commit="r{}".format(revision), add_commit_name=True)

                if "HEAD is now at" not in output:
                    merge_error_revision = revision
                    merge_error_exception = MarketArcadiaHotfix.BranchNotReadyException(
                        'Invalid arc.cherry-pick output'
                    )
                    logging.info('Invalid arc.cherry-pick command output:\n' + output)
                    break

                logging.info('Output of arc.cherry-pick command:\n' + output)
            except Exception as exception:
                logging.error('Arc cherry-pick error: failed picking revision {}. More info below:'.format(revision))
                logging.error(exception)
                merge_error_revision = revision
                merge_error_exception = exception

        script_commands = self.generate_hf_arc_script(
            revisions=revisions,
            branch=self.Parameters.branch
        )
        logging.info('Commands for manual merge:\n{}'.format(script_commands))
        self.create_script_resource(script_commands)

        if merge_error_exception is not None:
            self.unmount_and_raise_exception(
                'Failed to merge revision: {}\n{}'.format(merge_error_revision, merge_error_exception)
            )

        logging.info('Arc push: pushing to remote {}'.format(self.Parameters.branch))
        try:
            self.arc_push(upstream=self.Parameters.branch)
        except Exception as exception:
            self.unmount_and_raise_exception(
                'Failed to push branch: {}\n{}'.format(self.Parameters.branch, exception)
            )

        self.arc_unmount()
        logging.info('Hotfix branch ready')

    def unmount_and_raise_exception(self, msg):
        self.arc_unmount()
        raise MarketArcadiaHotfix.BranchNotReadyException(msg)

    def generate_hf_arc_script(self, revisions, branch):
        script_commands = ''
        script_commands += 'arc checkout trunk\n'
        script_commands += 'arc fetch --verbose trunk\n'
        script_commands += 'arc reset --hard r{} --force\n'.format(self.Parameters.base_revision)
        script_commands += 'arc checkout -b {}\n'.format(branch)

        for revision in revisions:
            script_commands += 'arc cherry-pick r{}\n'.format(revision)

        script_commands += 'arc push -u {}'.format(branch)

        return script_commands
