import re
import time
import logging

from sandbox.common.errors import SubprocessError
from sandbox import sdk2
from sandbox.sdk2 import yav
from sandbox.sdk2.vcs.git import Git

class MarketBitbucketHotfix(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        retry_count = 5
        sleep_time_between_retry = 5
        base_revision = sdk2.parameters.String('Base revision', default='', required=True)
        revs = sdk2.parameters.String('Revisions to pick, comma-separated, no spaces', default='', required=True)
        branch = sdk2.parameters.String('Branch name', default='', required=True)
        clone_branch = sdk2.parameters.String('Branch to clone from', default='master')
        bitbucket_project = sdk2.parameters.String('Bitbucket project', default='', required=True)
        bitbucket_repo = sdk2.parameters.String('Bitbucket repo', default='', required=True)
        yav_secret = sdk2.parameters.String("Yav secret with bitbucket OAuth token", required=True)
        yav_token_key = sdk2.parameters.String('Key for bitbucket OAuth token in yav', default='tsum.bitbucket.token')

    def on_execute(self):
        git = Git(self.create_git_url())

        self.with_retry(git, self.clone_repo)
        self.checkout(git)
        self.cherry_pick(git)
        self.push(git)

    def with_retry(self, git, func):
        run = 0
        while run < self.Parameters.retry_count - 1:
            try:
                func(git)
            except SubprocessError:
                continue
            else:
                return
            finally:
                run += 1
                time.sleep(self.Parameters.sleep_time_between_retry)
        func(git)

    def clone_repo(self, git):
        git.clone('.', self.Parameters.clone_branch)

    def checkout(self, git):
        commit = self.Parameters.base_revision
        git.execute("checkout", "-f", commit)

        branch = self.Parameters.branch
        git.execute("checkout", "-b", branch)

    def cherry_pick(self, git):
        revisions = self.Parameters.revs.split(',')

        for revision in revisions:
            try:
                git.execute('cherry-pick', revision)
            except SubprocessError as e:
                match = re.search('died with exit code (\d+)', str(e))
                if match:
                    error = match.group(1)
                    # error code 128 possibly means that we are trying to cherry-pick merge commit without '-m' option
                    # in this case we are trying to cherry-pick with '-m 1' option
                    if error == '128':
                        logging.info(
                            "Got error code 128 while cherry-picking. Trying to cherry-pick with '-m 1' option"
                        )
                        git.execute('cherry-pick', '-m', '1', revision)
                        return
                raise e

    def push(self, git):
        branch = self.Parameters.branch
        git.execute('push', '--set-upstream', 'origin', branch)

    def create_git_url(self):
        token = yav.Secret(self.Parameters.yav_secret).data()[self.Parameters.yav_token_key]
        return 'https://{username}:{password}@bb.yandex-team.ru/scm/{project}/{repo}.git'.format(
            username='x-oauth-token',
            password=token,
            project=self.Parameters.bitbucket_project,
            repo=self.Parameters.bitbucket_repo
        )
