# coding=utf-8

import os
import re
import tempfile

from sandbox.common.types.client import Tag
from sandbox.common.types.misc import RamDrive, RamDriveType
from sandbox.projects.common.debpkg import DebRelease
from sandbox.projects.common.decorators import retries
from sandbox.projects.market.infra import UnstripBinary, BuildLog
from sandbox.projects.market.infra.helpers.gpg_key import GpgKey
from sandbox.sdk2 import Task, Vault, parameters, ResourceData, svn
from sandbox.sdk2.helpers import subprocess as sp, ProcessLog
from sandbox.sdk2.path import Path
from sandbox.sdk2.ssh import Key


class MarketBuildDebPackageTask2(Task):
    """Build a debian package with debuild
    """

    _KEY_OWNER = 'MARKET'

    # The list of modules which are better to build of dedicated clients. The dedicated clients marked with Tag.MARKET.
    _MODULES_REQUIRE_DEDICATED_CLIENTS = (
        'market/marketindexer',
        'market/indexer',
    )

    class Requirements(Task.Requirements):
        client_tags = Tag.GENERIC | Tag.MARKET
        ramdrive = RamDrive(RamDriveType.TMPFS, 10240, None)

    class Parameters(Task.Parameters):
        module_path = parameters.String(
            'Module path',
            description='Example: market/backctld',
            required=True
        )
        branch_path = parameters.String(
            'Branch path',
            description='Example: market/idx/publisher/2018.1.1. If empty then trunk',
            default_value=None
        )
        revision = parameters.String(
            'Revision',
            description='If empty then HEAD',
            default_value=None
        )
        arcadia_patch = parameters.String(
            'Apply patch',
            description='Diff file rbtorrent, paste.y-t.ru link or plain text. Doc: https://nda.ya.ru/3QTTV4',
            multiline=True,
            default_value=None,
            required=False
        )
        run_tests = parameters.Bool(
            'Run tests',
            description='If not checked then add DEB_BUILD_OPTIONS=nocheck option for debuild',
            default_value=True,
            required=True
        )
        package_version = parameters.String(
            'Version of package',
            required=True
        )
        dist_repos = parameters.String(
            'Repository',
            description='Repository for upload command',
            default_value=None,
            required=False
        )
        deb_user = parameters.String(
            'Debian user',
            required=True,
            description='User that will sign deb package'
        )
        deb_mail = parameters.String(
            'Debian email',
            required=True,
            description='Email of debian user'
        )
        ssh_key_name = parameters.String(
            'SSH key name in Sandbox Vault',
            group='GPG',
            required=True
        )
        gpg_private_key_name = parameters.String(
            'Private GPG key in Sandbox Vault',
            group='GPG',
            required=True
        )
        gpg_public_key_name = parameters.String(
            'Public GPG key in Sandbox Vault',
            group='GPG',
            required=True
        )
        yt_token_key_name = parameters.String(
            'YT token in Sandbox Vault',
            group='YT',
            required=False
        )
        build_with_dist = parameters.Bool(
            'Do ya make with dist option',
            default_value=False,
            group='Build'
        )
        generate_unstrip_binary = parameters.Bool(
            'Generate unstrip binary',
            default_value=False,
            group='Generated resources',
        )
        generate_build_log = parameters.Bool(
            'Generate the build log resource',
            default_value=False,
            group='Generated resources',
        )

    def on_enqueue(self):
        is_heavy_module = str(self.Parameters.module_path) in self._MODULES_REQUIRE_DEDICATED_CLIENTS
        if is_heavy_module:
            self.Requirements.client_tags = Tag.MARKET | (Tag.SSD & (Tag.INTEL_E5_2650 | Tag.INTEL_E5_2660V1 | Tag.INTEL_E5_2660V4))
        else:
            self.Requirements.client_tags = Tag.INTEL_E5_2650 | Tag.INTEL_E5_2660V1 | Tag.INTEL_E5_2660V4

    def __make_debian_changelog(self, version, module_path):
        with ProcessLog(self, logger="dch_log") as pl:
            environment = os.environ.copy()
            environment['DEBEMAIL'] = '{user} <{mail}>'.format(
                user=self.Parameters.deb_user,
                mail=self.Parameters.deb_mail
            )
            sp.check_call(
                u'dch -b -v "{v}" --distribution unstable new release {v}'.format(
                    v=version
                ),
                shell=True,
                env=environment,
                cwd=module_path,
                stdout=pl.stdout,
                stderr=sp.STDOUT
            )

    def __debuild(self, module_path, out_dir=None):
        self.__debuild_log_path = None
        with ProcessLog(self, logger="debuild_log") as pl:
            environment = os.environ.copy()
            tmp_path = tempfile.mkdtemp()

            ya_make_opts = []
            if not bool(self.Parameters.run_tests):
                environment['DEB_BUILD_OPTIONS'] = 'nocheck'

            if out_dir:
                ya_make_opts.append(
                    '-o {out_dir}'.format(out_dir=out_dir)
                )

            if self.Parameters.yt_token_key_name:
                environment['YT_TOKEN'] = Vault.data(self._KEY_OWNER, self.Parameters.yt_token_key_name)
                ya_make_opts.append('--yt-store')

            if bool(self.Parameters.build_with_dist):
                ya_make_opts.append('--dist -E --add-result=".a"')

            if self.ramdrive:
                ya_make_opts.append('--test-param=ram_drive_path={}'.format(self.ramdrive.path))

            cmd = (
                'debuild '
                '--no-lintian '
                '--preserve-env '
                '--set-envvar=YA_CACHE_DIR="{tmp}" '
                '--set-envvar=YA_MAKE_OPTS="{ya_make_opts}" '
                '-k{mail}'
            ).format(tmp=tmp_path, ya_make_opts=' '.join(ya_make_opts), mail=self.Parameters.deb_mail)
            sp.check_call(
                cmd,
                shell=True,
                env=environment,
                cwd=module_path,
                stdout=pl.stdout,
                stderr=sp.STDOUT
            )

            self.__debuild_log_path = pl.stdout.path

    @retries(max_tries=5, delay=1)
    def __upload(self, repos, module_path):
        repos_config = {repo: {} for repo in re.sub('[\s+]', '', repos).split(',')}
        with DebRelease(dupload_config=repos_config, login=self.Parameters.deb_user) as deb:
            deb.debrelease(['-f', '--to', repos], work_dir=module_path)

    def __make_unstrip_resource(self, unstrip_dir, package_version, revision):
        market_dir = Path(unstrip_dir).joinpath('market')
        if market_dir.exists():
            root_path_bin = str(market_dir)
            for root, _, file_names in os.walk(root_path_bin):
                for filename in file_names:
                    absolute_bin_path = str(Path(root).joinpath(filename))
                    relative_bin_path = absolute_bin_path.replace(unstrip_dir, '')[1:]
                    is_unstrip_bin = (os.path.isfile(absolute_bin_path)
                                      and os.access(absolute_bin_path, os.X_OK)
                                      and 'test' not in absolute_bin_path)
                    if is_unstrip_bin:
                        res = UnstripBinary(self, filename, absolute_bin_path)
                        res.binary_path = os.path.dirname(relative_bin_path)
                        res.binary_name = os.path.basename(relative_bin_path)
                        res.package_version = package_version
                        res.revision = revision

                        app_res = ResourceData(res)
                        app_res.path.write_bytes(Path(absolute_bin_path).read_bytes())
                        app_res.ready()

    def __make_build_log_resource(self, build_log_path, package_version, revision):
        res = BuildLog(self, 'Complete debuild log, including test results', 'package-build.log')
        res.package_version = package_version
        res.revision = revision
        res.ttl = 2 * 366  # yes, 2 years

        app_res = ResourceData(res)
        app_res.path.write_bytes(Path(build_log_path).read_bytes())
        app_res.ready()

    def on_execute(self):
        if self.Parameters.branch_path:
            url = svn.Arcadia.branch_url(
                branch=self.Parameters.branch_path,
                revision=self.Parameters.revision
            )
        else:
            url = svn.Arcadia.trunk_url(
                revision=self.Parameters.revision
            )

        arcadia_root = svn.Arcadia.get_arcadia_src_dir(
            url=url,
            copy_trunk=True
        )
        if self.Parameters.arcadia_patch:
            svn.Arcadia.apply_patch(
                arcadia_root,
                self.Parameters.arcadia_patch,
                os.getcwd()
            )

        module_path = str(Path(arcadia_root).joinpath(self.Parameters.module_path))
        unstrip_dir = str(self.path())

        with Key(self, self._KEY_OWNER, self.Parameters.ssh_key_name):
            with GpgKey(self.Parameters.gpg_private_key_name, self.Parameters.gpg_public_key_name):
                self.__make_debian_changelog(
                    version=self.Parameters.package_version,
                    module_path=module_path
                )
                self.__debuild(module_path=module_path, out_dir=unstrip_dir)

                if self.Parameters.dist_repos:
                    self.__upload(
                        repos=str(self.Parameters.dist_repos),
                        module_path=module_path
                    )

        if self.Parameters.generate_unstrip_binary:
            self.__make_unstrip_resource(
                unstrip_dir=unstrip_dir,
                package_version=self.Parameters.package_version,
                revision=self.Parameters.revision
            )

        if self.Parameters.generate_build_log and self.__debuild_log_path:
            self.__make_build_log_resource(
                build_log_path=self.__debuild_log_path,
                package_version=self.Parameters.package_version,
                revision=self.Parameters.revision,
            )
